package ru.yandex.travel.orders.infrastructure;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.extern.slf4j.Slf4j;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.api.MigrationInfoService;
import org.flywaydb.core.api.callback.Callback;
import org.flywaydb.core.api.callback.Context;
import org.flywaydb.core.api.callback.Event;
import org.slf4j.helpers.MessageFormatter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("flyway")
@Slf4j
public class FlywayController {

    private Flyway flyway;

    @Autowired
    public FlywayController(Flyway flyway) {
        this.flyway = flyway;
    }

    @GetMapping("info")
    public FlywayInfo info() {
        return new FlywayInfo(flyway.info());
    }

    @PostMapping("migrate")
    public History migrate() {
        History result = new History();
        Flyway.configure().callbacks(result.getCallback());
        flyway.migrate();
        return result;
    }

    public static final class FlywayInfo {
        private MigrationInfoService migrationInfoService;

        public FlywayInfo(MigrationInfoService migrationInfoService) {
            this.migrationInfoService = migrationInfoService;
        }

        @JsonSerialize(typing = JsonSerialize.Typing.STATIC)
        public List<MigrationInfo> getApplied() {
            return Arrays.asList(migrationInfoService.applied());
        }

        @JsonSerialize(typing = JsonSerialize.Typing.STATIC)
        public List<MigrationInfo> getPending() {
            return Arrays.asList(migrationInfoService.pending());
        }
    }

    public static class HistoryEvent {
        private final String message;
        private final Instant time;

        public HistoryEvent(String message) {
            this.message = message;
            this.time = Instant.now();
        }

        public String getMessage() {
            return message;
        }

        public Long getTime() {
            return time.toEpochMilli();
        }
    }


    public static class History {

        private List<HistoryEvent> events = new ArrayList<>();

        public List<HistoryEvent> getEvents() {
            return events;
        }

        @JsonIgnore
        public Callback getCallback() {
            return new Callback() {
                @Override
                public boolean supports(Event event, Context context) {
                    return true; //supporting all events
                }

                @Override
                public boolean canHandleInTransaction(Event event, Context context) {
                    return true;
                }

                @Override
                public void handle(Event event, Context context) {
                    switch (event) {
                        case BEFORE_CLEAN:
                            add("Going to clean up");
                            return;
                        case AFTER_CLEAN:
                            add("Clean up done");
                            return;
                        case BEFORE_MIGRATE:
                            add("Going to migrate");
                            return;
                        case AFTER_MIGRATE:
                            add("Migration done");
                            return;
                        case BEFORE_EACH_MIGRATE:
                            add("Applying {} migration {} (checksum {})",
                                    context.getMigrationInfo().getType().name(),
                                    context.getMigrationInfo().getScript(),
                                    context.getMigrationInfo().getChecksum()
                            );
                            return;
                        case AFTER_EACH_MIGRATE:
                            add("Applied {} migration {} (checksum {})",
                                    context.getMigrationInfo().getType().name(),
                                    context.getMigrationInfo().getScript(),
                                    context.getMigrationInfo().getChecksum()
                            );
                            return;
                        case BEFORE_VALIDATE:
                            add("Validating");
                            return;
                        case AFTER_VALIDATE:
                            add("Validated");
                            return;
                        default:
                            // ignoring other events
                    }
                }

                private void add(String message, Object... params) {
                    events.add(new HistoryEvent(format(message, params)));
                }

                private String format(String format, Object... params) {
                    return MessageFormatter.arrayFormat(format, params).getMessage();
                }
            };
        }
    }

}
