package ru.yandex.travel.orders.management;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.travel.grpc.GrpcServerRunner;
import ru.yandex.travel.orders.OrdersApplicationProperties;

@Slf4j
@Component
public class ShutdownManager implements ApplicationContextAware, HealthIndicator {
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS");

    private AtomicBoolean shutdownInitiated = new AtomicBoolean(false);

    private LocalDateTime shutdownStartedAt;

    private ApplicationContext ctx;

    private ScheduledExecutorService shutdownSequenceExecutor;

    private Duration waitShutdownTimeout;

    public ShutdownManager(OrdersApplicationProperties properties) {
        this.waitShutdownTimeout = properties.getShutdownTimeout();
        shutdownSequenceExecutor = new ScheduledThreadPoolExecutor(
                1, new ThreadFactoryBuilder().setNameFormat("shutdown-thread-%s").build()
        );

    }

    public void initShutdown() {
        if (!shutdownInitiated.getAndSet(true)) {
            log.info("Shutdown initiated, health check flag switched off");
            shutdownStartedAt = LocalDateTime.now();
            shutdownSequenceExecutor.schedule(this::runShutdownSequence, waitShutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    private void runShutdownSequence() {
        ctx.getBeansOfType(GrpcServerRunner.class).forEach((name, bean) -> {
            log.info("Shutting down gRPC server runner with name {}", name);
            try {
                bean.stop();
            } catch (Exception e) {
                log.error("Unable to shutdown gRPC server runner with name {}", name, e);
            }
        });
        log.info("Shutting down application context");
        ((ConfigurableApplicationContext) ctx).close();
        log.info("Everything shut down. Exiting");
        System.exit(0);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.ctx = applicationContext;
    }

    @Override
    public Health health() {
        if (shutdownInitiated.get()) {
            return Health.down()
                    .withDetail("Shutdown started at", dateTimeFormatter.format(shutdownStartedAt))
                    .build();
        } else {
            return Health.up().build();
        }
    }
}
