package ru.yandex.travel.orders.management;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.error.StartrekClientException;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.Transition;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.PriceCheckOutcome;
import ru.yandex.travel.orders.entities.STClientRetryableException;
import ru.yandex.travel.orders.entities.Ticket;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.notifications.NotificationFailureReason;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.services.notifications.NotificationMeters;
import ru.yandex.travel.orders.workflow.order.proto.TCreateIssue;
import ru.yandex.travel.orders.workflow.ticket.proto.ETicketState;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@SuppressWarnings("UnusedReturnValue")
@Component
@EnableConfigurationProperties(StarTrekConfigurationProperties.class)
@Slf4j
public class StarTrekService {

    private static final String MISSING_QUEUE_NAME = "MISSING-QUEUE-TICKET";

    private final Session client;
    private final TicketRepository ticketRepository;
    private final OrderRepository orderRepository;
    private final NotificationRepository notificationRepository;
    private final VoucherRepository voucherRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowEventRepository eventRepository;
    private final StarTrekConfigurationProperties config;
    private final StarTrekUtils starTrekUtils;
    private final YandexPlusTopupRepository yandexPlusTopupRepository;
    private final NotificationMeters notificationMeters;

    public StarTrekService(StarTrekConfigurationProperties config, TicketRepository ticketRepository,
                           WorkflowRepository workflowRepository, OrderRepository orderRepository,
                           NotificationRepository notificationRepository, VoucherRepository voucherRepository,
                           WorkflowEventRepository eventRepository, StarTrekUtils starTrekUtils,
                           YandexPlusTopupRepository yandexPlusTopupRepository, NotificationMeters notificationMeters) {
        this.client = StartrekClientBuilder.newBuilder()
                .uri(config.getUrl())
                .maxConnections(10)
                .connectionTimeout(config.getConnectionTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .socketTimeout(config.getSocketTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .build(config.getOauthToken());
        this.ticketRepository = ticketRepository;
        this.workflowRepository = workflowRepository;
        this.orderRepository = orderRepository;
        this.notificationRepository = notificationRepository;
        this.voucherRepository = voucherRepository;
        this.eventRepository = eventRepository;
        this.config = config;
        this.starTrekUtils = starTrekUtils;
        this.yandexPlusTopupRepository = yandexPlusTopupRepository;
        this.notificationMeters = notificationMeters;
    }

    @TransactionMandatory
    public UUID createIssueForGeneralOrderError(UUID orderId, long failedEventId, Instant happenedAt,
                                                MessagingContext<?> context) {
        String title;
        String description;
        List<String> tags;
        try {
            Order order = orderRepository.getOne(orderId);
            WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
            title = String.format(config.getErrorIssueTitle(), order.getPrettyId());
            description = starTrekUtils.prepareDescriptionForError(order, failedEvent, happenedAt);
            tags = getTagsForOrderCrash(order);
        } catch (Exception e) {
            log.error("", e);
            title = String.format(config.getErrorIssueTitle(), orderId);
            description = String.format(config.getErrorGettingDescription(), orderId);
            tags = List.of();
        }
        return createIssueWorkflow(orderId, title, description, config.getAutoQueueName(), tags, context);
    }

    @TransactionMandatory
    public UUID createIssueForGenericWorkflowError(UUID workflowId, long failedEventId, MessagingContext<?> context) {
        Workflow workflow = workflowRepository.getOne(workflowId);
        WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
        String title = String.format(config.getWorkflowErrorIssueTitle(), workflow.getEntityType(), workflow.getId());
        String description = starTrekUtils.prepareDescriptionForGenericWorkflowError(workflow, failedEvent);
        return createIssueWorkflow(
                null,
                title,
                description,
                config.getAutoQueueName(), getTagsForWorkflowCrash(workflow.getEntityType()),
                context
        );
    }

    private List<String> getTagsForOrderCrash(Order order) {
        return List.of("auto:workflow_crashed", "auto:entity:" + order.getWorkflow().getEntityType(),
                starTrekUtils.getOrderTeamTag(order));
    }

    private List<String> getTagsForWorkflowCrash(String entityType) {
        var mappingList = List.of(
                Pair.of("hotel_duty_required", List.of(
                        WellKnownWorkflowEntityType.HOTEL_ORDER,
                        WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM,
                        WellKnownWorkflowEntityType.DOLPHIN_ORDER_ITEM,
                        WellKnownWorkflowEntityType.TRAVELLINE_ORDER_ITEM,
                        WellKnownWorkflowEntityType.BNOVO_ORDER_ITEM,
                        WellKnownWorkflowEntityType.VOUCHER,
                        WellKnownWorkflowEntityType.EMAIL)),
                Pair.of("avia_duty_required", List.of(
                        WellKnownWorkflowEntityType.AEROFLOT_ORDER,
                        WellKnownWorkflowEntityType.AEROFLOT_ORDER_ITEM,
                        WellKnownWorkflowEntityType.AEROFLOT_INVOICE,
                        WellKnownWorkflowEntityType.AEROFLOT_MQ_MESSAGE)),
                Pair.of("buses_duty_required", List.of(
                        WellKnownWorkflowEntityType.TRAIN_ORDER,
                        WellKnownWorkflowEntityType.TRAIN_ORDER_ITEM,
                        WellKnownWorkflowEntityType.TRAIN_TICKET_REFUND,
                        WellKnownWorkflowEntityType.TRAIN_INSURANCE_REFUND,
                        WellKnownWorkflowEntityType.BUS_ORDER_ITEM,
                        WellKnownWorkflowEntityType.BUS_TICKET_REFUND)),
                Pair.of("rasp_duty_required", List.of(
                        WellKnownWorkflowEntityType.SUBURBAN_ORDER_ITEM))
        );
        var dutyTag = "orders_duty_required";
        for (var pair : mappingList) {
            if (pair.getSecond().stream().anyMatch(x -> x.getDiscriminatorValue().equals(entityType))) {
                dutyTag = pair.getFirst();
            }
        }
        return List.of("auto:workflow_crashed", "auto:entity:" + entityType, dutyTag);
    }

    @TransactionMandatory
    public UUID createIssueForOrderPaidNotConfirmed(UUID orderId, long failedEventId, Instant happenedAt,
                                                    MessagingContext<?> context) {
        String title;
        String description;
        List<String> tags;
        try {
            Order order = orderRepository.getOne(orderId);
            WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
            title = String.format(config.getOrderPaidNotConfirmedIssueTitle(), order.getPrettyId());
            description = starTrekUtils.prepareDescriptionForOrderPaidHoldFailed(order, failedEvent, happenedAt);
            tags = List.of("auto:could_not_confirm_expedia_itinerary",
                    "auto:entity:" + order.getWorkflow().getEntityType());
        } catch (Exception e) {
            log.error("", e);
            title = String.format(config.getOrderPaidNotConfirmedIssueTitle(), orderId);
            description = String.format(config.getOrderPaidNotConfirmedErrorGettingDescription(), orderId);
            tags = List.of();
        }
        return createIssueWorkflow(orderId, title, description, config.getAutoQueueName(), tags, context);
    }

    @TransactionMandatory
    public UUID createIssueForPdfNotReceived(UUID voucherId, long failedEventId, MessagingContext<?> context) {
        String title;
        String description;
        List<String> tags;
        try {
            Voucher voucher = voucherRepository.getOne(voucherId);
            Order order = orderRepository.getOne(voucher.getOrderId());
            Workflow workflow = voucher.getWorkflow();
            WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
            title = String.format(config.getPdfNotReceivedIssueTitle(), order.getPrettyId());
            description = starTrekUtils.prepareDescriptionForPdfNotReceived(order, workflow, failedEvent);
            tags = List.of(
                    "auto:pdf_generator_failed",
                    "auto:entity:" + workflow.getEntityType(),
                    starTrekUtils.getVoucherTeamTag(voucher.getVoucherType())
            );
            return createIssueWorkflow(order.getId(), title, description, config.getAutoQueueName(), tags, context);
        } catch (Exception e) {
            title = String.format(config.getPdfNotReceivedErrorGettingTitle(), voucherId);
            description = String.format(config.getPdfNotReceivedErrorGettingDescription(), voucherId);
            tags = List.of("auto:pdf_generator_failed");
            return createIssueWorkflow(null, title, description, config.getAutoQueueName(), tags, context);
        }
    }

    @TransactionMandatory
    public UUID createIssueForTrustReceiptNotFetched(
            UUID orderId, String prettyId, List<UUID> failedAttachmentIds, MessagingContext<?> context) {
        String title;
        String description;
        List<String> tags;
        try {
            Order order = orderRepository.getOne(orderId);
            title = String.format(config.getTrustReceiptNotFetchedTitle(), prettyId);
            description = starTrekUtils.prepareDescriptionForTrustReceiptNotFetched(orderId, prettyId,
                    order, failedAttachmentIds);
            tags = List.of("auto:trust_receipt_not_fetched", "auto:entity:" + order.getWorkflow().getEntityType());
        } catch (Exception e) {
            title = String.format(config.getTrustReceiptNotFetchedTitle(), prettyId);
            description = starTrekUtils.prepareDescriptionForTrustReceiptNotFetched(
                    orderId, prettyId, null, failedAttachmentIds);
            tags = List.of("auto:trust_receipt_not_fetched");
        }
        return createIssueWorkflow(null, title, description, config.getAutoQueueName(), tags, context);
    }

    @TransactionMandatory
    public UUID createIssueForNotificationNotSent(UUID notificationId, long failedEventId,
                                                  MessagingContext<?> context) {
        String title;
        String description;
        List<String> tags;
        try {
            var notification = notificationRepository.getOne(notificationId);
            notificationMeters.getNotificationsFailed()
                    .get(notification.getChannel())
                    .get(NotificationFailureReason.UNHANDLED_EXCEPTION).increment();
            if (notification.getOrderId() != null) {
                // Order notification
                Order order = orderRepository.getOne(notification.getOrderId());
                Workflow workflow = notification.getWorkflow();
                WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
                title = String.format(config.getNotificationNotSentIssueTitle(), order.getPrettyId());
                description = starTrekUtils.prepareDescriptionForNotificationNotSent(order, notification.getChannel(),
                        workflow, failedEvent);
                tags = List.of("auto:notification_sender_failed", "auto:order_notification",
                        "auto:entity:" + order.getWorkflow().getEntityType());
            } else {
                // Non-order notification, e.g. partner notification
                Workflow workflow = notification.getWorkflow();
                WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
                title = String.format(config.getNotificationNotSentErrorGettingTitle(), notificationId);
                description = starTrekUtils.prepareDescriptionForNonOrderNotificationNotSent(notification.getChannel(),
                        workflow, failedEvent);
                tags = List.of("auto:notification_sender_failed", "auto:non_order_notification",
                        "auto:entity:" + workflow.getEntityType());
            }
        } catch (Exception e) {
            title = String.format(config.getNotificationNotSentErrorGettingTitle(), notificationId);
            description = String.format(config.getNotificationNotSentErrorGettingDescription(), notificationId);
            tags = List.of("auto:notification_sender_failed");
        }
        return createIssueWorkflow(null, title, description, config.getAutoQueueName(), tags, context);
    }

    @TransactionMandatory
    public UUID createIssueForPlusPointsTopupFailed(UUID topupId, long failedEventId, MessagingContext<?> context) {
        YandexPlusTopup topup = yandexPlusTopupRepository.getOne(topupId);
        OrderItem orderItem = topup.getOrderItem();
        Order order = orderItem != null ? orderItem.getOrder() : null;
        Workflow workflow = topup.getWorkflow();
        WorkflowEvent failedEvent = eventRepository.getOne(failedEventId);
        String title = String.format(config.getPlusPointsTopupFailedTitle(),
                order != null ? order.getPrettyId() : topup.getExternalOrderId());
        String description = starTrekUtils.prepareDescriptionForPlusPointsTopupFailed(order, orderItem, topup,
                workflow, failedEvent);

        List<String> tags = order != null
                ? List.of("auto:plus_topup_failed", "auto:entity:" + order.getWorkflow().getEntityType())
                : List.of("auto:plus_topup_failed");
        return createIssueWorkflow(null, title, description, config.getAutoQueueName(), tags, context);
    }

    @TransactionMandatory
    public UUID createIssueForAeroflotLostOrder(AeroflotMqData order, MessagingContext<?> context) {
        return createIssueWorkflow(
                null,
                String.format(config.getAeroflotLostOrderIssueTitle(), order.getPnr()),
                starTrekUtils.prepareDescriptionForAeroflotLostOrder(order),
                config.getAutoQueueName(), List.of("auto:aeroflot_lost_order", "auto:avia", "auto:entity" +
                        ":aeroflot_order"),
                context
        );
    }

    @TransactionMandatory
    public UUID createIssueForAeroflotCancelledOrderPaid(AeroflotOrder order, String ownerLastName,
                                                         MessagingContext<?> context) {
        return createIssueWorkflow(
                null,
                String.format(config.getAeroflotCancelledOrderPaidIssueTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForAeroflotCancelledOrderPaid(order, ownerLastName),
                config.getAutoQueueName(), List.of("auto:aeroflot_restored_order", "auto:avia", "auto:entity" +
                        ":aeroflot_order"),
                context
        );
    }

    @TransactionMandatory
    public UUID createIssueForAeroflotFailedTokenization(Order order, String purchaseToken, String failureDetails,
                                                         MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getAeroflotFailedTokenizationIssueTitle(), order.getId()),
                starTrekUtils.prepareDescriptionForAeroflotFailedTokenization(order, purchaseToken, failureDetails),
                config.getAutoQueueName(), List.of("auto:aeroflot_failed_tokenization", "auto:avia", "auto:entity" +
                        ":aeroflot_order"),
                context
        );
    }

    @TransactionMandatory
    public UUID createIssueForHotelOrderPriceMismatch(Order order, BigDecimal actualPrice, BigDecimal expectedPrice,
                                                      boolean isExceeding, boolean isRefundable,
                                                      PriceCheckOutcome outcome, MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getHotelOrderPriceMismatchIssueTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForHotelOrderPriceMismatch(order, actualPrice, expectedPrice,
                        isExceeding, isRefundable, outcome),
                config.getAutoQueueName(), List.of("auto:hotel_price_mismatch", "auto:hotels", "auto:entity" +
                        ":hotel_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForHotelManualProcessing(Order order, ManualServiceProcessingTicketData data,
                                                    MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getManualProcessingIssueTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForManualServiceProcessing(order, data),
                config.getAutoQueueName(), List.of("auto:manual_pr", "auto:hotels", "auto:entity:hotel_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainInsuranceNotAdded(Order order, MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainInsuranceNotAddedTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainInsuranceNotAdded(order),
                config.getAutoQueueName(), List.of("auto:insurance_not_added", "auto:trains", "auto:entity" +
                        ":train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainInsuranceNotConfirmed(Order order, MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainInsuranceNotConfirmedTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainInsuranceNotConfirmed(order),
                config.getAutoQueueName(), List.of("auto:insurance_not_confirmed", "auto:trains", "auto:entity" +
                        ":train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainInsuranceRefundFailed(Order order, TrainInsuranceRefund refund,
                                                         MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainInsuranceRefundFailedTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainInsuranceRefundFailed(order, refund),
                config.getAutoQueueName(), List.of("auto:insurance_refund_failed", "auto:trains", "auto:entity" +
                        ":train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainTicketRefundFailed(Order order, TrainTicketRefund refund,
                                                      MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainTicketRefundFailedTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainTicketRefundFailed(order, refund),
                config.getAutoQueueName(), List.of("auto:ticket_refund_failed", "auto:trains", "auto:entity" +
                        ":train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainOfficeRefundHandleError(Order order, Exception e,
                                                           WorkflowProcessService workflowProcessService) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainOfficeRefundHandleError(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainOfficeRefundHandleError(order, e), config.getAutoQueueName(),
                List.of("auto:office_refund_handle_error", "auto:trains", "auto:entity:train_order"),
                workflowProcessService);
    }

    @TransactionMandatory
    public UUID createIssueForEmptyTrainCarrierInn(Order order, MessagingContext<?> context) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getEmptyTrainCarrierInnTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForEmptyTrainCarrierInn(order),
                config.getAutoQueueName(), List.of("auto:empty_carrier_inn", "auto:trains", "auto:entity:train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrainInsuranceInvalidAmount(
            Order order, MessagingContext<?> context, BigDecimal expectedAmount, BigDecimal obtainedAmount) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrainInsuranceInvalidAmountTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForTrainInsuranceInvalidAmount(order, expectedAmount, obtainedAmount),
                config.getAutoQueueName(), List.of("auto:insurance_invalid_amount", "auto:trains", "auto:entity" +
                        ":train_order"),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForTrustRefundError(TrustInvoice workflowEntity, UUID trustRefundId,
                                               MessagingContext<?> context, String reason) {
        return createIssueWorkflow(
                workflowEntity.getOrder().getId(),
                String.format(config.getTrustRefundErrorTitle(), workflowEntity.getOrder().getPrettyId()),
                starTrekUtils.prepareDescriptionForTrustRefundError(workflowEntity, trustRefundId, reason),
                config.getAutoQueueName(), List.of("auto:trust_refund_error",
                        "auto:entity:" + WellKnownWorkflowEntityType.TRUST_REFUND.getDiscriminatorValue()),
                context);
    }

    @TransactionMandatory
    public UUID createIssueForInvoiceNotRefunded(Order order, UUID orderRefundId, UUID invoiceId,
                                                 MessagingContext<?> context, String reason) {
        return createIssueWorkflow(
                order.getId(),
                String.format(config.getTrustRefundErrorTitle(), order.getPrettyId()),
                starTrekUtils.prepareDescriptionForInvoiceNotRefunded(order, orderRefundId, invoiceId, reason),
                config.getAutoQueueName(), List.of("auto:trust_refund_error",
                        "auto:entity:" + WellKnownWorkflowEntityType.TRUST_REFUND.getDiscriminatorValue()),
                context);
    }

    @TransactionMandatory
    public UUID createIssueToRemindAboutExpiringDeferredOrder(PaymentScheduleItem scheduleItem,
                                                              WorkflowMessageSender workflowMessageSender) {
        return createIssueWorkflow(
                scheduleItem.getSchedule().getOrder().getId(),
                String.format(config.getDeferredPaymentReminderIssueTitle(),
                        scheduleItem.getSchedule().getOrder().getPrettyId()),
                starTrekUtils.prepareDescriptionForDeferredPaymentReminder(scheduleItem),
                config.getManualQueueName(), List.of("deferred_payments"), workflowMessageSender);
    }

    @TransactionMandatory
    public UUID createIssueAboutReturnedPaymentOrder(BankOrder bankOrder, String paymentBatchId,
                                                     WorkflowMessageSender workflowMessageSender) {
        if (Strings.isNullOrEmpty(config.getHotelAccountQueueName())) {
            return null;
        }
        Long contractId = bankOrder.getBankOrderPayment().getDetails().stream()
                .map(BankOrderDetail::getContractId)
                .distinct()
                .collect(CustomCollectors.exactlyOne());
        return createIssueWorkflow(
                null,
                String.format(config.getReturnedPaymentOrderIssueTitle(), bankOrder.getBankOrderId(), contractId),
                starTrekUtils.prepareDescriptionForReturnedPaymentOrderIssue(bankOrder, paymentBatchId),
                config.getHotelAccountQueueName(),
                List.of("returned_payment_order", String.format("contract_%s", contractId)), workflowMessageSender);
    }

    private UUID createIssueWorkflow(UUID id, String title, String description, String queue, List<String> tags,
                                     MessagingContext<?> messagingContext) {
        Ticket ticket = new Ticket();
        ticket.setState(ETicketState.TS_NEW);
        ticket.setId(UUID.randomUUID());
        ticket.setOrderId(id);
        ticket = ticketRepository.saveAndFlush(ticket);
        Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
        workflowForTicket = workflowRepository.saveAndFlush(workflowForTicket);
        messagingContext.scheduleExternalEvent(workflowForTicket.getId(), TCreateIssue.newBuilder()
                .setTitle(title)
                .setDescription(description)
                .setQueueName(Strings.nullToEmpty(queue))
                .setIssueType(config.getErrorIssueType())
                .addAllTags(tags)
                .build());
        return ticket.getId();
    }

    private UUID createIssueWorkflow(UUID id, String title, String description, String queue, List<String> tags,
                                     WorkflowProcessService workflowProcessService) {
        Ticket ticket = new Ticket();
        ticket.setState(ETicketState.TS_NEW);
        ticket.setId(UUID.randomUUID());
        ticket.setOrderId(id);
        ticket = ticketRepository.saveAndFlush(ticket);
        Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
        workflowForTicket = workflowRepository.saveAndFlush(workflowForTicket);
        workflowProcessService.scheduleEvent(workflowForTicket.getId(), TCreateIssue.newBuilder()
                .setTitle(title)
                .setDescription(description)
                .setQueueName(Strings.nullToEmpty(queue))
                .setIssueType(config.getErrorIssueType())
                .addAllTags(tags)
                .build());
        return ticket.getId();
    }

    private UUID createIssueWorkflow(UUID id, String title, String description, String queue, List<String> tags,
                                     WorkflowMessageSender workflowMessageSender) {
        Ticket ticket = new Ticket();
        ticket.setState(ETicketState.TS_NEW);
        ticket.setId(UUID.randomUUID());
        ticket.setOrderId(id);
        ticket = ticketRepository.saveAndFlush(ticket);
        Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
        workflowForTicket = workflowRepository.saveAndFlush(workflowForTicket);
        workflowMessageSender.scheduleEvent(workflowForTicket.getId(), TCreateIssue.newBuilder()
                .setTitle(title)
                .setDescription(description)
                .setQueueName(Strings.nullToEmpty(queue))
                .setIssueType(config.getErrorIssueType())
                .addAllTags(tags)
                .build());
        log.info("Created issue workflow: {}", ticket.getId());
        return ticket.getId();
    }

    public String createIssue(String title, String description, String queue, String issueType, ListF<String> tags) {
        if (Strings.isNullOrEmpty(queue)) {
            log.warn("Queue name not specified, skipping issue creation");
            return String.format("%s-%s", MISSING_QUEUE_NAME, Instant.now().getEpochSecond());
        }
        IssueCreate issueCreateRequest = IssueCreate.builder()
                .queue(queue)
                .summary(title)
                .type(issueType)
                .tags(tags)
                .description(description)
                .build();
        try {
            Issue issue = client.issues().create(issueCreateRequest);
            return issue.getKey();
        } catch (StartrekClientException e) {
            log.error("Error during creating startrek issue", e);
            if (e.getErrors().getStatusCode() >= 500) {
                throw new STClientRetryableException(e);
            } else {
                throw e;
            }
        }
    }

    public UUID linkIssueWorkflow(String key, UUID orderID) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(key), "Issue key should not be empty or null");
        Preconditions.checkNotNull(orderID, "Order ID should not be null");
        try {
            Preconditions.checkNotNull(client.issues().get(key));
        } catch (Exception e) {
            throw new RuntimeException("Startrek issue does not exist: " + key, e);
        }

        var orderTickets = ticketRepository.findAllByOrderId(orderID);
        if (orderTickets.stream().map(Ticket::getIssueId).anyMatch(key::equals)) {
            throw new RuntimeException(String.format("Startker issue %s already linked to order %s", key, orderID));
        }

        Ticket ticket = new Ticket();
        ticket.setState(ETicketState.TS_CREATED);
        ticket.setId(UUID.randomUUID());
        ticket.setOrderId(orderID);
        ticket.setIssueId(key);
        ticket = ticketRepository.saveAndFlush(ticket);
        Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
        workflowRepository.saveAndFlush(workflowForTicket);
        return ticket.getId();
    }

    public void commentIssue(String key, String comment) {
        if (key.startsWith(MISSING_QUEUE_NAME)) {
            log.warn("Attempt to comment an issue in a missing queue, skipping");
            return;
        }
        try {
            client.comments().create(key, CommentCreate.comment(comment).build());
        } catch (StartrekClientException e) {
            log.error("Error during creating startrek comment", e);
            if (e.getErrors().getStatusCode() >= 500) {
                throw new STClientRetryableException(e);
            } else {
                throw e;
            }
        }
    }

    /**
     * @return true if transition is performed successfully, false if the issue is already closed.
     * @throws IllegalStateException when issue doesn't have resolution and there's no closing transition
     */
    public boolean closeTicket(String ticketKey, String comment) {
        if (ticketKey.startsWith(MISSING_QUEUE_NAME)) {
            log.warn("Attempt to close an issue in a missing queue, skipping");
            return false;
        }
        try {
            Optional<Transition> optionalTransition = findCloseTransition(ticketKey);
            if (optionalTransition.isPresent()) {
                client.transitions().execute(
                        ticketKey,
                        optionalTransition.get(),
                        IssueUpdate
                                .comment(comment)
                                .resolution("fixed")
                                .build());
                return true;
            } else {
                return false;
            }
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new STClientRetryableException(e);
            } else {
                throw e;
            }
        }
    }

    private Optional<Transition> findCloseTransition(String ticketKey) {
        Optional<Transition> closeTransition = client.transitions()
                .getAll(ticketKey)
                .stream()
                .filter(transition -> transition.getTo().getKey().equals("closed"))
                .findFirst();
        if (closeTransition.isEmpty()) {
            Issue issue = client.issues().get(ticketKey);
            if (issue.getResolution().isEmpty()) {
                throw new IllegalStateException("Can't find close transition for not closed ticket. Key:" + ticketKey +
                        ". Current status: " + issue.getStatus().getKey());
            }
        }
        return closeTransition;
    }

    public Map<String, Integer> countOpenIssues(List<String> queueNames, int max) {
        return queueNames.stream().map(queue -> new Tuple2<>(queue, client.issues()
                .find(String.format("queue: %s resolution: empty()", queue))
                .take(max)
                .count())).collect(Collectors.toMap(Tuple2::get1, Tuple2::get2));
    }
}
