package ru.yandex.travel.orders.management;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.PriceCheckOutcome;
import ru.yandex.travel.orders.entities.TrainInsuranceRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.notifications.NotificationChannelType;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.entities.WorkflowStateTransition;

@Slf4j
@Service
@RequiredArgsConstructor
public class StarTrekUtils {
    private static final ZoneId DEFAULT_DISPLAYED_TIMEZONE = ZoneId.of("Europe/Moscow");
    private static final DateTimeFormatter DEFAULT_DT_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss O");
    private static final DateTimeFormatter RU_FORMATTER =
            DateTimeFormatter.ofLocalizedDateTime(FormatStyle.MEDIUM).withLocale(new Locale("ru"));

    private final StarTrekConfigurationProperties starTrekConfigurationProperties;

    private static void prepareException(StringBuilder builder, Throwable throwable) {
        builder
                .append("<{Исключение: ").append(throwable.getMessage()).append("\n")
                .append("%%").append(ExceptionUtils.getStackTrace(throwable)).append("%%}>\n");
    }

    private static void prepareException(StringBuilder builder, Workflow workflow) {
        Optional<WorkflowStateTransition> optionalTransition = workflow.getStateTransitions().stream()
                .filter(transition -> EWorkflowState.WS_CRASHED.equals(transition.getToState()))
                .max(Comparator.comparing(WorkflowStateTransition::getCreatedAt));
        if (optionalTransition.isPresent()) {
            TError error = (TError) optionalTransition.get().getData();
            builder.append("<{Исключение: ").append(error.getCode()).append(": ").append(error.getMessage()).append(
                    "\n");
            prepareStacktrace(builder, error, 0);
            builder.append("\n").append("}>").append("\n");
        } else {
            builder.append("Нет информации об исключении.");
        }
    }

    private static void prepareStacktrace(StringBuilder trace, TError error, int depth) {
        String tab = new String(new char[depth * 2]).replace('\0', ' ');
        trace.append(tab).append(error.getCode()).append(": ").append(error.getMessage()).append("\n");
        if (error.getAttributeMap().containsKey("class")) {
            trace.append(tab).append("class: ")
                    .append(error.getAttributeMap().get("class"));
        }
        if (error.getAttributeMap().containsKey("stack_trace")) {
            trace.append("%%").append(tab).append("stack_trace: ")
                    .append(error.getAttributeMap().get("stack_trace"))
                    .append("%%");
        }
        trace.append("----");
        for (String key : error.getAttributeMap().keySet()) {
            if (!"class".equals(key) && !"stack_trace".equals(key)) {
                trace
                        .append(tab).append(key).append(": ")
                        .append(tab).append(error.getAttributeMap().get(key)).append("\n");
            }
        }
        for (TError nestedError : error.getNestedErrorList()) {
            prepareStacktrace(trace, nestedError, depth + 1);
        }
    }

    private static String getFqdn() {
        String fqdn;
        try {
            fqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        } catch (UnknownHostException e) {
            fqdn = "Не удалось получить название хоста";
        }
        return fqdn;
    }

    public String prepareDescriptionForError(Order order, WorkflowEvent event, Instant happenedAt) {
        String teamName = getOrderTeamName(order);
        String teamDutyScheduleUrl = getOrderTeamDutyScheduleUrl(order);
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам в команду **").append(teamName)
                .append("**. ((").append(teamDutyScheduleUrl).append(" График дежурств)).").append("\n")
                .append("2. Памятка разработчику: ")
                .append(starTrekConfigurationProperties.getBoyDutyHowToUrl()).append("\n");
        //TODO add instruction filler
        builder
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("  * Время поломки: ").append(happenedAt.toString()).append("\n")
                .append("  * Состояние воркфлоу : ").append(order.getEntityState())
                .append("/").append(order.getWorkflow().getState()).append("\n")
                .append("  * ");
        prepareException(builder, order.getWorkflow());
        builder.append("\n");

        IntStream.range(0, order.getOrderItems().size()).forEach(idx -> {
            OrderItem item = order.getOrderItems().get(idx);
            builder
                    .append("* Услуга #").append(idx).append("\n")
                    .append("  * ID записи: ").append(item.getId()).append("\n")
                    .append("  * ID воркфлоу: ").append(item.getWorkflow().getId()).append("\n")
                    .append("  * Тип воркфлоу: ").append(item.getWorkflow().getEntityType()).append("\n")
                    .append("  * Состояние воркфлоу : ").append(item.getItemState())
                    .append("/").append(item.getWorkflow().getState()).append("\n")
                    .append("  * ");
            prepareException(builder, item.getWorkflow());
            builder.append("\n");
        });
        IntStream.range(0, order.getInvoices().size()).forEach(idx -> {
            Invoice invoice = order.getInvoices().get(idx);
            builder
                    .append("* Платёж #").append(idx).append("\n")
                    .append("  * ID записи: ").append(invoice.getId()).append("\n")
                    .append("  * ID воркфлоу: ").append(invoice.getWorkflow().getId()).append("\n")
                    .append("  * Тип воркфлоу: ").append(invoice.getWorkflow().getEntityType()).append("\n")
                    .append("  * Состояние воркфлоу : ").append(invoice.getInvoiceState())
                    .append("/").append(invoice.getWorkflow().getState()).append("\n")
                    .append("  * ");
            prepareException(builder, invoice.getWorkflow());
            builder.append("\n");
        });

        return builder.toString();
    }

    public String getOrderTeamTag(Order order) {
        switch (order.getDisplayType()) {
            case DT_AVIA:
                return "avia_duty_required";
            case DT_BUS:
            case DT_TRAIN:
                return "buses_duty_required";
            case DT_HOTEL:
                return "hotel_duty_required";
            case DT_SUBURBAN:
                return "rasp_duty_required";
            default:
                return "orders_duty_required";
        }
    }

    public String getVoucherTeamTag(EVoucherType voucherType) {
        switch (voucherType) {
            case VT_BUSES_TICKET:
                return "buses_duty_required";
            case VT_HOTELS:
            case VT_HOTELS_BUSINESS_TRIP:
                return "hotel_duty_required";
            default:
                return "orders_duty_required";
        }
    }

    private String getOrderTeamName(Order order) {
        switch (order.getDisplayType()) {
            case DT_AVIA:
                return "Авиа";
            case DT_BUS:
            case DT_TRAIN:
                return "ЖД и Автобусы";
            case DT_HOTEL:
                return "Отели";
            case DT_SUBURBAN:
                return "Электрички";
            default:
                return "Оркестратор";
        }
    }

    private String getOrderTeamDutyScheduleUrl(Order order) {
        StarTrekConfigurationProperties.Duty dutyUrls = starTrekConfigurationProperties.getDuty();
        switch (order.getDisplayType()) {
            case DT_AVIA:
                return dutyUrls.getAvia();
            case DT_BUS:
                return dutyUrls.getBuses();
            case DT_HOTEL:
                return dutyUrls.getHotels();
            case DT_SUBURBAN:
                return dutyUrls.getSuburban();
            case DT_TRAIN:
                return dutyUrls.getTrains();
            default:
                return dutyUrls.getOrchestrator();
        }
    }

    public String prepareDescriptionForOrderPaidHoldFailed(Order order, WorkflowEvent event,
                                                           Instant happenedAt) {
        //TODO change that to something meaningful
        return prepareDescriptionForError(order, event, happenedAt);
    }

    public String prepareDescriptionForPdfNotReceived(Order order, Workflow workflow, WorkflowEvent event) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("----\n");
        prepareException(builder, workflow);
        return builder.toString();
    }

    public String prepareDescriptionForNotificationNotSent(Order order, NotificationChannelType channel,
                                                           Workflow workflow, WorkflowEvent event) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Призвать в тикет дежурного разработчика.").append("\n")
                .append("2. Дежурному: перепослать письмо").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Канал: ").append(channel.getValue()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * Тип: ").append(order.getDisplayType()).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("  * Ссылка на PDF: ").append(order.getDocumentUrl())
                .append("----\n");
        prepareException(builder, workflow);
        return builder.toString();
    }

    public String prepareDescriptionForNonOrderNotificationNotSent(NotificationChannelType channel,
                                                                   Workflow workflow, WorkflowEvent event) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Призвать в тикет дежурного разработчика.").append("\n")
                .append("2. Разобраться в проблеме:").append("\n")
                .append("  * В месячных расшифровках может быть не указан/указан с ошибкой email, некорректная xlsx;").append("\n")
                .append("  * В расшифровках по ПП кроме проблем с email могут платежки находиться в неправильных статуса (НЕ CONCILED);").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Канал: ").append(channel.getValue()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n")
                .append("* Воркфлоу: ").append("\n")
                .append("  * ID: ").append(workflow.getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(workflow.getEntityType()).append("\n")
                .append("  * Состояние воркфлоу: ").append(workflow.getState()).append("\n")
                .append("  * Воркфлоу супервизор: ").append(workflow.getSupervisorId()).append("\n")
                .append("  * Время создания: ").append(workflow.getCreatedAt())
                .append("----\n");
        prepareException(builder, workflow);
        return builder.toString();
    }

    public String prepareDescriptionForPlusPointsTopupFailed(Order order, OrderItem orderItem, YandexPlusTopup topup,
                                                             Workflow workflow, WorkflowEvent event) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Призвать в тикет дежурного разработчика.").append("\n")
                .append("2. Разобраться почему не удалось, возможно повторить попытку.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n");
        if (order != null) {
            builder
                    .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                    .append("  * ID записи: ").append(order.getId()).append("\n")
                    .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        }
        if (orderItem != null) {
            builder
                    .append("* Услуга").append("\n")
                    .append("  * ID записи: ").append(orderItem.getId()).append("\n")
                    .append("  * Тип: ").append(orderItem.getPublicType().toString()).append("\n");
        }
        if (order == null && orderItem == null) {
            builder
                    .append("* Внешний заказ").append("\n")
                    .append("  * ID: ").append(topup.getExternalOrderId()).append("\n");
        }
        builder.append("* Top-up").append("\n")
                .append("  * ID записи: ").append(topup.getId()).append("\n")
                .append("  * Кол-во баллов: ").append(topup.getAmount()).append("\n")
                .append("----\n");
        prepareException(builder, workflow);
        return builder.toString();
    }

    public String prepareDescriptionForAeroflotLostOrder(AeroflotMqData order) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Нужно удостоверится, что заказ оформился. Инструкция по ссылке  https://wiki.yandex-team" +
                        ".ru/eva/Yandex.Travel/QAavia/#rabotasoshibkamivypiskibiletaajeroflota").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ \n")
                .append("  * Дата бронирования: ").append(DEFAULT_DT_FORMATTER.format(order.getBookingDateMsc())).append("\n")
                .append("  * Дата отправки уведомления: ").append(DEFAULT_DT_FORMATTER.format(order.getSentDateMsc())).append("\n")
                .append("  * PNR: ").append(order.getPnr()).append("\n")
                .append("  * PNR date: ").append(order.getPnrDate()).append("\n")
                .append("  * Билеты: ").append(order.getTickets()).append("\n")
                .append("  * <{Все данные уведомления: ").append("\n")
                .append("%%").append(order.getSourceMessage().replace("%%", " % % ")).append("%%}>\n");
        return builder.toString();
    }

    public String prepareDescriptionForAeroflotCancelledOrderPaid(AeroflotOrder order, String ownerLastName) {
        AeroflotServicePayload payload = order.getAeroflotOrderItem().getPayload();
        AeroflotOrderRef orderRef = payload.getBookingRef();
        String pnr = null;
        String pnrDate = null;
        ZonedDateTime bookingDate = order.getCreatedAt().atZone(DEFAULT_DISPLAYED_TIMEZONE);
        if (orderRef != null) {
            pnr = orderRef.getPnr();
            pnrDate = orderRef.getPnrDate();
        }
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Нужно удостоверится, что заказ оформился. Инструкция по ссылке  https://wiki.yandex-team" +
                        ".ru/eva/Yandex.Travel/QAavia/#rabotasoshibkamivypiskibiletaajeroflota").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * Дата бронирования: ").append(DEFAULT_DT_FORMATTER.format(bookingDate)).append("\n")
                .append("  * PNR: ").append(pnr).append("\n")
                .append("  * PNR date: ").append(pnrDate).append("\n")
                .append("  * На кого оформлены билеты: ").append(ownerLastName).append("\n")
                .append("  * Билеты: ").append(payload.getTickets().values()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForAeroflotFailedTokenization(Order order, String purchaseToken,
                                                                  String failureDetails) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам (призывом через комментарии, " +
                        "если таких тикетов 10 и более, то позвонить).").append("\n")
                .append("2. Разработчику завести тикет в Траст через форму ")
                .append(starTrekConfigurationProperties.getTrustSupportFormUrl())
                .append(" . Отдельно через дежурного попросить связаться с РБС, если проблема массовая.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("  * ID корзины: ").append(purchaseToken).append("\n")
                .append("  * <{Детали ошибки: \n")
                .append("%%").append(failureDetails.replace("%%", " % % ")).append("%%}>\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainInsuranceNotAdded(Order order) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Проверить массовость.").append("\n")
                .append("2. Передать тикет разработчикам, если проблема массовая.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainInsuranceNotConfirmed(Order order) {
        var builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainOfficeRefundHandleError(Order order, Exception e) {
        var builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("* Ошибка: ").append(e.toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainInsuranceRefundFailed(Order order, TrainInsuranceRefund refund) {
        var builder = new StringBuilder();
        Set<Integer> failedOperations = refund.getPayload().getItems().stream()
                .filter(i -> i.getRefundOperationStatus() != ImOperationStatus.OK)
                .map(InsuranceItemInfo::getBuyOperationId)
                .collect(Collectors.toSet());

        Money lostCost = Money.zero(ProtoCurrencyUnit.RUB);
        if (refund.getOrderItem().getPublicType() == EServiceType.PT_TRAIN) {
            var trainOrderItem = (TrainOrderItem) refund.getOrderItem();
            lostCost = trainOrderItem.getPayload().getPassengers().stream()
                    .filter(p -> p.getInsurance() != null && failedOperations.contains(p.getInsurance().getPartnerOperationId()))
                    .map(p -> p.getInsurance().getAmount())
                    .reduce(Money::add)
                    .orElse(Money.zero(ProtoCurrencyUnit.RUB));
        }
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Призвать кого:tyrande чтобы записать невозвращенные страховки в потери").append("\n")
                .append("2. Если проблема массовая призвать разработчиков.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Возврат: ").append(refund.getId().toString()).append("\n")
                .append("  * Провален возврат страховок: ").append(failedOperations.size()).append(" шт.\n")
                .append("  * Сумма потерь: ").append(lostCost.toString()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainTicketRefundFailed(Order order, TrainTicketRefund refund) {
        var builder = new StringBuilder();
        String refundedBlanks = refund.getPayload().getRefundedItems().stream()
                .map(passengerRefundInfo -> String.valueOf(passengerRefundInfo.getBlankId()))
                .collect(Collectors.joining(", "));
        String refundFailedBlanks = refund.getPayload().getItems().stream()
                .filter(x -> x.getRefundOperationStatus() != ImOperationStatus.OK)
                .map(passengerRefundInfo -> String.valueOf(passengerRefundInfo.getBlankId()))
                .collect(Collectors.joining(", "));
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Если проблема массовая, то передать разработчику, если нет - закрыть").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Возврат: ").append(refund.getId().toString()).append("\n")
                .append("  * возвращены бланки: ").append(refundedBlanks).append("\n")
                .append("  * провален возврат бланков: ").append(refundFailedBlanks).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrainInsuranceInvalidAmount(
            Order order, BigDecimal expectedAmount, BigDecimal obtainedAmount) {
        String failureDetails = MessageFormat.format("ожидалась сумма {0}, получена {1}",
                expectedAmount, obtainedAmount);
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n")
                .append("  * Детали ошибки: ").append(failureDetails).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForEmptyTrainCarrierInn(Order order) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("В данных от партнёра не был указан ИНН перевозчика, подставили ИНН партнёра.").append("\n")
                .append("===Инструкция для оператора").append("\n")
                .append("1. Проверить массовость, передать разработчику").append("\n")
                .append("2. Разработчику связаться с партнёром.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID заказа: ").append(order.getId()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForTrustReceiptNotFetched(UUID orderId, String prettyId, Order order,
                                                              List<UUID> failedAttachmentIds) {
        String failureDetails = failedAttachmentIds.stream().map(UUID::toString).collect(Collectors.joining(","));
        String orderLink = order == null ? getOrderAdminLink(orderId, prettyId) : getOrderAdminLink(order);
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("* Через час-два после создания тикета проверить в админке доступен ли чек на принятие денег" +
                        " и возврат (если был clear).").append("\n")
                .append("* Если чек был - закрываем тикет.").append("\n")
                .append("* Если чека на возврат нет и в скрудже у возврата Status=error:").append("\n")
                .append("  * повторить возврат в админке в блоке 'Повторная попытка возврата средств'").append("\n")
                .append("  * если попытка возврата успешная, то можно закрывать таск").append("\n")
                .append("  * если появился новый привязанный к заказу таск \"Ошибка при возврате денег по заказу\", " +
                        "то закрываем новый таск как дубль и создаем тикет в траст по ((")
                .append(starTrekConfigurationProperties.getTrustSupportFormUrl()).append(" форме)).\n")
                .append("* Если что-то другое, то передаем разработчику:").append("\n")
                .append("  * Разработчику проверить из-за чего не проводит генерация чека или его скачивание." +
                        "Частая причина - подвисла печать. Определить можно по множеству однообразных сообщений " +
                        "про попытку скачать %%fiscal receipt%% (использовать для поиска по логу): " +
                        "<[Fetching fiscal receipt (id: 0000000, type: ACQUIRE)]>\n")
                .append("  * Если дело в Траст-е, то завести им тикет через ((")
                .append(starTrekConfigurationProperties.getTrustSupportFormUrl()).append(" форму)) .\n")
                .append("  * Если чек в итоге успешно скачался и доступен в админке, то закрываем тикет задачу.\n")
                .append("  * Если письмо о возврате, убедиться, что возврат прошел, иначе перезапустить.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ ").append(orderLink).append("\n")
                .append("  * Незагруженные вложения: ").append(failureDetails).append("\n");
        if (order != null) {
            builder
                    .append("  * ID записи: ").append(order.getId()).append("\n")
                    .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                    .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                    .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        }

        return builder.toString();
    }

    public String prepareDescriptionForGenericWorkflowError(Workflow workflow, WorkflowEvent event) {
        StringBuilder builder = new StringBuilder();
        builder
                .append("===Инструкция для оператора").append("\n")
                .append("1. Передать тикет разработчикам.").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Событие").append("\n")
                .append("  * ID: ").append(event.getId()).append("\n")
                .append("  * Тип: ").append(event.getData().getClass().getName()).append("\n")
                .append("  * Кол-во попыток: ").append(event.getTimesTried()).append("\n")
                .append("* Workflow \n")
                .append("  * ID: ").append(workflow.getId()).append("\n")
                .append("  * Entity ID: ").append(workflow.getEntityId()).append("\n")
                .append("  * Entity Type: ").append(workflow.getEntityType()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForHotelOrderPriceMismatch(Order order, BigDecimal actualPrice,
                                                               BigDecimal expectedPrice, boolean isExceeding,
                                                               boolean isRefundable,
                                                               PriceCheckOutcome outcome) {
        StringBuilder builder = new StringBuilder();
        String spreadStr = isExceeding ? "**превышает**" : "**не превышает**";
        String refundableStr = isRefundable ? "**возращаемый**" : "**невозвращаемый**";
        String outcomeStr = "";
        switch (outcome) {
            case CANCELLED:
                outcomeStr = "**отменено**";
                break;
            case CONFIRMED:
                outcomeStr = "**подтверждено**";
                break;
            case CRASHED:
                outcomeStr = "**приостановлено**";
                break;
        }
        builder.append("===Инструкция для оператора").append("\n")
                .append("Расхождение цены ").append(spreadStr).append(" допустимый предел, заказ ").append(refundableStr).append(".\n")
                .append("Бронирование будет ").append(outcomeStr).append(".\n")
                .append("Необходимо:").append("\n");
        if (outcome == PriceCheckOutcome.CONFIRMED) {
            builder.append("* Ввести в табличку https://nda.ya.ru/3VodQ5 информацию о списании убытка в ")
                    .append(actualPrice.subtract(expectedPrice).setScale(2, RoundingMode.HALF_UP))
                    .append(" руб;\n");
        }
        builder.append("* Призвать в тикет разработчиков для расследования причин расхождения.").append("\n");
        if (outcome != PriceCheckOutcome.CRASHED) {
            builder.append("\n").append("**Дежурных ночью будить не надо, просто призовите в тикет**").append("\n");
        }
        builder
                .append("===Техническая информация").append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("* Цена в заказе у партнера: ").append(actualPrice.setScale(2, RoundingMode.HALF_UP).toString()).append("\n")
                .append("* Расчитанная цена: ").append(expectedPrice.setScale(2, RoundingMode.HALF_UP).toString()).append("\n")
                .append("* Заказ отменяемый: ").append(isRefundable ? "да" : "нет").append("\n");

        return builder.toString();
    }

    public String prepareDescriptionForManualServiceProcessing(Order order, ManualServiceProcessingTicketData data) {
        StringBuilder builder = new StringBuilder();
        builder.append(String.format("Невозможно автоматически подтвердить бронирование заказа %s у партнера «%s»:",
                prettyIdNoFormatting(data.getOrderPrettyId()), data.getPartnerName())).append("\n")
                .append(String.format("**%s**", data.getReason())).append("\n");
        if (data.getInstructions() != null) {
            builder.append("===Инструкция для оператора:").append("\n");
            builder.append(data.getInstructions()).append("\n");
        }

        builder.append("===Информация о заказе").append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("* Расчитанная цена: ").append(order.calculateTotalCost().toString()).append("\n")
                .append("* Заказ отменяемый: ").append(data.isServiceRefundable() ? "да" : "нет").append("\n")
                .append("* Есть бронирование у партнера: ").append(data.isHasReservationAtPartner() ? "да" : "нет").append("\n")
                .append("* Взяты деньги с клиента: ").append(data.isHasMoneyOnHold() ? "да" : "нет").append("\n");

        if (data.getExtraDetails() != null) {
            for (var detail : data.getExtraDetails().entrySet()) {
                builder.append("* ").append(detail.getKey()).append(": ").append(detail.getValue()).append("\n");
            }
        }
        return builder.toString();
    }

    public String prepareDescriptionForTrustRefundError(TrustInvoice invoice, UUID trustRefundId, String reason) {
        Order order = invoice.getOrder();
        StringBuilder builder = new StringBuilder()
                .append("Ошибка при возврате денег по заказу :").append(prettyIdNoFormatting(order.getPrettyId())).append("\n")
                .append("Причина: ").append(reason).append("\n")
                .append("===Инструкция для оператора").append("\n")
                .append("1. Повторить возврат в админке в блоке 'Повторная попытка возврата средств'").append("\n")
                .append("2. Если новых тикетов в ST нет и новый возврат в трасте успешный, то можно закрывать").append("\n")
                .append("3. Если повторная попытка снова с ошибкой, то завести тикет в Траст через форму ((")
                .append(starTrekConfigurationProperties.getTrustSupportFormUrl()).append(" форму))").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Заказ: ").append(getOrderAdminLink(order)).append("\n")
                .append("* Id заказа: ").append(invoice.getOrder().getId()).append("\n")
                .append("* Id платежа: ").append(invoice.getId()).append("\n")
                .append("* Id возврата платежа: ").append(trustRefundId).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForInvoiceNotRefunded(Order order, UUID orderRefundId, UUID invoiceId, String reason) {
        var builder = new StringBuilder();
        builder
                .append("Ошибка при возврате денег по заказу :").append(prettyIdNoFormatting(order.getPrettyId())).append("\n")
                .append("Причина: ").append(reason).append("\n")
                .append("===Инструкция для оператора").append("\n")
                .append("1. Повторить возврат в админке в блоке 'Повторная попытка возврата средств'").append("\n")
                .append("2. Если новых тикетов в ST нет и новый возврат в трасте успешный, то можно закрывать").append("\n")
                .append("3. Если повторная попытка снова с ошибкой, то завести тикет в Траст через форму ((")
                .append(starTrekConfigurationProperties.getTrustSupportFormUrl()).append(" форму))").append("\n")
                .append("===Техническая информация").append("\n")
                .append("* Хост: ").append(getFqdn()).append("\n")
                .append("* Возврат: ").append(orderRefundId == null ? "null" : orderRefundId.toString()).append("\n")
                .append("* Invoice: ").append(invoiceId).append("\n")
                .append("* Заказ ").append(getOrderAdminLink(order)).append("\n")
                .append("  * ID записи: ").append(order.getId()).append("\n")
                .append("  * Состояние заказа: ").append(order.getEntityState()).append("\n")
                .append("  * ID воркфлоу: ").append(order.getWorkflow().getId()).append("\n")
                .append("  * Тип воркфлоу: ").append(order.getWorkflow().getEntityType()).append("\n")
                .append("  * Состояние воркфлоу: ").append(order.getWorkflow().getState()).append("\n")
                .append("  * Время создания: ").append(order.getCreatedAt().toString()).append("\n");
        return builder.toString();
    }

    public String prepareDescriptionForDeferredPaymentReminder(PaymentScheduleItem item) {
        Order order = item.getSchedule().getOrder();

        String paymentEndsAt = RU_FORMATTER.format(item.getPaymentEndsAt().atZone(DEFAULT_DISPLAYED_TIMEZONE));
        String penaltyString = "";
        if (!item.getPenaltyIfUnpaid().isZero()) {
            penaltyString = String.format(", а из суммы предоплаты будет удержан штраф %s",
                    item.getPenaltyIfUnpaid().toString());
        }
        String template = "" +
                "Заказ {0}, созданный на условиях оплаты в рассрочку, до сих пор полностью не оплачен. " +
                "До конца периода оплаты осталось менее суток.\n" +
                "Если пользователь не проведет оплату до {1}, заказ отменится{2}.\n\n " +
                "Необходимо связаться с пользователем и напомнить ему о необходимости оплаты.\n\n " +
                "===Информация о заказе\n" +
                "* Заказ {0}\n" +
                "* Общая стоимость: {3}\n" +
                "* Внесено: {4}\n" +
                "* Необходимо доплатить: {5}\n" +
                "* Крайний срок оплаты: {1}\n" +
                "* Штраф при возврате в случае неуплаты: {6}\n";
        return MessageFormat.format(template,
                getOrderAdminLink(order),
                paymentEndsAt,
                penaltyString,
                order.calculateTotalCost(),
                item.getSchedule().getPaidAmount(),
                item.getPendingInvoice().getTotalAmount(),
                item.getPenaltyIfUnpaid());
    }

    public String prepareDescriptionForReturnedPaymentOrderIssue(BankOrder bankOrder, String paymentBatchId) {
        Long contractId = bankOrder.getBankOrderPayment().getDetails().stream()
                .map(BankOrderDetail::getContractId)
                .distinct()
                .collect(CustomCollectors.exactlyOne());
        String template = "" +
                "Ранее отправленное платежное поручение {0} по контракту с ID {1} перешло в статус RETURNED.\n" +
                "Вероятнее всего это связано с тем, что для контрагента указаны неверные реквизиты, и банк вернул " +
                "платеж.\n" +
                "\n" +
                "Стоит проверить реквизиты, связаться с контрагентом и тп.\n\n" +
                "===Дополнительная информация:\n" +
                "* Платежное поручение: {2}\n" +
                "* Payment batch: {3}\n" +
                "* ContractID: {1}\n" +
                "* Ссылка на дашборд: {4}\n";
        return MessageFormat.format(template,
                bankOrder.getBankOrderId(),
                String.valueOf(contractId),
                bankOrder.getDescription(),
                paymentBatchId,
                getPaymentDashboardLink(bankOrder.getBankOrderId()));
    }

    private String prettyIdNoFormatting(String prettyId) {
        // ST parses our pretty IDs as ticket links: "YA-1234" + "-...."
        // let's prevent it to confuse people less
        return "%%" + prettyId + "%%";
    }

    private String getOrderAdminLink(Order order) {
        return getOrderAdminLink(order.getId(), order.getPrettyId());
    }

    private String getOrderAdminLink(UUID orderId, String prettyId) {
        if (Strings.isNullOrEmpty(starTrekConfigurationProperties.getTravelAdminUrl())) {
            return prettyId;
        } else {
            var url = String.format(starTrekConfigurationProperties.getTravelAdminUrl(), orderId);
            return String.format("((%s %s))", url, prettyId);
        }
    }

    private String getPaymentDashboardLink(String bankOrderId) {
        return String.format(starTrekConfigurationProperties.getPaymentDashboardLink(), bankOrderId);
    }
}
