package ru.yandex.travel.orders.management.metrics;

import java.math.BigInteger;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.persistence.DiscriminatorValue;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ProtocolMessageEnum;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

// TODO (mbobrov): refactor this code to add modularity

@RequiredArgsConstructor
@Slf4j
public class GaugeService implements ApplicationContextAware, InitializingBean, DisposableBean {
    private static final String SENSOR_FINANCIAL_PROCESSOR_DELAY = "financialProcessorDelay";

    private final OrderRepository orderRepository;
    private final WorkflowRepository workflowRepository;
    private final StarTrekService starTrekService;
    private final TransactionTemplate transactionTemplate;
    private final WorkflowProcessService workflowProcessService;
    private final GaugeServiceProperties properties;
    private final Map<String, AtomicReference<Map<GaugeKey, Double>>> meteredValues = new ConcurrentHashMap<>();

    private final AtomicBoolean orderSupervisorIsAlive = new AtomicBoolean(true);
    private final AtomicBoolean genericSupervisorIsAlive = new AtomicBoolean(true);
    private final Map<String, Map<Integer, String>> orderStateMaps = mapEntityStates(Order.class, "getState");
    private final Map<String, Set<EDisplayOrderType>> displayTagsMap = mapOrderDisplayTypes();

    @SuppressWarnings("rawtypes")
    private Collection<TaskProcessor> taskProcessors;
    private Collection<FinancialEventProcessor> financialEventProcessors;
    private Collection<GaugeServiceDbMetric> dbMetrics;

    private final AtomicInteger workflowsInProcessCount = new AtomicInteger(0);
    private final AtomicInteger pendingWorkflowsCount = new AtomicInteger(0);

    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(
            new ThreadFactoryBuilder()
                    .setNameFormat("bg-gauge-updater")
                    .setDaemon(true)
                    .build());

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        // TODO (mbobrov): refactor it to configuration
        taskProcessors = applicationContext.getBeansOfType(TaskProcessor.class).values();
        financialEventProcessors = applicationContext.getBeansOfType(FinancialEventProcessor.class).values();
        dbMetrics = applicationContext.getBeansOfType(GaugeServiceDbMetric.class).values();
    }

    @Override
    public void afterPropertiesSet() {
        registerPossibleMetrics();
        startCollectingMetrics();
    }

    private static <T> Set<Set<T>> cartesianProduct(Set<T>... sets) {
        Preconditions.checkArgument(sets.length >= 2, "at least 2 sets are required");
        return cartesianProductRecursive(0, sets);
    }

    private static <T> Set<Set<T>> cartesianProductRecursive(int index, Set<T>... sets) {
        var res = new HashSet<Set<T>>();
        if (index == sets.length) {
            res.add(new HashSet<>());
        } else {
            for (T t : sets[index]) {
                for (Set<T> set : cartesianProductRecursive(index + 1, sets)) {
                    set.add(t);
                    res.add(set);
                }
            }
        }
        return res;
    }

    private void updateFromDb() {
        try {
            transactionTemplate.execute(ignored -> {
                updateOrders();
                updateWorkflows();
                updateSupervisorState();
                updateTaskProcessors();
                updateFinancialStats();
                updateOtherDbMetrics();
                return null;
            });
        } catch (Exception e) {
            log.error("Unable to update db metrics", e);
        }
    }

    private void updateFromSt() {
        try {
            log.debug("Updating ST issue statistics");
            var gaugeMap = new HashMap<GaugeKey, Double>();
            var name = getPrefixedGaugeName("st");
            var issuesPerQueue = starTrekService.countOpenIssues(properties.getSt().getQueueNames(),
                    properties.getSt().getMaxIssues());
            issuesPerQueue.forEach((queue, count) -> {
                var tags = Set.of(new GaugeTag("queue", queue));
                var key = new GaugeKey(name, tags);
                gaugeMap.put(key, Double.valueOf(count));
            });
            meteredValues.get(name).set(gaugeMap);
        } catch (Exception e) {
            log.error("Unable to update st metrics", e);
        }
    }

    private void updateSupervisorState() {
        orderSupervisorIsAlive.set(workflowRepository.getOne(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid()).getState() == EWorkflowState.WS_RUNNING);
        genericSupervisorIsAlive.set(workflowRepository.getOne(WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid()).getState() == EWorkflowState.WS_RUNNING);
    }

    HashMap<GaugeKey, Double> getUpdateOrdersMap(String name) {
        var gaugeMap = new HashMap<GaugeKey, Double>();
        orderRepository.countOrdersByTypesAndState().forEach(fields -> {
            String orderType = (String) fields[0];
            EDisplayOrderType orderDisplayType = EDisplayOrderType.forNumber((Integer) fields[1]);
            int orderStateValue = (Integer) fields[2];
            int count = ((BigInteger) fields[3]).intValue();

            String orderState = null;
            if (orderStateMaps.containsKey(orderType)) {
                orderState = orderStateMaps.get(orderType).get(orderStateValue);
            }
            if (orderState == null) {
                log.error("Unable to map state {} for order type {}", orderStateValue, orderType);
            } else {
                Set<GaugeTag> tags = Set.of(
                        new GaugeTag("type", orderType),
                        new GaugeTag("state", orderState),
                        new GaugeTag("displayType", orderDisplayType.toString())
                );
                GaugeKey key = new GaugeKey(name, tags);
                gaugeMap.put(key, (double) count);
            }
        });
        return gaugeMap;
    }

    private void updateOrders() {
        log.debug("Updating order statistics");
        String name = getPrefixedGaugeName("orders");
        meteredValues.get(name).set(getUpdateOrdersMap(name));
    }

    private void updateWorkflows() {
        log.debug("Updating workflow statistics");
        var gaugeMap = new HashMap<GaugeKey, Double>();
        String name = getPrefixedGaugeName("wf");
        workflowRepository.countWorkflowByStateAndEntityType().forEach(fields -> {
            String entityType = (String) fields[0];
            EWorkflowState state = (EWorkflowState) fields[1];
            int count = ((Long) fields[2]).intValue();
            Set<GaugeTag> tags = Set.of(
                    new GaugeTag("entityType", entityType),
                    new GaugeTag("state", state.toString())
            );
            GaugeKey key = new GaugeKey(name, tags);
            gaugeMap.put(key, (double) count);
        });
        meteredValues.get(name).set(gaugeMap);

        pendingWorkflowsCount.set(workflowProcessService.getPendingWorkflowsCount());
        workflowsInProcessCount.set(workflowProcessService.getWorkflowsInProcessCount());

        // letting the impl re-calculate its db-based metrics under the common transaction
        workflowProcessService.updateMetrics();
    }

    private void updateTaskProcessors() {
        log.debug("Updating task processors");
        var gaugeMap = new HashMap<GaugeKey, Double>();
        String name = "taskProcessor.pendingTasks";
        for (TaskProcessor<?> tp : taskProcessors) {
            GaugeKey key = new GaugeKey(name, Set.of(new GaugeTag("name", tp.getName())));
            gaugeMap.put(key, (double) tp.getPendingTasksCount());
        }
        meteredValues.get(name).set(gaugeMap);
    }

    private void updateFinancialStats() {
        var gaugeMap = new HashMap<GaugeKey, Double>();
        for (FinancialEventProcessor feProcessor : financialEventProcessors) {
            GaugeKey key = new GaugeKey(SENSOR_FINANCIAL_PROCESSOR_DELAY, Set.of(new GaugeTag("name",
                    feProcessor.getName())));
            gaugeMap.put(key, (double) feProcessor.getCurrentProcessingDelay().toMillis());
        }
        // these metrics could be disabled, e.g. in integration tests
        if (meteredValues.containsKey(SENSOR_FINANCIAL_PROCESSOR_DELAY)) {
            meteredValues.get(SENSOR_FINANCIAL_PROCESSOR_DELAY).set(gaugeMap);
        }
    }

    private void updateOtherDbMetrics() {
        var gaugeMaps = new HashMap<String, HashMap<GaugeKey, Double>>();
        for (GaugeServiceDbMetric metric : dbMetrics) {
            GaugeKey key = new GaugeKey(metric.getSensorName(), Set.of(new GaugeTag("name", metric.getMetricName())));
            gaugeMaps.computeIfAbsent(metric.getSensorName(), k -> new HashMap<>()).put(key, metric.getValue());
        }
        for (Map.Entry<String, HashMap<GaugeKey, Double>> gaugeMapEntry : gaugeMaps.entrySet()) {
            meteredValues.get(gaugeMapEntry.getKey()).set(gaugeMapEntry.getValue());
        }
    }

    private Set<GaugeTag> getAllOrderTypeTags() {
        return orderStateMaps.keySet().stream().map(type -> new GaugeTag("type", type)).collect(Collectors.toSet());
    }

    private Set<GaugeTag> getAllOrderStateTags() {
        return orderStateMaps.values().stream().flatMap(v -> v.values().stream()).map(v -> new GaugeTag("state", v)).collect(Collectors.toSet());
    }

    private Set<GaugeTag> getAllSTQueueTags() {
        return properties.getSt().getQueueNames().stream().map(name -> new GaugeTag("queue", name)).collect(Collectors.toSet());
    }

    private Set<GaugeTag> getAllWorkflowEntityTypeTags() {
        return Arrays.stream(WellKnownWorkflowEntityType.values())
                .map(WellKnownWorkflowEntityType::getDiscriminatorValue)
                .map(s -> new GaugeTag("entityType", s))
                .collect(Collectors.toSet());
    }

    private Set<GaugeTag> getAllWorkflowStateTags() {
        return Arrays.stream(EWorkflowState.values())
                .map(Object::toString)
                .filter(s -> !s.equals("UNRECOGNIZED"))
                .map(s -> new GaugeTag("state", s))
                .collect(Collectors.toSet());
    }

    private void registerPossibleMetrics() {
        registerSupervisorGauges();
        registerOrderGauges();
        registerGauges(getPrefixedGaugeName("wf"), getAllWorkflowEntityTypeTags(), getAllWorkflowStateTags());
        registerGauges(getPrefixedGaugeName("st"), getAllSTQueueTags());

        for (TaskProcessor<?> taskProcessor : taskProcessors) {
            registerGauges("taskProcessor.pendingTasks", Set.of(new GaugeTag("name", taskProcessor.getName())));
        }
        for (FinancialEventProcessor feProcessor : financialEventProcessors) {
            registerGauges(SENSOR_FINANCIAL_PROCESSOR_DELAY, Set.of(new GaugeTag("name", feProcessor.getName())));
        }
        for (GaugeServiceDbMetric metric : dbMetrics) {
            registerGauges(metric.getSensorName(), Set.of(new GaugeTag("name", metric.getMetricName())));
        }

        Gauge.builder(getPrefixedGaugeName("workflowsInProcess"), workflowsInProcessCount::get).register(Metrics.globalRegistry);
        Gauge.builder(getPrefixedGaugeName("pendingWorkflows"), pendingWorkflowsCount::get).register(Metrics.globalRegistry);
        Gauge.builder(getPrefixedGaugeName("maxWorkflows"), workflowProcessService::getMaxWorkflows).register(Metrics.globalRegistry);
    }

    private void startCollectingMetrics() {
        if (!properties.isSchedulerEnabled()) {
            return;
        }

        executor.scheduleWithFixedDelay(this::updateFromDb, 0, properties.getDb().getUpdateInterval().toMillis(),
                TimeUnit.MILLISECONDS);

        if (properties.getSt().getEnabled()) {
            executor.scheduleWithFixedDelay(this::updateFromSt, 0,
                    properties.getSt().getUpdateInterval().toMillis(),
                    TimeUnit.MILLISECONDS);
        }
    }

    private Map<String, Map<Integer, String>> mapEntityStates(Class<?> baseClass, String stateAccessorMethod) {
        var res = new HashMap<String, Map<Integer, String>>();

        var reflections = new Reflections("ru.yandex.travel.orders.entities");
        for (var subtype : reflections.getSubTypesOf(baseClass)) {
            var annotation = subtype.getAnnotation(DiscriminatorValue.class);
            String discriminatorValue = annotation.value();
            try {
                var stateEnumType =
                        subtype.getMethod(stateAccessorMethod).getReturnType().asSubclass(ProtocolMessageEnum.class);
                var valueMethod = stateEnumType.getMethod("getNumber");
                Map<Integer, String> stateMap = new HashMap<>();
                for (var value : stateEnumType.getEnumConstants()) {
                    if ("UNRECOGNIZED".equals(value.toString())) {
                        continue;
                    }
                    Integer intValue = (Integer) valueMethod.invoke(value);
                    stateMap.put(intValue, value.toString());
                }
                res.put(discriminatorValue, stateMap);
            } catch (Exception e) {
                log.error("Unable to introspect types", e);
            }
        }
        return res;
    }

    private Map<String, Set<EDisplayOrderType>> mapOrderDisplayTypes() {
        return Map.of(
                WellKnownWorkflowEntityType.HOTEL_ORDER.getDiscriminatorValue(), Set.of(EDisplayOrderType.DT_HOTEL),
                WellKnownWorkflowEntityType.AEROFLOT_ORDER.getDiscriminatorValue(), Set.of(EDisplayOrderType.DT_AVIA),
                WellKnownWorkflowEntityType.TRAIN_ORDER.getDiscriminatorValue(), Set.of(EDisplayOrderType.DT_TRAIN),
                WellKnownWorkflowEntityType.GENERIC_ORDER.getDiscriminatorValue(),
                Arrays.stream(EDisplayOrderType.values()).collect(Collectors.toSet())
        );
    }

    private void registerSupervisorGauges() {
        Gauge.builder(properties.getGaugePrefix() + ".wf.supervisor", () -> orderSupervisorIsAlive.get() ? 1 : 0)
                .register(Metrics.globalRegistry);
        Gauge.builder(
                properties.getGaugePrefix() + ".wf.genericSupervisor", () -> genericSupervisorIsAlive.get() ? 1 : 0
        ).register(Metrics.globalRegistry);
    }

    private void registerOrderGauges() {
        for (var entry : orderStateMaps.entrySet()) {
            var typeTags = Set.of(new GaugeTag("type", entry.getKey()));
            var displayTypeTags = displayTagsMap.get(entry.getKey()).stream()
                    .map(displayType -> new GaugeTag("displayType", displayType.toString())).collect(Collectors.toSet());
            var stateTags =
                    entry.getValue().values().stream().map(state -> new GaugeTag("state", state)).collect(Collectors.toSet());

            registerGauges(getPrefixedGaugeName("orders"), typeTags, displayTypeTags, stateTags);
        }
    }

    private void registerGauges(String name, Set<GaugeTag>... allTags) {
        meteredValues.putIfAbsent(name, new AtomicReference<>(null));
        var tagCombinations = (allTags.length == 1) ? Set.of(allTags[0]) : cartesianProduct(allTags);
        for (var tags : tagCombinations) {
            var key = new GaugeKey(name, tags);
            var gauge = Gauge.builder(name, () -> {
                var gauges = meteredValues.get(name).get();
                if (gauges != null && gauges.get(key) != null) {
                    return gauges.get(key);
                } else {
                    return null;
                }
            });
            for (var tag : tags) {
                gauge.tag(tag.name, tag.value);
            }
            gauge.register(Metrics.globalRegistry);
        }
    }

    String getPrefixedGaugeName(String name) {
        return properties.getGaugePrefix() + "." + name;
    }

    @Override
    public void destroy() {
        Duration timeout = properties.getShutdownTimeout();
        MoreExecutors.shutdownAndAwaitTermination(executor, timeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Data
    static class GaugeKey {
        private final String name;
        private final Set<GaugeTag> tags;
    }

    @Value
    static class GaugeTag {
        private final String name;
        private final String value;
    }
}
