package ru.yandex.travel.orders.repository;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collection;
import java.util.List;

import javax.persistence.Tuple;

import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;

import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.BillingTransactionIdProjection;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;

public interface BillingTransactionRepository extends JpaRepository<BillingTransaction, Long> {
    Collection<Long> NO_EXCLUDE_IDS = List.of(-1L);

    BillingTransaction findBySourceFinancialEventAndPaymentTypeAndPaymentSystemType(
            FinancialEvent event,
            BillingTransactionPaymentType paymentType,
            BillingTransactionPaymentSystemType paymentSystemType
    );

    List<BillingTransaction> findAllByServiceOrderId(String prettyId);

    List<BillingTransaction> findAllBySourceFinancialEvent(FinancialEvent financialEvent);

    List<BillingTransaction> findAllByYtIdInAndPaymentTypeIn(Collection<Long> ytIds,
                                                             Collection<BillingTransactionPaymentType> paymentType);

    @Query(value = "select count(*) from billing_transactions t " +
            "join billing_partner_configs c on t.partner_id = c.billing_client_id " +
            "where yt_id is null and payout_at < ?1 and c.export_to_yt = true and t.paused = false", nativeQuery = true)
    long countReadyTransactionsWithoutYtId(Instant maxPayoutAt);

    /**
     * We prepare an ordered generated ids set to allow earlier transactions have lesser yt ids.
     * It's also expected that all refund transactions have identifiers greater
     * than the ids of their according payments.
     */
    @Modifying
    @Query(value = "" +
            "with ids as (" +
            "    select t.id, nextval('billing_transactions_yt_id_seq') yt_id " +
            "    from billing_transactions t " +
            "    where yt_id is null and payout_at < ?1 and exists (" +
            "       select 1 from billing_partner_configs c" +
            "       where c.billing_client_id = t.partner_id and c.export_to_yt = true" +
            "    ) and t.paused = false " +
            "    order by t.payout_at asc, t.id asc " +
            ") " +
            "update billing_transactions t " +
            "set yt_id = (select yt_id from ids where ids.id = t.id) " +
            "where t.id in (select id from ids)", nativeQuery = true)
    int generateNewYtIdsForReadyTransactions(Instant maxPayoutAt);

    @Query(value = "select min(payout_at), count(*) from billing_transactions t " +
            "join billing_partner_configs c on t.partner_id = c.billing_client_id " +
            "where yt_id is null and payout_at < ?1 and c.export_to_yt = true and t.paused = false", nativeQuery = true)
    Tuple findOldestTimestampReadyForYtIdGenerationRaw(Instant maxPayoutAt);

    default ProcessingTasksInfo findOldestTimestampReadyForYtIdGeneration(Instant maxPayoutAt) {
        return convertProcessingTasksInfo(findOldestTimestampReadyForYtIdGenerationRaw(maxPayoutAt));
    }

    @Query("select min(t.payoutAt), count(*) from BillingTransaction t where ytId is not null and exportedToYt = false")
    Tuple findOldestTimestampReadyForYtExportRaw();

    default ProcessingTasksInfo findOldestTimestampReadyForYtExport() {
        return convertProcessingTasksInfo(findOldestTimestampReadyForYtExportRaw());
    }

    @Query(value = "SELECT b FROM BillingTransaction b " +
            "WHERE b.ytId is not null and b.exportedToYt = false")
    List<BillingTransaction> findReadyForExport(Pageable pageable);

    @Query("select count(*) from BillingTransaction t where ytId is not null and exportedToYt = false")
    long countReadyForExport();

    @Query("select t.id from BillingTransaction t where t.exportedToYt = true and t.actCommitted = false " +
            "and t.payoutAt < ?1 and t.accountingActAt < ?2 and t.id not in ?3")
    List<Long> findIdsReadyForActCommit(Instant maxPayoutAt, Instant maxActAt, Collection<Long> excludeIds,
                                        Pageable pageable);

    @Query("select count(t) from BillingTransaction t where t.exportedToYt = true and t.actCommitted = false " +
            "and t.payoutAt < ?1 and t.accountingActAt < ?2 and t.id not in ?3")
    long countReadyForActCommit(Instant maxPayoutAt, Instant maxActAt, Collection<Long> excludeIds);


    @Query("SELECT b FROM BillingTransaction b WHERE b.partnerId = ?1 AND b.exportedToYtAt >= ?2 AND b.exportedToYtAt <= ?3" +
            " AND b.exportedToYt = TRUE ORDER BY b.exportedToYtAt")
    List<BillingTransactionIdProjection> billingTransactionsForPayoutReport(Long partnerId, Instant exportedToYtAtBegin,
                                                                            Instant exportedToYtAtEnd);

    @Query("SELECT b FROM BillingTransaction b WHERE b.partnerId = ?1 AND b.accountingActAt >= ?2 " +
            "AND b.accountingActAt <= ?3 AND b.actCommitted = TRUE ORDER BY b.accountingActAt")
    List<BillingTransactionIdProjection> billingTransactionsForOrdersReport(Long partnerId, Instant accountingActBegin,
                                                                            Instant accountingActEnd);

    @Query(value = "select extract(epoch from max(" +
            "   least(cast(?1 as timestamp) - t.payout_at, cast(?2 as timestamp) - t.accounting_act_at)" +
            ")) as max_delay, count(*) " +
            "from billing_transactions t where t.exported_to_yt = true and t.act_committed = false " +
            "and t.payout_at < ?1 and t.accounting_act_at < ?2", nativeQuery = true)
    Tuple findMaxDelaySecondsOfTxReadyForActCommitRaw(Instant maxPayoutAt, Instant maxActAt);

    default ProcessingTasksInfo findMaxDelaySecondsOfTxReadyForActCommit(Instant maxPayoutAt, Instant maxActAt) {
        Tuple tuple = findMaxDelaySecondsOfTxReadyForActCommitRaw(maxPayoutAt, maxActAt);
        Number seconds = tuple.get(0, Number.class);
        // this param isn't a real timestamp but we pass like that for unification purposes
        Instant ts = seconds != null ? Instant.ofEpochSecond(seconds.longValue()) : null;
        Number count = tuple.get(1, Number.class);
        return new ProcessingTasksInfo(ts, count.longValue());
    }

    static ProcessingTasksInfo convertProcessingTasksInfo(Tuple tuple) {
        Object rawTs = tuple.get(0);
        Instant ts = rawTs instanceof Timestamp ? tuple.get(0, Timestamp.class).toInstant() : (Instant) rawTs;
        Number count = tuple.get(1, Number.class);
        return new ProcessingTasksInfo(ts, count.longValue());
    }
}
