package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.workflow.EWorkflowState;

@Component
@Slf4j
@RequiredArgsConstructor
public class CustomAeroflotInvoiceRepositoryImpl implements CustomAeroflotInvoiceRepository {

    private final EntityManager em;

    @Override
    @SuppressWarnings("unchecked")
    public List<UUID> getInvoiceIdsAwaitingRefreshInStateWithExclusions(Instant nextCheckStatusAt,
                                                                        EAeroflotInvoiceState invoiceState,
                                                                        EWorkflowState workflowState,
                                                                        Collection<UUID> exclusions,
                                                                        Pageable pageable) {
        Query query = em.createQuery(jpaQl("i.id", exclusions) + " order by i.nextCheckStatusAt");
        query.setParameter("nextCheckStatusAt", nextCheckStatusAt);
        query.setParameter("invoiceState", invoiceState);
        query.setParameter("workflowState", workflowState);
        if (!exclusions.isEmpty()) {
            query.setParameter("exclusions", exclusions);
        }
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize());
        return query.getResultList();
    }

    @Override
    public long countInvoicesAwaitingRefreshInStateWithExclusions(Instant nextCheckStatusAt,
                                                                  EAeroflotInvoiceState invoiceState,
                                                                  EWorkflowState workflowState,
                                                                  Collection<UUID> exclusions) {
        Query query = em.createQuery(jpaQl("COUNT(i.id)", exclusions));
        query.setParameter("nextCheckStatusAt", nextCheckStatusAt);
        query.setParameter("invoiceState", invoiceState);
        query.setParameter("workflowState", workflowState);
        if (!exclusions.isEmpty()) {
            query.setParameter("exclusions", exclusions);
        }
        return (Long) query.getResultList().get(0);
    }

    private String jpaQl(String selectPart, Collection<UUID> exclusions) {
        return "SELECT " + selectPart + " FROM ru.yandex.travel.orders.entities.AeroflotInvoice i " +
                "inner join i.workflow w " +
                "where i.nextCheckStatusAt is not null and i.nextCheckStatusAt <= :nextCheckStatusAt " +
                "AND w.state = :workflowState AND i.backgroundJobActive = true " +
                (exclusions.isEmpty() ? "" : "AND i.id NOT IN (:exclusions) ") +
                "AND i.state = :invoiceState";
    }
}
