package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.workflow.EWorkflowState;

@Component
@Slf4j
@RequiredArgsConstructor
public class CustomNotificationRepositoryImpl implements CustomNotificationRepository {
    private final EntityManager em;

    @Override
    public List<UUID> getNotificationsPreparingTill(Instant now, ENotificationState state, EWorkflowState wfState,
                                                    boolean expired, Collection<UUID> excludeIds, Pageable pageable) {
        Query query = em.createQuery(jpaQl("n.id", excludeIds));
        query.setParameter("expired", expired);
        query.setParameter("state", state);
        query.setParameter("workflowState", wfState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize());
        return query.getResultList();
    }

    @Override
    public long countNotificationsPreparingTill(Instant now, ENotificationState state, EWorkflowState wfState,
                                                boolean expired, Collection<UUID> excludeIds) {
        Query query = em.createQuery(jpaQl("count(n.id)", excludeIds));
        query.setParameter("expired", expired);
        query.setParameter("state", state);
        query.setParameter("workflowState", wfState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        return (Long) query.getResultList().get(0);
    }

    private String jpaQl(String selectPart, Collection<UUID> excludeIds) {
        return "SELECT " + selectPart + " FROM ru.yandex.travel.orders.entities.notifications.Notification n " +
                "INNER JOIN n.workflow w " +
                "WHERE n.state = :state " +
                exclusions(excludeIds) +
                "AND n.preparingAttachmentsTill is not null AND n.preparingAttachmentsTill <= :now " +
                "AND n.expired = :expired " +
                "AND w.state = :workflowState";
    }

    private String exclusions(Collection<UUID> exclusions) {
        if (!exclusions.isEmpty()) {
            return "AND n.id NOT IN (:exclusions) ";
        } else {
            return "";
        }
    }
}
