package ru.yandex.travel.orders.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AdminListOrdersParams;
import ru.yandex.travel.orders.entities.Order;

@Component
@Slf4j
@AllArgsConstructor
public class CustomOrderRepositoryImpl implements CustomOrderRepository {

    private final EntityManager entityManager;

    @SuppressWarnings("unchecked")
    public List<Order> findOrdersOwnedByUser(String passportId,
                                             Set<EDisplayOrderType> displayTypes,
                                             Set<EDisplayOrderState> displayStates,
                                             Pageable pageable) {
        String orderIdJpaQl = "SELECT au.id.orderId FROM ru.yandex.travel.orders.entities.AuthorizedUser au" +
                " where au.id.passportId = :passportId and au.role = 'OWNER'";
        Query orderIdQuery = entityManager.createQuery(orderIdJpaQl);
        orderIdQuery.setParameter("passportId", passportId);
        List<UUID> orderIds = orderIdQuery.getResultList();

        if (orderIds.size() == 0) {
            return List.of();
        }

        String aggregateStatesJpaQl = "SELECT oas.id FROM ru.yandex.travel.orders.entities.OrderAggregateState oas" +
                "  WHERE oas.id IN :orderIds AND oas.orderDisplayState IN :displayStates";
        Query displayStateFilterQuery = entityManager.createQuery(aggregateStatesJpaQl);
        displayStateFilterQuery.setParameter("orderIds", orderIds);
        displayStateFilterQuery.setParameter("displayStates", displayStates);
        List<UUID> idsFilteredByState = displayStateFilterQuery.getResultList();

        if (idsFilteredByState.size() == 0) {
            return List.of();
        }

        String jpql = "SELECT o FROM ru.yandex.travel.orders.entities.Order o" +
                "  WHERE id in (:orderIds) AND o.displayType in (:displayTypes)" +
                "  AND (o.removed IS NULL OR o.removed = FALSE)" +
                "  ORDER BY (CASE WHEN o.servicedAt < :day THEN o.servicedAt ELSE :day END) desc," +
                " o.servicedAt asc, str(o.id)";
        Query query = entityManager.createQuery(jpql);
        query.setParameter("orderIds", idsFilteredByState);
        query.setParameter("displayTypes", displayTypes);
        query.setParameter("day", LocalDate.now().atStartOfDay());
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize() + 1);
        return query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public List<Order> selectNotRemovedOrders(Set<UUID> orderIds) {
        String jpql = "SELECT o FROM ru.yandex.travel.orders.entities.Order o" +
                "  WHERE id in (:orderIds)" +
                "  AND (o.removed IS NULL OR o.removed = FALSE)" +
                "  ORDER BY (CASE WHEN o.servicedAt < :day THEN o.servicedAt ELSE :day END) desc," +
                " o.servicedAt asc";
        Query query = entityManager.createQuery(jpql);
        query.setParameter("orderIds", orderIds);
        query.setParameter("day", LocalDate.now().atStartOfDay());
        return query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public List<Order> findOrdersOwnedByUserWithoutExcluded(String passportId,
                                                            Set<UUID> excludedOrderIds,
                                                            Set<EDisplayOrderType> displayTypes,
                                                            Set<EDisplayOrderState> displayStates,
                                                            Pageable pageable) {
        String orderIdJpaQl = "SELECT au.id.orderId FROM ru.yandex.travel.orders.entities.AuthorizedUser au" +
                " where au.id.passportId = :passportId and au.role = 'OWNER'";
        Query orderIdQuery = entityManager.createQuery(orderIdJpaQl);
        orderIdQuery.setParameter("passportId", passportId);
        Set<UUID> orderIds = new HashSet<UUID>(orderIdQuery.getResultList());
        orderIds.removeAll(excludedOrderIds);

        if (orderIds.size() == 0) {
            return List.of();
        }

        String aggregateStatesJpaQl = "SELECT oas.id FROM ru.yandex.travel.orders.entities.OrderAggregateState oas" +
                "  WHERE oas.id IN :orderIds AND oas.orderDisplayState IN :displayStates";
        Query displayStateFilterQuery = entityManager.createQuery(aggregateStatesJpaQl);
        displayStateFilterQuery.setParameter("orderIds", orderIds);
        displayStateFilterQuery.setParameter("displayStates", displayStates);
        List<UUID> idsFilteredByState = displayStateFilterQuery.getResultList();

        if (idsFilteredByState.size() == 0) {
            return List.of();
        }

        String jpql = "SELECT o FROM ru.yandex.travel.orders.entities.Order o" +
                "  WHERE id in (:orderIds) AND o.displayType in (:displayTypes)" +
                "  AND (o.removed IS NULL OR o.removed = FALSE)" +
                "  ORDER BY (CASE WHEN o.servicedAt < :day THEN o.servicedAt ELSE :day END) desc," +
                " o.servicedAt asc";
        Query query = entityManager.createQuery(jpql);
        query.setParameter("orderIds", idsFilteredByState);
        query.setParameter("displayTypes", displayTypes);
        query.setParameter("day", LocalDate.now().atStartOfDay());
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize() + 1);
        return query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public List<Order> findAllOrdersOwnedByUser(String passportId) {
        String orderIdJpaQl = "SELECT au.id.orderId FROM ru.yandex.travel.orders.entities.AuthorizedUser au" +
                " where au.id.passportId = :passportId and au.role = 'OWNER'";
        Query orderIdQuery = entityManager.createQuery(orderIdJpaQl);
        orderIdQuery.setParameter("passportId", passportId);
        List<UUID> orderIds = orderIdQuery.getResultList();

        if (orderIds.size() == 0) {
            return List.of();
        }

        String jpaQl = "SELECT o FROM ru.yandex.travel.orders.entities.Order o WHERE o.id IN (:orderIds)";
        Query query = entityManager.createQuery(jpaQl);
        query.setParameter("orderIds", orderIds);
        return query.getResultList();
    }

    @Override
    public List<Order> findOrdersWithAdminFilters(AdminListOrdersParams params) {
        if (params.getOrderIdList() == null || params.getOrderIdList().size() == 0) {
            return new ArrayList<>();
        }
        CriteriaBuilder cb = entityManager.getCriteriaBuilder();
        CriteriaQuery<Order> cbQuery = cb.createQuery(Order.class);
        Root<Order> orderRoot = cbQuery.from(Order.class);
        cbQuery.select(orderRoot);
        Predicate orderIdInList = orderRoot.get("id").in(params.getOrderIdList());
        Predicate orderIsNotRemoved = cb.isFalse(orderRoot.get("removed"));
        cbQuery.where(cb.and(orderIsNotRemoved, orderIdInList));
        cbQuery.orderBy(params.prepareSorters(orderRoot));

        TypedQuery<Order> typedQuery = entityManager.createQuery(cbQuery);
        return typedQuery.getResultList();
    }
}
