package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.AllArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;

@AllArgsConstructor
@Component
public class CustomPaymentScheduleItemRepositoryImpl implements CustomPaymentScheduleItemRepository {
    private final EntityManager entityManager;

    @Override
    public List<PaymentScheduleItem> findItemsForEmailReminderWithNoNotifications(Instant reminderInstant,
                                                                                  Set<UUID> activeIds, Pageable pageable) {
        String jpql = "select psi from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.emailReminderAt < :reminderInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.reminderEmailSent = FALSE AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("reminderInstant", reminderInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        if (!pageable.isUnpaged()) {
            query = query
                    .setFirstResult(pageable.getPageNumber() * pageable.getPageSize())
                    .setMaxResults(pageable.getPageSize());
        }
        return query.getResultList();
    }

    @Override
    public Long countItemsForEmailReminderWithNoNotifications(Instant reminderInstant, Set<UUID> activeIds) {
        String jpql = "select count(distinct psi.id) from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.emailReminderAt < :reminderInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.reminderEmailSent = FALSE AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("reminderInstant", reminderInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        return (Long) query.getResultList().get(0);
    }

    @Override
    public List<PaymentScheduleItem> findItemsForTicketReminderWithNoNotifications(Instant reminderInstant,
                                                                                  Set<UUID> activeIds, Pageable pageable) {
        String jpql = "select psi from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.ticketReminderAt < :reminderInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.reminderTicket is null AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("reminderInstant", reminderInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        if (!pageable.isUnpaged()) {
            query = query
                    .setFirstResult(pageable.getPageNumber() * pageable.getPageSize())
                    .setMaxResults(pageable.getPageSize());
        }
        return query.getResultList();
    }

    @Override
    public Long countItemsForTicketReminderWithNoNotifications(Instant reminderInstant, Set<UUID> activeIds) {
        String jpql = "select count(distinct psi.id) from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.ticketReminderAt < :reminderInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.reminderTicket is null AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("reminderInstant", reminderInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        return (Long) query.getResultList().get(0);
    }

    @Override
    public List<PaymentScheduleItem> findItemsForAutoPayment(Instant autoPaymentInstant, Set<UUID> activeIds,
                                                             Pageable pageable) {
        String jpql = "select psi from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.autoPaymentAt < :autoPaymentInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.autoPaymentStarted = FALSE AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("autoPaymentInstant", autoPaymentInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        if (!pageable.isUnpaged()) {
            query = query
                    .setFirstResult(pageable.getPageNumber() * pageable.getPageSize())
                    .setMaxResults(pageable.getPageSize());
        }
        return query.getResultList();
    }

    @Override
    public Long countItemsForAutoPayment(Instant autoPaymentInstant, Set<UUID> activeIds) {
        String jpql = "select count(distinct psi.id) from ru.yandex.travel.orders.entities.PaymentScheduleItem psi " +
                "inner join psi.schedule as ps " +
                "inner join psi.pendingInvoice as pi " +
                "where pi.state = :invoiceState AND " +
                "psi.autoPaymentAt < :autoPaymentInstant AND " +
                "ps.state = :scheduleState AND " +
                "psi.autoPaymentStarted = FALSE AND " +
                "psi.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("invoiceState", EPaymentState.PS_INVOICE_PENDING)
                .setParameter("autoPaymentInstant", autoPaymentInstant)
                .setParameter("scheduleState", EPaymentState.PS_PARTIALLY_PAID)
                .setParameter("excluded", activeIds);
        return (Long) query.getResultList().get(0);
    }

}
