package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.AllArgsConstructor;
import org.springframework.data.domain.Pageable;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;

@AllArgsConstructor
@Component
public class CustomPaymentScheduleRepositoryImpl implements CustomPaymentScheduleRepository {

    private final NamedParameterJdbcTemplate namedParameterJdbcTemplate;
    private final EntityManager entityManager;

    public List<PaymentSchedule> findNonExpiredSchedulesInGivenStatesExpiringBeforeInstant(Set<EPaymentState> scheduleStates,
                                                                                           Set<EPaymentState> itemStates,
                                                                                           Instant expirationInstant,
                                                                                           Set<UUID> excluded,
                                                                                           Pageable pageable) {
        String jpql = "select ps from ru.yandex.travel.orders.entities.PaymentSchedule ps " +
                "inner join ps.items as i " +
                "inner join i.pendingInvoice as pi " +
                "where pi.state in (:itemStates) AND " +
                "i.paymentEndsAt < (:expirationInstant) AND " +
                "ps.state in (:scheduleStates) AND " +
                "ps.expired = FALSE AND " +
                "ps.id not in (:excluded)";

        Query query = entityManager.createQuery(jpql)
                .setParameter("itemStates", itemStates)
                .setParameter("expirationInstant", expirationInstant)
                .setParameter("scheduleStates", scheduleStates)
                .setParameter("excluded", excluded);
        if (!pageable.isUnpaged()) {
            query = query
                    .setFirstResult(pageable.getPageNumber() * pageable.getPageSize())
                    .setMaxResults(pageable.getPageSize());
        }
        return query.getResultList();
    }

    public Long countNonExpiredSchedulesInGivenStatesExpiringBeforeInstant(Set<EPaymentState> scheduleStates,
                                                                           Set<EPaymentState> itemStates,
                                                                           Instant expirationInstant,
                                                                           Set<UUID> activeIds) {
        String jpql = "SELECT COUNT(distinct ps.id) from ru.yandex.travel.orders.entities.PaymentSchedule ps " +
                "inner join ps.items as i " +
                "inner join i.pendingInvoice as pi " +
                "where pi.state in (:itemStates) AND " +
                "i.paymentEndsAt < (:expirationInstant) AND " +
                "ps.state in (:scheduleStates) AND " +
                "ps.expired = FALSE AND " +
                "ps.id not in (:activeIds) ";

        Query query = entityManager.createQuery(jpql)
                .setParameter("itemStates", itemStates)
                .setParameter("expirationInstant", expirationInstant)
                .setParameter("scheduleStates", scheduleStates)
                .setParameter("activeIds", activeIds);
        return (Long) query.getResultList().get(0);
    }
}
