package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.ETrainInsuranceRefundState;
import ru.yandex.travel.workflow.EWorkflowState;

@Component
@Slf4j
@RequiredArgsConstructor
public class CustomTrainInsuranceRefundRepositoryImpl implements CustomTrainInsuranceRefundRepository {
    private final EntityManager em;

    @Override
    @SuppressWarnings("unchecked")
    public List<UUID> getInsuranceRefundsForCheck(Instant now,
                                                  ETrainInsuranceRefundState state,
                                                  EWorkflowState workflowState,
                                                  Collection<UUID> excludeIds,
                                                  Pageable pageable) {
        Query query = em.createQuery(refundsForCheckQuery("i.id", excludeIds));
        query.setParameter("state", state);
        query.setParameter("workflowState", workflowState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize());
        return query.getResultList();
    }

    @Override
    public long countInsuranceRefundsForCheck(Instant now, ETrainInsuranceRefundState state,
                                              EWorkflowState workflowState, Collection<UUID> excludeIds) {
        Query query = em.createQuery(refundsForCheckQuery("count(i.id)", excludeIds));
        query.setParameter("state", state);
        query.setParameter("workflowState", workflowState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        return (Long) query.getResultList().get(0);
    }

    private String refundsForCheckQuery(String selectPart, Collection<UUID> excludeIds) {
        return "SELECT " + selectPart + " FROM ru.yandex.travel.orders.entities.TrainInsuranceRefund i " +
                "INNER JOIN i.workflow w WHERE i.state = :state " +
                "AND i.backgroundJobActive = true " +
                exclusions(excludeIds) +
                "AND i.nextCheckAt is not null AND i.nextCheckAt <= :now " +
                "AND w.state = :workflowState";
    }

    private String exclusions(Collection<UUID> exclusions) {
        if (!exclusions.isEmpty()) {
            return "AND i.id NOT IN (:exclusions) ";
        } else {
            return "";
        }
    }
}
