package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.workflow.EWorkflowState;

@Component
@Slf4j
@RequiredArgsConstructor
public class CustomTrainOrderItemRepositoryImpl implements CustomTrainOrderItemRepository {
    private final EntityManager em;

    @Override
    @SuppressWarnings("unchecked")
    public List<UUID> getOrderItemsForCheckReserved(Instant now,
                                                    Collection<EOrderItemState> states,
                                                    EWorkflowState workflowState,
                                                    Collection<UUID> excludeIds,
                                                    Pageable pageable) {
        Query query = em.createQuery(jpaQl("i.id", excludeIds));
        query.setParameter("states", states);
        query.setParameter("workflowState", workflowState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize());
        return query.getResultList();
    }

    @Override
    public long countOrderItemsForCheckReserved(Instant now, Collection<EOrderItemState> states,
                                                EWorkflowState workflowState, Collection<UUID> excludeIds) {
        Query query = em.createQuery(jpaQl("count(i.id)", excludeIds));
        query.setParameter("states", states);
        query.setParameter("workflowState", workflowState);
        query.setParameter("now", now);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        return (Long) query.getResultList().get(0);
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<UUID> getExpiredOrderItems(Instant expiredAt, Collection<EOrderItemState> itemStates,
                                           EWorkflowState workflowState, Collection<UUID> excludeIds,
                                           Pageable pageable) {
        Query query = em.createQuery(queryOrderItemsJoinOrders("i.id", excludeIds));
        query.setParameter("itemStates", itemStates);
        query.setParameter("expiredAt", expiredAt);
        query.setParameter("workflowState", workflowState);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        query.setFirstResult(Long.valueOf(pageable.getOffset()).intValue());
        query.setMaxResults(pageable.getPageSize());
        return query.getResultList();
    }

    @Override
    public long countExpiredOrderItems(Instant expiredAt, Collection<EOrderItemState> itemStates,
                                       EWorkflowState workflowState, Collection<UUID> excludeIds) {
        Query query = em.createQuery(queryOrderItemsJoinOrders("count(i.id)", excludeIds));
        query.setParameter("itemStates", itemStates);
        query.setParameter("expiredAt", expiredAt);
        query.setParameter("workflowState", workflowState);
        if (!excludeIds.isEmpty()) {
            query.setParameter("exclusions", excludeIds);
        }
        return (Long) query.getResultList().get(0);
    }

    private String jpaQl(String selectPart, Collection<UUID> excludeIds) {
        return "SELECT " + selectPart + " FROM ru.yandex.travel.orders.entities.TrainOrderItem i " +
            "INNER JOIN i.workflow w " +
            "WHERE i.state IN (:states) " +
            "AND i.backgroundJobActive = true " +
            exclusions(excludeIds) +
            "AND i.nextCheckConfirmedAt is not null AND i.nextCheckConfirmedAt <= :now " +
            "AND w.state = :workflowState";
    }

    private String exclusions(Collection<UUID> exclusions) {
        if (!exclusions.isEmpty()) {
            return "AND i.id NOT IN (:exclusions) ";
        } else {
            return "";
        }
    }

    private String queryOrderItemsJoinOrders(String selectPart, Collection<UUID> excludeIds) {
        return "SELECT " + selectPart + " FROM ru.yandex.travel.orders.entities.TrainOrderItem i " +
            "INNER JOIN i.workflow w " +
            "WHERE i.isExpired = false " +
            "AND i.state IN (:itemStates) " +
            "AND w.state = :workflowState " +
            "AND i.expiresAt < :expiredAt " +
            exclusions(excludeIds);
    }
}
