package ru.yandex.travel.orders.repository.finances;

import java.time.Instant;
import java.time.LocalDate;
import java.util.List;

import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;

import ru.yandex.travel.orders.entities.finances.BankOrder;

public interface BankOrderRepository extends JpaRepository<BankOrder, Long> {

    @Query(
            value = "SELECT * FROM bank_orders WHERE payment_batch_id = ?1 AND bank_order_id = ?2",
            nativeQuery = true
    )
    BankOrder findByPaymentBatchIdAndBankOrderId(String paymentBatchId, String bankOrderId);

    /**
     * similar to {@link #findByPaymentBatchIdAndBankOrderId(String, String)}, but also checks bank orders with null
     * <code>bank order id</code>.
     */
    @Query(
            value = "SELECT * FROM bank_orders WHERE payment_batch_id = ?1 AND" +
                    " (bank_order_id = ?2 or bank_order_id is null)",
            nativeQuery = true
    )
    BankOrder lookForExistingBankOrders(String paymentBatchId, String bankOrderId);

    @Query(
            value = "SELECT MIN(eventtime) FROM bank_orders WHERE oebs_payment_status NOT IN ('reconciled', 'returned')",
            nativeQuery = true
    )
    LocalDate earliestNotFullyProcessedPaymentDate();

    @Query(
            value = "SELECT MAX(eventtime) FROM bank_orders WHERE oebs_payment_status IN ('reconciled', 'returned')",
            nativeQuery = true
    )
    LocalDate latestFullyProcessedPaymentDate();

    @Query(
            value = "SELECT SUM(bo.sum) FROM bank_orders bo " +
                    "WHERE bo.payment_batch_id IN (" +
                        "SELECT bod.payment_batch_id FROM bank_order_details bod " +
                        "WHERE bod.contract_id = ?1" +
                        ") " +
                    "AND bo.oebs_payment_status = 'reconciled' " +
                    "AND ?2 <= bo.eventtime AND bo.eventtime <= ?3",
            nativeQuery = true
    )
    Double getReconciledAmountForContractIdInPeriod(Long contractId, LocalDate periodFrom, LocalDate periodTo);


    /**
     * @implNote not filtering by service id, cause ATM, there are only hotel bank orders. Fetching only payments with
     * details. We can use only 'fetched', other records are probably a sign of errors which should be fixed separately.
     */
    @Query("FROM BankOrder bo " +
            "JOIN bo.bankOrderPayment bop " +
            "WHERE bo.updatedAt BETWEEN ?1 AND ?2 " +
            " AND bop.status = 'fetched'"
    )
    List<BankOrder> getRecenttlyUpdatedBankOrders(
            Instant updatedFrom,
            Instant updatedTill,
            Pageable pageable
    );
}
