package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.util.UUID;

import javax.money.CurrencyUnit;

import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.Account;
import ru.yandex.travel.orders.entities.AccountTransaction;
import ru.yandex.travel.orders.entities.FxContext;
import ru.yandex.travel.orders.entities.MoneyTransferConfig;
import ru.yandex.travel.orders.entities.WellKnownAccount;
import ru.yandex.travel.orders.repository.AccountRepository;
import ru.yandex.travel.orders.repository.AccountTransactionRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Component
public class AccountService {

    private AccountRepository accountRepository;
    private AccountTransactionRepository accountTransactionRepository;

    @Autowired
    public AccountService(AccountRepository accountRepository,
                          AccountTransactionRepository accountTransactionRepository) {
        this.accountRepository = accountRepository;
        this.accountTransactionRepository = accountTransactionRepository;
    }

    @TransactionMandatory
    public void transferMoney(UUID idFrom, UUID idTo, BigDecimal amount, CurrencyUnit currency) {
        MoneyTransferConfig config = MoneyTransferConfig.create()
                .setBaseCurrency(currency)
                .addTransfer(idFrom, idTo, amount);
        transferMoney(config);
    }

    @TransactionMandatory
    public void transferMoney(MoneyTransferConfig config) {
        ProtoCurrencyUnit protoCur = ProtoCurrencyUnit.fromCurrencyCode(config.getBaseCurrency().getCurrencyCode());
        AccountTransaction.Builder builder = AccountTransaction.builder(new FxContext(protoCur, config.getFxRate()));
        for (Tuple2<UUID, UUID> idPair : config.getTransfers().keySet()) {
            Account fromAcc = accountRepository.getOne(idPair.get1());
            Account toAcc = accountRepository.getOne(idPair.get2());
            builder.addTransfer(fromAcc, toAcc, config.getTransfers().get(idPair));
        }
        AccountTransaction transaction = builder.build();
        accountTransactionRepository.saveAndFlush(transaction);
    }

    @TransactionMandatory
    public void createWellKnownAccs() {
        for (WellKnownAccount account : WellKnownAccount.values()) {
            if (!accountRepository.existsById(account.getUuid())) {
                Account repAccount = Account.createWellKnownAccount(account);
                accountRepository.save(repAccount);
            }
        }
        accountRepository.flush();
    }

    @TransactionMandatory
    public Account createAccount(ProtoCurrencyUnit protoCurrencyUnit) {
        Account account = Account.createAccount(protoCurrencyUnit);
        return accountRepository.save(account);
    }

    @TransactionMandatory
    public Money getAccountBalance(UUID accountId) {
        Account account = accountRepository.getOne(accountId);
        BigDecimal accountBalance = accountRepository.getCurrentBalanceForAccount(accountId);
        if (accountBalance == null) {
            accountBalance = BigDecimal.ZERO;
        }
        return Money.of(accountBalance, account.getCurrency());
    }

}
