package ru.yandex.travel.orders.services;

import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EAdminAction;
import ru.yandex.travel.orders.commons.proto.EAdminRole;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AdminUser;
import ru.yandex.travel.orders.repository.AdminUserRepository;
import ru.yandex.travel.orders.services.admin.AdminUserCapabilitiesManager;

@Slf4j
@Service
@EnableConfigurationProperties(AuthorizationAdminConfigurationProperties.class)
public class AuthorizationAdminService {

    private final AuthorizationAdminConfigurationProperties config;
    private final UserInfoService userInfoService;
    private final AdminUserRepository adminUserRepository;

    @Autowired
    public AuthorizationAdminService(AuthorizationAdminConfigurationProperties config,
                                     UserInfoService userInfoService,
                                     AdminUserRepository adminUserRepository) {
        this.config = config;
        this.userInfoService = userInfoService;
        this.adminUserRepository = adminUserRepository;
    }

    public Set<EAdminRole> getRolesForUser(String login) {
        return adminUserRepository.findById(login).map(user ->
                user.isSuperuser() ? Set.of(EAdminRole.AR_SUPERUSER) : user.getRoles()
        ).orElseGet(() -> {
            log.warn("User {} is not allowed to access any administration methods", login);
            return Set.of(EAdminRole.AR_UNKNOWN);
        });
    }

    public AdminUserCapabilitiesManager getAdminUserCapabilities(String login) {
        Set<EAdminRole> roles = getRolesForUser(login);

        return new AdminUserCapabilitiesManager(roles, config.isEnabled());
    }

    public boolean authorizeUserForAdminAction(UserCredentials userCredentials, EAdminAction action, EDisplayOrderType type) {
        if (!config.isEnabled()) {
            log.warn("Admin authorization is DISABLED!");
            return true;
        }
        Preconditions.checkNotNull(userCredentials, "Credentials are not enabled");
        String login = userInfoService.getLoginByCredentials(userCredentials);
        log.info("User {} is trying to access administration action {}", login, action);
        AdminUserCapabilitiesManager adminUserCapabilities = getAdminUserCapabilities(login);

        return adminUserCapabilities.canDoAction(action, type);
    }

    public void addRole(String login, EAdminRole role) {
        adminUserRepository.findById(login).ifPresentOrElse(user -> {
            Set<EAdminRole> userRoles = user.getRoles();
            if (userRoles.contains(role)) {
                throw Error.with(EErrorCode.EC_ALREADY_EXISTS, "User " + login + " already has a role " + userRoles).toEx();
            } else {
                log.info("Updating admin user {} with role {}", login, role);
                user.addRole(role);
                user.setSuperuser(false); // Setting superuser to false because it should be temporary
                adminUserRepository.save(user);
            }
        }, () -> {
            log.info("Adding new admin user {} with role {}", login, role);
            AdminUser newUser = new AdminUser();
            newUser.setLogin(login);
            newUser.addRole(role);
            newUser.setSuperuser(false);
            adminUserRepository.save(newUser);
        });
    }

    public void removeRole(String login, EAdminRole role) {
        if (adminUserRepository.existsById(login)) {
            log.info("Deleting role {} from user {}", role, login);
            var user = adminUserRepository.getOne(login);
            user.removeRole(role);
            if (user.getRoles().isEmpty()) {
                adminUserRepository.deleteById(login);
            }
        } else {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "Admin user " + login + " not found").toEx();
        }
    }

    public List<AdminUser> getAllRoles() {
        return adminUserRepository.findAll();
    }
}
