package ru.yandex.travel.orders.services;

import java.security.MessageDigest;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.crypto.HashingUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;

@Service
@RequiredArgsConstructor
public class AuthorizationService {
    private final AuthorizedUserRepository authRepository;

    public boolean checkAuthorizationFor(UUID orderId, String sessionKey, String passportId) {
        return authRepository.findByIdOrderId(orderId).stream().anyMatch(auth -> {
            if (auth.getPassportId() != null) {
                if (passportId == null) {
                    return false;
                } else {
                    return MessageDigest.isEqual(auth.getPassportId().getBytes(), passportId.getBytes());
                }
            } else {
                return sessionKey != null && MessageDigest.isEqual(auth.getSessionKey().getBytes(),
                        sessionKey.getBytes());
            }
        });
    }

    public boolean checkAndAuthorizeUserForOrder(Order order, UserCredentials credentials, String secret) {
        boolean authorized = false;
        if (checkAuthorizationFor(order.getId(), credentials.getSessionKey(), credentials.getPassportId())) {
            authorized = true;
        } else {
            if (secret.equalsIgnoreCase(order.getEmail())) {
                authorized = true;
            } else {
                if (HashingUtils.hashPhone(secret).equals(HashingUtils.hashPhone(order.getPhone()))) {
                    authorized = true;
                }
            }
            if (authorized) {
                authorizeForOrder(order.getId(), credentials, AuthorizedUser.OrderUserRole.VIEWER);
            }
        }
        return authorized;
    }

    public void authorizeForOrder(UUID orderId, UserCredentials credentials, AuthorizedUser.OrderUserRole role) {
        var authorization = AuthorizedUser.create(orderId, credentials, role);
        authRepository.saveAndFlush(authorization);
    }

    public AuthorizedUser getOrderOwner(UUID orderId) {
        return authRepository.findByIdOrderIdAndRole(orderId, AuthorizedUser.OrderUserRole.OWNER).stream().findAny().orElse(null);
    }

    public AuthorizedUser getRequiredOrderOwner(UUID orderId) {
        return authRepository.findByIdOrderIdAndRole(orderId, AuthorizedUser.OrderUserRole.OWNER).stream().findAny()
                .orElseThrow(
                        () -> new IllegalStateException(String.format("Couldn't find owner for order: %s", orderId))
                );
    }

    public List<AuthorizedUser> getOrderAuthorizedUsers(UUID orderId) {
        return authRepository.findByIdOrderId(orderId);
    }

}
