package ru.yandex.travel.orders.services;

import java.io.IOException;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.entities.BlackboxResponse;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
@EnableConfigurationProperties(BlackboxProperties.class)
public class BlackboxService implements UserInfoService {
    private static final Set<Integer> SUCCESSFUL_RESPONSE_CODES = ImmutableSet.of(200, 201);
    public static final String SERVICE_TICKET_HEADER_NAME = "X-Ya-Service-Ticket";

    private final BlackboxProperties config;
    private final TvmWrapper tvm;
    private final AsyncHttpClientWrapper httpClient;

    private final ObjectMapper objectMapper;
    private final LoadingCache<UserCredentials, String> cache;

    public BlackboxService(BlackboxProperties blackboxProperties,
                           TvmWrapper tvm,
                           AsyncHttpClientWrapper httpClient) {
        this.config = blackboxProperties;
        this.tvm = tvm;
        this.httpClient = httpClient;

        this.objectMapper = new ObjectMapper()
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .registerModule(new JavaTimeModule());

        CacheLoader<UserCredentials, String> loader = new CacheLoader<UserCredentials, String>() {
            @Override
            public String load(UserCredentials key) throws Exception {
                String passportId = key.getPassportId();
                String userIp = key.getUserIp();
                return getLoginFromBlackbox(passportId, userIp);
            }
        };
        cache = CacheBuilder.newBuilder()
                .expireAfterWrite(24, TimeUnit.HOURS)
                .maximumSize(100)
                .build(loader);

        tvm.validateAlias(config.getTvmAlias());
    }

    @Override
    public String getLoginByCredentials(UserCredentials userCredentials) {
        try {
            return cache.get(userCredentials);
        } catch (ExecutionException e) {
            throw new RuntimeException("Unable to get login for passportId: " + userCredentials.getPassportId(), e);
        }
    }

    private String getLoginFromBlackbox(String passportUid, String userIp) {
        String serviceTicket = tvm.getServiceTicket(config.getTvmAlias());
        RequestBuilder builder = new RequestBuilder("GET")
                .setUrl(config.getUrl())
                .setHeader(SERVICE_TICKET_HEADER_NAME, serviceTicket)
                .addQueryParam("method", "userinfo")
                .addQueryParam("uid", passportUid)
                .addQueryParam("userip", userIp)
                .addQueryParam("format", "json");
        CompletableFuture<Response> futureResponse = httpClient.executeRequest(builder);

        try {
            BlackboxResponse response = futureResponse
                    .thenApply(this::parseResponse)
                    .get();
            if (response.getUsers() != null && response.getUsers().size() == 1) {
                return response.getUsers().get(0).getLogin();
            } else {
                throw new RuntimeException("Login not found in blackbox for passportId: " + passportUid);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Interrupted while acquiring login for passportId: " + passportUid, e);
        } catch (ExecutionException e) {
            throw new RuntimeException("An error occurred while acquiring login for passportId: " + passportUid, e);
        }
    }

    private BlackboxResponse parseResponse(Response response) {
        int statusCode = response.getStatusCode();
        String body = response.getResponseBody();
        if (SUCCESSFUL_RESPONSE_CODES.contains(statusCode)) {
            try {
                return objectMapper.readValue(body, BlackboxResponse.class);
            } catch (IOException e) {
                log.error("Unable to parse response from blackbox", e);
                throw new RuntimeException("Unable to parse response from blackbox", e);
            }
        } else {
            log.info("Blackbox API exception: statusCode={}, message={}", statusCode, body);
            throw new RuntimeException("Got response from blackbox with status code: " + statusCode + ", body: " + body);
        }
    }
}
