package ru.yandex.travel.orders.services;

import java.util.UUID;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.entities.DeduplicationKey;

@Service
@Slf4j
public class DeduplicationService {
    private final EntityManager entityManager;
    private final TransactionTemplate transactionTemplate;

    public DeduplicationService(EntityManager entityManager, PlatformTransactionManager transactionManager) {
        this.entityManager = entityManager;
        DefaultTransactionDefinition txDefinition = new DefaultTransactionDefinition();
        txDefinition.setName("DeduplicationService");
        txDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        txDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_READ_COMMITTED);
        this.transactionTemplate = new TransactionTemplate(transactionManager, txDefinition);
    }

    public void registerAtMostOnceCall(UUID deduplicationKey) {
        try {
            transactionTemplate.execute(ignored -> {
                log.debug("Trying to acquire the process-at-most-once lock: deduplicationKey={}", deduplicationKey);
                DeduplicationKey key = new DeduplicationKey();
                key.setUuid(deduplicationKey);
                entityManager.persist(key);
                return null;
            });
        } catch (EntityExistsException | DataIntegrityViolationException e) {
            log.warn("An attempt to call a non-idempotent external service twice: deduplicationKey={}",
                    deduplicationKey);
            throw new IllegalStateException("The external service call has already been made once. " +
                    "We can't re-try this non-transactional operation. deduplicationKey=" + deduplicationKey, e);
        }
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void unregisterAtMostOnceCall(UUID deduplicationKey) {
        log.debug("Releasing the process-at-most-once lock: deduplicationKey={}", deduplicationKey);
        DeduplicationKey key = entityManager.find(DeduplicationKey.class, deduplicationKey);
        Preconditions.checkNotNull(key, "No such deduplicationKey: {}", deduplicationKey);
        entityManager.remove(key);
    }
}
