package ru.yandex.travel.orders.services;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.workflow.EWorkflowState;

public class EDisplayOrderStateMapper {
    private static final Set<ETrustInvoiceState> HOLD_MONEY_INVOICE_STATES =
            ImmutableSet.of(ETrustInvoiceState.IS_HOLD, ETrustInvoiceState.IS_CLEARING);

    private static final Map<EAeroflotOrderState, EDisplayOrderState> AEROFLOT_ORDER_TO_EDISPLAY_ORDER_STATE =
            ImmutableMap.<EAeroflotOrderState, EDisplayOrderState>builder()
                    .put(EAeroflotOrderState.OS_NEW, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_CHECK_AVAILABILITY, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED, EDisplayOrderState.OS_AWAITS_PAYMENT)
                    .put(EAeroflotOrderState.OS_WAIT_ORDER_CREATED, EDisplayOrderState.OS_AWAITS_PAYMENT)
                    .put(EAeroflotOrderState.OS_WAIT_ORDER_PAID, EDisplayOrderState.OS_AWAITS_PAYMENT)
                    .put(EAeroflotOrderState.OS_CONFIRMED, EDisplayOrderState.OS_FULFILLED)
                    .put(EAeroflotOrderState.OS_CANCELLED, EDisplayOrderState.OS_CANCELLED)
                    .put(EAeroflotOrderState.OS_AUTO_CONFIRMING_INVOICE, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_CONFIRMING_ORDER_ITEM, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_CANCELLING_ORDER_ITEM, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_CANCELLING_INVOICE, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_AUTO_RESTORING_INVOICE, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_CONFIRMING_ORDER, EDisplayOrderState.OS_IN_PROGRESS)
                    .put(EAeroflotOrderState.OS_EXTERNALLY_CANCELLED, EDisplayOrderState.OS_REFUNDED)
                    .build();

    static {
        ensureAeroflotOrderStatesMappingIntegrity();
    }

    private EDisplayOrderStateMapper() {
    }

    private static void ensureAeroflotOrderStatesMappingIntegrity() {
        for (EAeroflotOrderState state : EAeroflotOrderState.values()) {
            if (state == EAeroflotOrderState.UNRECOGNIZED || state == EAeroflotOrderState.OS_UNKNOWN) {
                continue;
            }
            Preconditions.checkArgument(AEROFLOT_ORDER_TO_EDISPLAY_ORDER_STATE.containsKey(state),
                    "No EDisplayOrderState mapping for the %s state", state);
        }
    }

    public static EDisplayOrderState fromOrder(Order order) {
        if (order instanceof HotelOrder) {
            return fromHotelOrder((HotelOrder) order);
        } else if (order instanceof AeroflotOrder) {
            return fromAeroflotOrder((AeroflotOrder) order);
        } else if (order instanceof TrainOrder) {
            return fromTrainOrder((TrainOrder) order);
        } else if (order instanceof GenericOrder) {
            return fromGenericOrder((GenericOrder) order);
        } else {
            throw new IllegalArgumentException(
                    String.format("Don't know how to map EDisplayOrderState for order of type %s", order.getClass())
            );
        }
    }

    private static EDisplayOrderState fromAeroflotOrder(AeroflotOrder order) {
        EAeroflotOrderState state = order.getEntityState();
        if (order.getWorkflow().getState() == EWorkflowState.WS_CRASHED) {
            // there is no extra info for crashed workflows,
            // simply treating all unhandled errors as handled ones
            state = EAeroflotOrderState.OS_CANCELLED;
        }
        return AEROFLOT_ORDER_TO_EDISPLAY_ORDER_STATE.get(state);
    }

    private static EDisplayOrderState fromHotelOrder(HotelOrder order) {
        return getHotelOrderState(
                order.isUserActionScheduled(), order.getState(), order.getWorkflow().getState(),
                order.getInvoices().stream()
                        .sorted(Comparator.comparing(Invoice::getCreatedAt))
                        .map(i -> ((TrustInvoice) i).getState()).collect(Collectors.toList())
        );
    }

    private static EDisplayOrderState fromTrainOrder(TrainOrder order) {
        ETrainOrderState orderState = order.getEntityState();
        EWorkflowState workflowState = order.getWorkflow().getState();
        boolean userActionScheduled = order.isUserActionScheduled();
        List<ETrustInvoiceState> invoiceStates =
                order.getInvoices().stream().map(i -> ((TrustInvoice) i).getState()).collect(Collectors.toList());
        switch (workflowState) {
            case WS_CRASHED:
                switch (orderState) {
                    case OS_WAITING_REFUND_AFTER_CANCELLATION:
                    case OS_WAITING_INSURANCE_REFUND:
                    case OS_WAITING_CONFIRMATION:
                    case OS_WAITING_PAYMENT:
                        return EDisplayOrderState.OS_IN_PROGRESS;
                    default:
                        return EDisplayOrderState.OS_CANCELLED;
                }
            default:
                if (userActionScheduled) {
                    return EDisplayOrderState.OS_IN_PROGRESS;
                } else {
                    switch (orderState) {
                        case OS_CANCELLED:
                            // if there is money on hold report order as "in progress".
                            // If there was no payment there will be no invoices, so report order as "cancelled"
                            // If the payment was taken but returned, the invoice will be cleared, so report order as
                            // "cancelled"
                            if (invoiceStates.stream().anyMatch(HOLD_MONEY_INVOICE_STATES::contains)) {
                                return EDisplayOrderState.OS_IN_PROGRESS;
                            } else {
                                return EDisplayOrderState.OS_CANCELLED;
                            }
                        case OS_WAITING_PAYMENT:
                            if (invoiceStates.isEmpty()) {
                                return EDisplayOrderState.OS_AWAITS_PAYMENT;
                            } else if (invoiceStates.stream().anyMatch(is -> is == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT)) {
                                return EDisplayOrderState.OS_AWAITS_PAYMENT;
                            } else {
                                return EDisplayOrderState.OS_IN_PROGRESS;
                            }
                        case OS_CONFIRMED:
                        case OS_MANUAL_PROCESSING:
                            return EDisplayOrderState.OS_FULFILLED;
                        case OS_REFUNDED:
                            return EDisplayOrderState.OS_REFUNDED;
                        default:
                            return EDisplayOrderState.OS_IN_PROGRESS;
                    }
                }
        }
    }

    private static EDisplayOrderState fromGenericOrder(GenericOrder order) {
        switch (order.getEntityState()) {
            case OS_NEW:
            case OS_WAITING_RESERVATION:
            case OS_WAITING_CONFIRMATION:
            case OS_WAITING_CANCELLATION:
            case OS_REFUNDING:
                return EDisplayOrderState.OS_IN_PROGRESS;
            case OS_RESERVED:
            case OS_WAITING_PAYMENT:
                return EDisplayOrderState.OS_AWAITS_PAYMENT;
            case OS_CONFIRMED:
                return EDisplayOrderState.OS_FULFILLED;
            case OS_REFUNDED:
                return EDisplayOrderState.OS_REFUNDED;
            default:
                return EDisplayOrderState.OS_CANCELLED;
        }
    }

    private static EDisplayOrderState getHotelOrderState(boolean actionScheduled, EHotelOrderState orderState,
                                                         EWorkflowState workflowState,
                                                         List<ETrustInvoiceState> invoiceStates) {
        switch (workflowState) {
            case WS_CRASHED:
                switch (orderState) {
                    case OS_WAITING_CONFIRMATION:
                    case OS_WAITING_INVOICE_REFUND:
                    case OS_WAITING_PAYMENT:
                    case OS_WAITING_SERVICE_REFUND:
                    case OS_WAITING_REFUND_AFTER_CANCELLATION:
                    case OS_WAITING_CANCELLATION:
                    case OS_MANUAL_PROCESSING:
                        return EDisplayOrderState.OS_IN_PROGRESS;
                    default:
                        return EDisplayOrderState.OS_CANCELLED;
                }
            default:
                if (actionScheduled) {
                    return EDisplayOrderState.OS_IN_PROGRESS;
                } else {
                    switch (orderState) {
                        case OS_CANCELLED:
                            // if there is money on hold report order as "in progress".
                            // If there was no payment there will be no invoices, so report order as "cancelled"
                            // If the payment was taken but returned, the invoice will be cleared, so report order as
                            // "cancelled"
                            if (invoiceStates.stream().anyMatch(HOLD_MONEY_INVOICE_STATES::contains)) {
                                return EDisplayOrderState.OS_IN_PROGRESS;
                            } else {
                                return EDisplayOrderState.OS_CANCELLED;
                            }
                        case OS_WAITING_PAYMENT:
                            if (invoiceStates.isEmpty()) {
                                return EDisplayOrderState.OS_AWAITS_PAYMENT;
                            } else if (invoiceStates.stream().anyMatch(is -> is == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT)) {
                                return EDisplayOrderState.OS_AWAITS_PAYMENT;
                            } else if (invoiceStates.get(invoiceStates.size() - 1) == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                                // last invoice is non-authorized, i.e. payment has failed but may be started again
                                return EDisplayOrderState.OS_AWAITS_PAYMENT;
                            } else {
                                return EDisplayOrderState.OS_IN_PROGRESS;
                            }
                        case OS_CONFIRMED:
                            return EDisplayOrderState.OS_FULFILLED;
                        case OS_REFUNDED:
                            return EDisplayOrderState.OS_REFUNDED;
                        default:
                            return EDisplayOrderState.OS_IN_PROGRESS;
                    }
                }
        }
    }
}
