package ru.yandex.travel.orders.services;

import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.train.model.TrainPlace;

public class FiscalTitleGenerator {
    public static String getFiscalTitle(Order order) {
        if (order.getPublicType() == EOrderType.OT_TRAIN) {
            Preconditions.checkState(order.getOrderItems().size() == 1,
                    "Number of items in TrainOrder %s is %s, expected 1.", order.getId(), order.getOrderItems().size());
            TrainOrderItem orderItem = (TrainOrderItem) order.getOrderItems().get(0);
            return getTrainFiscalTitle(orderItem);
        } else if (OrderCompatibilityUtils.isTrainOrder(order)) {
            return getTrainFiscalTitle(OrderCompatibilityUtils.getTrainOrderItems(order));
        }
        return null;
    }

    private static String getTrainFiscalTitle(List<TrainOrderItem> orderItems) {
        boolean manyTickets = orderItems.stream().flatMap(x -> x.getPayload().getPassengers().stream())
                .map(x -> x.getTicket().getBlankId()).distinct().count() > 1;
        boolean manyTrains = orderItems.size() > 1;
        String ticketString = manyTickets ? "билетов" : "билета";
        String trainString = manyTrains ? "поезда" : "поезд";
        List<String> rows = new ArrayList<>();
        rows.add(String.format("Заказ %s на %s:", ticketString, trainString));
        for (var orderItem : orderItems) {
            var departure = ImHelpers.toLocalDateTime(orderItem.getReservation().getDepartureTime(),
                    orderItem.getReservation().getStationFromTimezone());
            rows.add(String.format("Поезд %s из %s %s %s %s, вагон %s",
                    orderItem.getReservation().getReservationRequestData().getTrainTicketNumber(),
                    orderItem.getReservation().getUiData().getStationFromTitleGenitive(),
                    orderItem.getReservation().getUiData().getStationToPreposition(),
                    orderItem.getReservation().getUiData().getStationToTitleAccusative(),
                    DateTimeFormatter.ofPattern("dd MMMM yyyy года в HH:mm").withLocale(new Locale("ru")).format(departure),
                    orderItem.getReservation().getCarNumber()));
        }
        return String.join("\n", rows);
    }

    private static String getTrainFiscalTitle(TrainOrderItem orderItem) {
        var departure = ImHelpers.toLocalDateTime(orderItem.getReservation().getDepartureTime(),
                orderItem.getReservation().getStationFromTimezone());
        String ticketString = "билета";
        if (orderItem.getReservation().getPassengers().size() > 1) {
            ticketString = "билетов";
        }
        return String.format("Заказ %s на поезд %s из %s %s %s %s, вагон %s",
                ticketString,
                orderItem.getReservation().getReservationRequestData().getTrainTicketNumber(),
                orderItem.getReservation().getUiData().getStationFromTitleGenitive(),
                orderItem.getReservation().getUiData().getStationToPreposition(),
                orderItem.getReservation().getUiData().getStationToTitleAccusative(),
                DateTimeFormatter.ofPattern("dd MMMM yyyy года в HH:mm").withLocale(new Locale("ru")).format(departure),
                orderItem.getReservation().getCarNumber());
    }

    public static String makeTrainFiscalItemTitle(String titleStart, List<TrainPlace> places, boolean manyTrains,
                                                  String trainNumber) {
        String result = titleStart;
        if (manyTrains) {
            result += ", поезд " + trainNumber;
        }
        if (places == null || places.size() == 0) {
            // pass
        } else if (places.size() == 1) {
            result += ", место " + places.get(0).getNumber();
        } else {
            result += ", места " + places.stream().map(TrainPlace::getNumber).collect(Collectors.joining(", "));
        }
        return result;
    }

    public static String makeBusFiscalItemTitle(String titleStart, String place) {
        String result = titleStart;
        if (!Strings.isNullOrEmpty(place)) {
            result += ", место " + place;
        }
        return result;
    }
}
