package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Comparator;
import java.util.Locale;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.hotels.common.LocationType;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.notifications.hotels.ConfirmedOrderMailRequest;
import ru.yandex.travel.orders.entities.notifications.hotels.MailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.hotels.RefundMailRequest;
import ru.yandex.travel.orders.entities.promo.mir2020.MirPromoOrderEligibility;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.mir2020.Mir2020PromoService;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.order.hotel.HotelWorkflowProperties;

@Service
@Slf4j
@EnableConfigurationProperties(HotelWorkflowProperties.class)
public class MailSenderHelper {
    private final HotelWorkflowProperties config;
    private final Mir2020PromoService mir2020PromoService;

    public MailSenderHelper(HotelWorkflowProperties config, Mir2020PromoService mir2020PromoService) {
        this.config = config;
        this.mir2020PromoService = mir2020PromoService;
    }

    public String getPhoneForLocation(LocationType location) {
        switch (location) {
            case MOSCOW:
                return config.getMail().getMoscowPhone();
            case GLOBAL:
                return config.getMail().getGlobalPhone();
            case RUSSIA:
            default:
                return config.getMail().getRussiaPhone();
        }
    }

    public String prepareCheckinTimes(OrderDetails orderDetails) {
        String result = null;
        if (StringUtils.isNotEmpty(orderDetails.getCheckinBegin())) {
            result = "с " + orderDetails.getCheckinBegin();
        }
        if (StringUtils.isNotEmpty(orderDetails.getCheckinEnd())) {
            result = result + " по " + orderDetails.getCheckinEnd();
        }
        return result;
    }

    public String unquoteName(String name) {
        if (name.startsWith("«") && name.endsWith("»")) {
            return name.substring(1, name.length() - 1);
        } else {
            return name;
        }
    }

    public String prepareOrderUrl(String orderId) {
        return String.format(config.getMail().getOrderUrlBase(), orderId);
    }

    private void prepareBaseMailRequestForHotel(HotelOrderItem hotelOrderItem, MailSenderArgs request) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("dd MMMM yyyy", new Locale("ru", "RU"));
        request.setOrderId(hotelOrderItem.getOrder().getPrettyId());
        HotelItinerary itinerary = hotelOrderItem.getHotelItinerary();
        request.setCheckinDate(formatter.format(itinerary.getOrderDetails().getCheckinDate()));
        request.setCheckoutDate(formatter.format(itinerary.getOrderDetails().getCheckoutDate()));

        Preconditions.checkArgument(itinerary.getGuests().size() > 0, "There should be at least 1 guest");
        request.setContactsName(itinerary.getGuests().get(0).getFirstName());
        request.setGuestsList(itinerary.getGuests());

        request.setHotelName(unquoteName(itinerary.getOrderDetails().getHotelName()));
        request.setRoomName(itinerary.getOrderDetails().getRoomName());
        request.setTotalOrderVolume(hotelOrderItem.getOrder().calculateTotalCost().getNumber().intValue());
        request.setSupportPhone(getPhoneForLocation(itinerary.getOrderDetails().getLocationType()));
        request.setOrderUrl(prepareOrderUrl(hotelOrderItem.getOrder().getId().toString()));
    }

    public ConfirmedOrderMailRequest prepareConfirmedOrderMailRequest(Order hotelOrder) {
        OrderCompatibilityUtils.ensureHotelOrder(hotelOrder);
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(hotelOrder);
        ConfirmedOrderMailRequest request = new ConfirmedOrderMailRequest();
        prepareBaseMailRequestForHotel(orderItem, request);
        if (orderItem.getHotelItinerary().getOrderDetails().getCheckinBegin() != null) {
            request.setCheckinTimes(prepareCheckinTimes(orderItem.getHotelItinerary().getOrderDetails()));
        }
        if (orderItem.getHotelItinerary().getOrderDetails().getCheckoutEnd() != null) {
            request.setCheckoutTimes(orderItem.getHotelItinerary().getOrderDetails().getCheckoutEnd());
        }
        var mirPromo = mir2020PromoService.getPromoParticipationStatus(hotelOrder);
        if (mirPromo != null) {
            boolean mirCashback = mirPromo.getEligibility() == MirPromoOrderEligibility.ELIGIBLE &&
                    mirPromo.getPaidWithMir() != null && mirPromo.getPaidWithMir();
            log.info("Will include mir_cashback flag = {}", mirCashback);
            request.setMirCashback(mirCashback);
        } else {
            log.info("Will skip mir_cashback flag");
        }
        if (hotelOrder.getPaymentSchedule() != null) {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("dd MMMM yyyy HH:mm по московскому времени",
                            new Locale("ru", "RU"))
                    .withZone(ZoneId.of("Europe/Moscow"));
            Money paidAmount = hotelOrder.getPaymentSchedule().getPaidAmount();
            Money remainingAmount = hotelOrder.getPaymentSchedule().getItems().stream()
                    .map(i -> i.getPendingInvoice().getTotalAmount())
                    .reduce(Money::add)
                    .orElse(null);
            Instant timeToPay = hotelOrder.getPaymentSchedule().getItems().stream()
                    .min(Comparator.comparing(PaymentScheduleItem::getPaymentEndsAt))
                    .map(PaymentScheduleItem::getPaymentEndsAt)
                    .orElse(null);
            RefundRule ruleAtPayment = orderItem.getHotelItinerary().getRefundRules().getRuleAtInstant(timeToPay);
            Preconditions.checkState(remainingAmount != null, "No remaining amount");
            Preconditions.checkState(!remainingAmount.isZero(), "Unexpected zero in remaining amount");
            Preconditions.checkState(timeToPay != null, "No instant to pay");
            Preconditions.checkState(ruleAtPayment != null, "Refund rule unknown for deferred payment");
            int penalty;
            switch (ruleAtPayment.getType()) {
                case FULLY_REFUNDABLE:
                    penalty = 0;
                    break;
                case REFUNDABLE_WITH_PENALTY:
                    penalty = ruleAtPayment.getPenalty().getNumber().intValue();
                    break;
                default:
                    throw new IllegalStateException("Unexpected non-refundable penalty");
            }
            if (hotelOrder.getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
                request.setDeferMode(ConfirmedOrderMailRequest.DeferMode.DEFERRED);
            } else {
                request.setDeferMode(ConfirmedOrderMailRequest.DeferMode.INITIAL);
            }
            request.setPaidAmount(paidAmount.getNumber().intValue());
            request.setDeferredAmount(remainingAmount.getNumber().intValue());
            request.setDeferredDate(formatter.format(timeToPay));
            request.setDeferredPenalty(penalty);
            request.setPaymentUrl(String.format(config.getMail().getPaymentUrlBase(), hotelOrder.getId().toString()));
        } else {
            request.setDeferMode(ConfirmedOrderMailRequest.DeferMode.FULL);
        }
        return request;
    }

    public RefundMailRequest prepareRefundMailRequest(HotelOrder hotelOrder) {
        Preconditions.checkArgument(hotelOrder.getOrderItems().size() == 1,
                "Confirmation email is currently possible only for Hotel orders with 1 item");
        HotelOrderItem hotelOrderItem = (HotelOrderItem) hotelOrder.getOrderItems().get(0);
        RefundMailRequest request = new RefundMailRequest();
        prepareBaseMailRequestForHotel(hotelOrderItem, request);
        if (hotelOrderItem.getHotelItinerary().getRefundInfo() != null) {
            BigDecimal refundAmount =
                    BigDecimal.valueOf(Double.parseDouble(hotelOrderItem.getHotelItinerary().getRefundInfo().getRefund().getAmount()));
            BigDecimal penaltyAmount =
                    BigDecimal.valueOf(Double.parseDouble(hotelOrderItem.getHotelItinerary().getRefundInfo().getPenalty().getAmount()));
            request.setTotalRefundVolume(refundAmount.intValue());
            request.setPenaltyAmount(penaltyAmount.intValue());
            switch (hotelOrderItem.getHotelItinerary().getRefundInfo().getReason()) {
                case USER:
                    request.setUserCancelled(true);
                    break;
                case SCHEDULE:
                    request.setScheduleCancelled(true);
                    break;
            }
        }
        request.setSupportPhone(getPhoneForLocation(hotelOrderItem.getHotelItinerary().getOrderDetails().getLocationType()));
        return request;
    }
}
