package ru.yandex.travel.orders.services;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Realm;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.entities.notifications.MailSenderAttachment;
import ru.yandex.travel.orders.entities.notifications.MailSenderRetryableException;
import ru.yandex.travel.orders.entities.notifications.MailTarget;

/**
 * Sender (Rassylyator) docs:
 * https://wiki.yandex-team.ru/sender/api/
 * <p>
 * Transactional emails:
 * https://github.yandex-team.ru/sendr/sendr/blob/master/docs/transaction-api.md
 */
@Slf4j
@Service
@EnableConfigurationProperties(MailSenderConfigurationProperties.class)
public class MailSenderServiceImpl implements MailSenderService {
    private final static ObjectMapper mapper = new ObjectMapper()
            .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);

    private final MailSenderConfigurationProperties config;
    private final AsyncHttpClientWrapper ahcClient;

    public MailSenderServiceImpl(MailSenderConfigurationProperties config,
                                 @Qualifier("mailSenderAhcClientWrapper") AsyncHttpClientWrapper ahcClient) {
        this.config = config;
        this.ahcClient = ahcClient;
    }

    @Override
    public void sendEmailSync(String campaignId, String targetEmail, Collection<MailTarget> multipleTargets,
                              Collection<MailTarget> copies, Collection<MailTarget> blindCopies,
                              Object args, Collection<MailSenderAttachment> attachments) {
        try {
            String argumentsJson = mapper.writeValueAsString(args);
            String attachmentsJson = mapper.writeValueAsString(attachments != null ? attachments : List.of());
            String multipleTargetsJson = multipleTargets == null || multipleTargets.isEmpty() ? null :
                    mapper.writeValueAsString(multipleTargets);
            String copiesJson = copies == null || copies.isEmpty() ? null :
                    mapper.writeValueAsString(copies);
            String blindCopiesJson = blindCopies == null || blindCopies.isEmpty() ? null :
                    mapper.writeValueAsString(blindCopies);
            sync(sendEmailAsync(campaignId, targetEmail, multipleTargetsJson, copiesJson, blindCopiesJson, argumentsJson, attachmentsJson));
        } catch (JsonProcessingException e) {
            log.error("Error occurred while serializing email info", e);
            throw new RuntimeException("Error occurred while serializing email info", e);
        }
    }

    public static JsonNode convertArguments(Object args) {
        return mapper.valueToTree(args);
    }

    private CompletableFuture<Response> sendEmailAsync(String campaign, String targetEmail,
                                                       String multipleTargets, String copies, String blindCopies,
                                                       String args, String attachments) {
        if (config.getDebug().isEnabled() && config.getDebug().getFailEmail().equals(targetEmail)) {
            return CompletableFuture.failedFuture(new RuntimeException("Exception for testing purposes"));
        }
        Realm realm = new Realm.Builder(config.getAuthenticationKey(), "")
                .setScheme(Realm.AuthScheme.BASIC)
                .setUsePreemptiveAuth(true)
                .build();
        // old configs used to set campaign ids as "<CAMPAIGN_ID>/send", now we support proper ids without this suffix
        String actionSuffix = campaign.endsWith("/send") ? "" : "/send";
        String url = config.getMailerUrlBase() + campaign + actionSuffix;
        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod("POST")
                .setUrl(url)
                .setRealm(realm)
                .addFormParam("async", "true")
                .addFormParam("args", args)
                .addFormParam("attachments", attachments);
        if (!Strings.isNullOrEmpty(targetEmail)) {
            requestBuilder.addQueryParam("to_email", targetEmail);
        } else {
            requestBuilder.addFormParam("to", multipleTargets);
        }
        if (!Strings.isNullOrEmpty(copies)) {
            requestBuilder.addFormParam("cc", copies);
        }
        if (!Strings.isNullOrEmpty(blindCopies)) {
            requestBuilder.addFormParam("bcc", blindCopies);
        }
        return ahcClient.executeRequest(requestBuilder);
    }


    private Response sync(CompletableFuture<Response> future) {
        try {
            Response response = future.get();
            int statusCode = response.getStatusCode();
            if (statusCode >= 200 && statusCode < 300) {
                return response;
            } else if (statusCode >= 500) {
                throw new MailSenderRetryableException(
                        String.format("Exception happened while executing http call: %s. Response body: %s",
                                response.getStatusText(), response.getResponseBody()));
            } else {
                throw new RuntimeException(
                        String.format("Exception happened while executing http call: %s. Response body: %s",
                                response.getStatusText(), response.getResponseBody()));
            }
        } catch (InterruptedException e) {
            log.error("Mailer call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new MailSenderRetryableException(e);
        } catch (ExecutionException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            }
            Throwable cause = e.getCause();
            if (cause instanceof TimeoutException) {
                throw new MailSenderRetryableException(e);
            } else if (cause instanceof IOException) {
                throw new MailSenderRetryableException(e);
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e);
            }
        }
    }
}
