package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.FiscalReceiptType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.ImAttachmentProviderData;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.entities.notifications.SmsChannelInfo;
import ru.yandex.travel.orders.entities.notifications.TemplatedEmailChannelInfo;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.EInvoiceRefundType;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.order.hotel.HotelWorkflowProperties;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties({HotelWorkflowProperties.class, TrainWorkflowProperties.class})
public class NotificationHelper {
    private static final Locale RU_LOCALE = new Locale("ru", "RU");

    private final MailSenderHelper mailSenderHelper;
    private final TrainNotificationHelper trainNotificationHelper;
    private final WorkflowRepository workflowRepository;
    private final NotificationRepository notificationRepository;

    private final HotelWorkflowProperties hotelWorkflowProperties;
    private final TrainWorkflowProperties trainWorkflowProperties;

    public static String normalizeFreeFormatPhone(String phone) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(phone), "Notification phone can't be empty");
        // check many phones separated by ...
        for (String separator : List.of(";", ",", ".", "/", ":", " ", "-")) {
            if (!phone.contains(separator)) {
                continue;
            }
            List<String> phones = Arrays.stream(phone.split(Pattern.quote(separator)))
                    .filter(p -> !Strings.isNullOrEmpty(p))
                    .map(NotificationHelper::cutPhoneSymbols)
                    .filter(p -> p.length() >= 11)
                    .collect(Collectors.toList());
            if (phones.size() > 0) {
                return phones.get(0);
            }
        }
        return cutPhoneSymbols(phone);
    }

    private static String cutPhoneSymbols(String phone) {
        phone = phone.replaceAll("[^\\d\\+]+", "");
        return (phone.startsWith("+") ? "+" : "") + phone.replaceAll("\\+", "");
    }

    public UUID createWorkflowForSuccessfulHotelNotification(Order order) {
        OrderCompatibilityUtils.ensureHotelOrder(order);
        Preconditions.checkState(OrderCompatibilityUtils.isConfirmed(order),
                "Order must be confirmed for successful notification");
        Notification email;
        if (hotelWorkflowProperties.getMail().isUseTemplatedMailSenderService()) {
            var emailInfo = new TemplatedEmailChannelInfo();
            emailInfo.setTemplate(hotelWorkflowProperties.getMail().getOrderConfirmedTemplate());
            emailInfo.setArguments(new POJONode(mailSenderHelper.prepareConfirmedOrderMailRequest(order)));
            emailInfo.setTarget(order.getEmail());
            email = Notification.createTemplatedEmailNotification(order, emailInfo);
        } else {
            var emailInfo = new EmailChannelInfo();
            emailInfo.setCampaign(hotelWorkflowProperties.getMail().getOrderConfirmedCampaign());
            emailInfo.setArguments(new POJONode(mailSenderHelper.prepareConfirmedOrderMailRequest(order)));
            emailInfo.setTarget(order.getEmail());
            email = Notification.createEmailNotification(order, emailInfo);
        }
        email.setPreparingAttachmentsTill(Instant.now().plus(hotelWorkflowProperties.getMail().getPreparingAttachmentsTime()));
        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        var voucher = Attachment.createUrlAttachment(email, "Voucher_" + email.getOrderPrettyId() + ".pdf",
                NotificationFileTypes.PDF, true, order.getDocumentUrl());
        workflowRepository.save(Workflow.createWorkflowForEntity(voucher, workflow.getId()));
        if (order.getPayments().size() > 0) {
            List<FiscalReceipt> receipts = order.getPayments().stream()
                    .flatMap(p -> p.getReceipts().stream())
                    .filter(r -> r.getReceiptType() != FiscalReceiptType.REFUND)
                    .collect(Collectors.toList());
            if (receipts.isEmpty()) {
                log.info("No fiscal receipts exist yet");
            } else {
                for (var i = 0; i < receipts.size(); i++) {
                    var r = receipts.get(i);
                    var trustFiscalReceipt = Attachment.createFiscalReceiptAttachment(email,
                            "Receipt_" + email.getOrderPrettyId() + "_" + (i + 1) + ".pdf",
                            NotificationFileTypes.PDF, false, r.getId());
                    workflowRepository.save(Workflow.createWorkflowForEntity(trustFiscalReceipt, workflow.getId()));
                }
            }
        } else if (!order.isFullyPostPaid()) {
            // legacy case
            TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
            var receipt = invoice.getFiscalReceipts().stream()
                    .filter(i -> i.getReceiptType() == FiscalReceiptType.ACQUIRE)
                    .collect(Collectors.toList());
            Preconditions.checkState(receipt.size() == 1, "exactly 1 receipt is expected");
            var trustFiscalReceipt = Attachment.createFiscalReceiptAttachment(email,
                    "Receipt_" + email.getOrderPrettyId() + ".pdf",
                    NotificationFileTypes.PDF, false, receipt.get(0).getId());
            workflowRepository.save(Workflow.createWorkflowForEntity(trustFiscalReceipt, workflow.getId()));
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    public UUID createWorkflowForRefundHotelNotification(HotelOrder order) {
        Preconditions.checkState(order.getState() == EHotelOrderState.OS_REFUNDED, "Order must be refunded for " +
                "successful refund notification");
        Preconditions.checkArgument(order.getOrderItems().size() == 1, "There should be 1 order item");
        OrderItem orderItem = order.getOrderItems().get(0);

        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(hotelWorkflowProperties.getMail().getOrderRefundedCampaign());
        emailInfo.setArguments(new POJONode(mailSenderHelper.prepareRefundMailRequest(order)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(hotelWorkflowProperties.getMail().getPreparingAttachmentsTime()));
        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        if (order.getPayments().size() > 0) {
            List<FiscalReceipt> receipts = order.getPayments().stream()
                    .flatMap(p -> p.getReceipts().stream())
                    .filter(r -> r.getReceiptType() != FiscalReceiptType.ACQUIRE)
                    .collect(Collectors.toList());
            for (var i = 0; i < receipts.size(); i++) {
                var r = receipts.get(i);
                var trustFiscalReceipt = Attachment.createFiscalReceiptAttachment(email,
                        "Receipt_" + email.getOrderPrettyId() + "_" + (i + 1) + ".pdf",
                        NotificationFileTypes.PDF, false, r.getId());
                workflowRepository.save(Workflow.createWorkflowForEntity(trustFiscalReceipt, workflow.getId()));
            }
        } else if (!order.isFullyPostPaid()) {
            // legacy case
            TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
            var isCancelledBeforeClearing = invoice.getState() == ETrustInvoiceState.IS_CANCELLED;
            if (isCancelledBeforeClearing) {
                var receipt = invoice.getFiscalReceipts().stream()
                        .filter(i -> i.getReceiptType() == FiscalReceiptType.CLEAR)
                        .collect(Collectors.toList());
                Preconditions.checkState(receipt.size() == 1, "exactly 1 receipt is expected");
                var trustFiscalReceipt = Attachment.createFiscalReceiptAttachment(email,
                        "Receipt_" + email.getOrderPrettyId() + ".pdf",
                        NotificationFileTypes.PDF, false, receipt.get(0).getId());
                workflowRepository.save(Workflow.createWorkflowForEntity(trustFiscalReceipt, workflow.getId()));
            } else {
                var workflowId = workflow.getId();
                invoice.getFiscalReceipts().stream()
                        .filter(i -> i.getReceiptType() == FiscalReceiptType.REFUND)
                        .forEach(receipt -> {
                            var trustFiscalReceipt = Attachment.createFiscalReceiptAttachment(
                                    email,
                                    "Refund_" + receipt.getId() + "_for_" + email.getOrderPrettyId() + ".pdf",
                                    NotificationFileTypes.PDF,
                                    false,
                                    receipt.getId());
                            workflowRepository.save(Workflow.createWorkflowForEntity(trustFiscalReceipt, workflowId));
                        });
            }
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    public UUID createWorkflowForHotelOrderRequiresPayment(HotelOrder order) {
        Preconditions.checkState(order.getPaymentSchedule() != null &&
                        order.getPaymentSchedule().getState() == EPaymentState.PS_PARTIALLY_PAID,
                "PaymentNotification expects deferred order");
        Preconditions.checkArgument(order.getOrderItems().size() == 1, "There should be 1 order item");
        OrderItem orderItem = order.getOrderItems().get(0);
        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(hotelWorkflowProperties.getMail().getOrderRequiresPaymentCampaign());
        emailInfo.setArguments(new POJONode(mailSenderHelper.prepareConfirmedOrderMailRequest(order)));
        emailInfo.setTarget(order.getEmail());
        Notification email = Notification.createEmailNotification(order, emailInfo);
        // It's crucial to notify the user, so let's crash the order if we could not send the email for some reason,
        //  so using order's WF as the supervisor
        Workflow workflow = Workflow.createWorkflowForEntity(email, order.getWorkflow().getId());
        workflow = workflowRepository.save(workflow);
        notificationRepository.save(email);
        return workflow.getId();
    }

    public UUID createWorkflowForSuccessfulTrainEmailV2(GenericOrder order, OrderRefund insuranceAutoRefund) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getName());
        Preconditions.checkState(OrderCompatibilityUtils.isConfirmed(order), "Order must be confirmed for " +
                "successful notification");

        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(trainWorkflowProperties.getMail().getOrderConfirmedGenericCampaign());
        emailInfo.setArguments(new POJONode(trainNotificationHelper.prepareGenericConfirmedMailSenderArgs(order)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(trainWorkflowProperties.getMail().getPreparingAttachmentsTime()));

        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);

        var orderItems = OrderCompatibilityUtils.getTrainOrderItems(order);
        var orderItemsToGetBlank =
                orderItems.stream().filter(x -> !x.getPayload().isSlaveItem()).collect(Collectors.toList());
        for (int i = 0, orderItemsToGetBlankSize = orderItemsToGetBlank.size(); i < orderItemsToGetBlankSize; i++) {
            TrainOrderItem orderItem = orderItemsToGetBlank.get(i);
            var blankRequest = new OrderReservationBlankRequest(orderItem.getPayload().getPartnerOrderId(), null);
            var ticket = Attachment.createImAttachment(email,
                    "Ticket_" + email.getOrderPrettyId() + (orderItemsToGetBlank.size() > 1 ? "_" + i : "") + ".pdf",
                    NotificationFileTypes.PDF, true,
                    // Nasty hack to omit using real im client
                    new ImAttachmentProviderData(blankRequest, orderItem.getTestContext() != null)
            );
            workflowRepository.save(Workflow.createWorkflowForEntity(ticket, workflow.getId()));
        }
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        var receipt = invoice.getFiscalReceipts().stream()
                .filter(i -> i.getReceiptType() == FiscalReceiptType.ACQUIRE)
                .collect(Collectors.toList());
        Preconditions.checkState(receipt.size() == 1, "exactly 1 receipt is expected");
        var trustCheck = Attachment.createFiscalReceiptAttachment(email, "Check_" + email.getOrderPrettyId() + ".pdf",
                NotificationFileTypes.PDF, false, receipt.get(0).getId());
        workflowRepository.save(Workflow.createWorkflowForEntity(trustCheck, workflow.getId()));
        if (insuranceAutoRefund != null) {
            List<FiscalReceipt> refundReceipts = order.getCurrentInvoice().getFiscalReceipts().stream()
                    .filter(i -> insuranceAutoRefund.getId().equals(i.getOrderRefundId()))
                    .collect(Collectors.toList());
            Preconditions.checkState(refundReceipts.size() == 1, "exactly 1 insurance refund receipt is expected");
            var trustRefundCheck = Attachment.createFiscalReceiptAttachment(email,
                    "Refund_" + email.getOrderPrettyId() + ".pdf",
                    NotificationFileTypes.PDF, false, refundReceipts.get(0).getId());
            workflowRepository.save(Workflow.createWorkflowForEntity(trustRefundCheck, workflow.getId()));
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    public UUID createWorkflowForSuccessfulTrainSms(Order order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getName());
        Preconditions.checkState(OrderCompatibilityUtils.isConfirmed(order), "Order must be confirmed for " +
                "successful notification");
        var text = trainNotificationHelper.getConfirmationSmsText(order);
        var channelInfo = new SmsChannelInfo();
        channelInfo.setPhone(normalizeFreeFormatPhone(order.getPhone()));
        channelInfo.setText(text);
        var sms = Notification.createSmsNotification(order, channelInfo);
        Workflow workflow = Workflow.createWorkflowForEntity(sms, WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        notificationRepository.save(sms);
        return workflow.getId();
    }

    public UUID createWorkflowForTrainRefundEmail(Order order, OrderRefund refund,
                                                  TrainTicketRefund ticketRefund) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getName());
        TrainOrderItem orderItem = OrderCompatibilityUtils.getOnlyTrainOrderItem(order);
        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(trainWorkflowProperties.getMail().getOrderRefundCampaign());
        boolean isResize = isResize(order, refund);
        emailInfo.setArguments(new POJONode(trainNotificationHelper.prepareRefundMailSenderArgs(order, orderItem,
                ticketRefund, isResize)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(trainWorkflowProperties.getMail().getPreparingAttachmentsTime()));
        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        if (refund.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND) {
            List<Integer> refundOperations = ticketRefund.getPayload().getRefundedItems().stream()
                    .map(PassengerRefundInfo::getRefundOperationId).filter(Objects::nonNull).distinct()
                    .collect(Collectors.toList());
            for (Integer refundOperation : refundOperations) {
                var blankRequest = new OrderReservationBlankRequest(orderItem.getPayload().getPartnerOrderId(),
                        refundOperation);
                var ticket = Attachment.createImAttachment(email, "Refund_" + refundOperation.toString() + ".pdf",
                        NotificationFileTypes.PDF, false, new ImAttachmentProviderData(blankRequest));
                workflowRepository.save(Workflow.createWorkflowForEntity(ticket, workflow.getId()));
            }
        }
        if (!ticketRefund.getPayload().calculateActualRefundSum().isZero()) {
            TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
            List<FiscalReceipt> refundReceipts = invoice.getFiscalReceipts().stream()
                    .filter(i -> refund.getId().equals(i.getOrderRefundId()))
                    .collect(Collectors.toList());
            Preconditions.checkState(refundReceipts.size() == 1,
                    "exactly 1 refund receipt is expected but got %s", refundReceipts.size());
            var trustRefundCheck = Attachment.createFiscalReceiptAttachment(email,
                    "Refund_" + email.getOrderPrettyId() + ".pdf",
                    NotificationFileTypes.PDF, false, refundReceipts.get(0).getId());
            workflowRepository.save(Workflow.createWorkflowForEntity(trustRefundCheck, workflow.getId()));
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    public UUID createWorkflowForTrainRefundEmailV2(GenericOrder order, OrderRefund refund,
                                                    List<TrainTicketRefund> ticketRefunds) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getName());
        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(trainWorkflowProperties.getMail().getOrderRefundGenericCampaign());
        boolean isResize = isResize(order, refund);
        emailInfo.setArguments(new POJONode(trainNotificationHelper.prepareGenericRefundMailSenderArgs(order,
                ticketRefunds, isResize)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(trainWorkflowProperties.getMail().getPreparingAttachmentsTime()));
        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        if (refund.getRefundType() == EOrderRefundType.RT_GENERIC_USER_REFUND) {
            for (TrainTicketRefund ticketRefund : ticketRefunds) {
                if (ticketRefund.getState() != ETrainTicketRefundState.RS_REFUNDED) {
                    continue;
                }
                TrainOrderItem orderItem = ticketRefund.getOrderItem();
                List<Integer> refundOperations = ticketRefund.getPayload().getRefundedItems().stream()
                        .map(PassengerRefundInfo::getRefundOperationId).filter(Objects::nonNull).distinct()
                        .collect(Collectors.toList());
                for (Integer refundOperation : refundOperations) {
                    var blankRequest = new OrderReservationBlankRequest(orderItem.getPayload().getPartnerOrderId(),
                            refundOperation);
                    var ticket = Attachment.createImAttachment(email,
                            "Refund_blank_" + refundOperation.toString() + ".pdf",
                            NotificationFileTypes.PDF, false, new ImAttachmentProviderData(blankRequest));
                    workflowRepository.save(Workflow.createWorkflowForEntity(ticket, workflow.getId()));
                }
            }
        }
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        List<FiscalReceipt> refundReceipts = invoice.getFiscalReceipts().stream()
                .filter(i -> refund.getId().equals(i.getOrderRefundId()))
                .collect(Collectors.toList());
        if (refundReceipts.size() > 0) {
            Preconditions.checkState(refundReceipts.size() == 1,
                    "exactly 1 refund receipt is expected but got %s", refundReceipts.size());
            var trustRefundCheck = Attachment.createFiscalReceiptAttachment(email,
                    "Refund_check_" + refund.getId() + ".pdf",
                    NotificationFileTypes.PDF, false, refundReceipts.get(0).getId());
            workflowRepository.save(Workflow.createWorkflowForEntity(trustRefundCheck, workflow.getId()));
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    public static boolean isResize(Order order, OrderRefund refund) {
        var isResize = false;
        if (refund.getInvoiceRefundType() != null) {
            isResize = refund.getInvoiceRefundType() == EInvoiceRefundType.EIR_RESIZE;
        } else if (order.getCurrentInvoice() instanceof TrustInvoice) {
            var invoiceState = ((TrustInvoice) order.getCurrentInvoice()).getState();
            isResize = invoiceState == ETrustInvoiceState.IS_HOLD;
        }
        return isResize;
    }

    public static String formatMoney(BigDecimal m) {
        if (m.remainder(BigDecimal.ONE).compareTo(BigDecimal.ZERO) == 0) {
            return m.toBigInteger().toString();
        } else {
            return m.setScale(2, RoundingMode.HALF_EVEN).toString();
        }
    }

    public static String humanDate(LocalDateTime dateTime) {
        var now = LocalDateTime.now();
        if (now.getYear() != dateTime.getYear()) {
            return dateTime.format(DateTimeFormatter.ofPattern("dd MMMM yyyy", RU_LOCALE));
        }
        return dateTime.format(DateTimeFormatter.ofPattern("dd MMMM", RU_LOCALE));
    }

    public static String humanTime(LocalDateTime dateTime) {
        return dateTime.format(DateTimeFormatter.ofPattern("HH:mm", RU_LOCALE));
    }

}
