package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.UInt32Value;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TFiscalReceipt;
import ru.yandex.travel.hotels.models.booking_flow.promo.WhiteLabelPromoCampaign;
import ru.yandex.travel.hotels.proto.EWhiteLabelEligibility;
import ru.yandex.travel.orders.admin.proto.EMir2020PromoEligibility;
import ru.yandex.travel.orders.admin.proto.EOrderTaxi2020PromoStatusEnum;
import ru.yandex.travel.orders.admin.proto.TAdminActionToken;
import ru.yandex.travel.orders.admin.proto.TAdminGeneratedPromoCodesInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderFlags;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInvoiceInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderItemInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderPriceInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderRefundInfo;
import ru.yandex.travel.orders.admin.proto.TAdminPayment;
import ru.yandex.travel.orders.admin.proto.TAdminPaymentSchedule;
import ru.yandex.travel.orders.admin.proto.TAdminPaymentScheduleItem;
import ru.yandex.travel.orders.admin.proto.TAdminPendingInvoice;
import ru.yandex.travel.orders.admin.proto.TAdminPromoCodeApplicationResult;
import ru.yandex.travel.orders.admin.proto.TAdminTrustRefundItemInfo;
import ru.yandex.travel.orders.admin.proto.TOrderPromoCampaignsInfo;
import ru.yandex.travel.orders.admin.proto.TOrderTaxi2020PromoCampaignInfo;
import ru.yandex.travel.orders.admin.proto.TYandexPlusInfo;
import ru.yandex.travel.orders.commons.proto.EAdminAction;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderAggregateState;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.TrustRefund;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivationsStrategy;
import ru.yandex.travel.orders.entities.promo.PromoCodeApplication;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;
import ru.yandex.travel.orders.entities.promo.mir2020.Mir2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.mir2020.MirPromoOrderEligibility;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus;
import ru.yandex.travel.orders.grpc.helpers.OrderProtoUtils;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.proto.ETaxi2020PromoStatusEnum;
import ru.yandex.travel.orders.proto.TGeneratedPromoCodesInfo;
import ru.yandex.travel.orders.proto.TMir2020PromoCampaignInfo;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderInvoiceInfo;
import ru.yandex.travel.orders.proto.TOrderPriceInfo;
import ru.yandex.travel.orders.proto.TOrderRefund;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TPaymentBreakdown;
import ru.yandex.travel.orders.proto.TPaymentInfo;
import ru.yandex.travel.orders.proto.TPromoCampaignsInfo;
import ru.yandex.travel.orders.proto.TPromoCodeApplicationResult;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TTaxi2020PromoCampaignInfo;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateErrorInfo;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateStateExtInfo;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.proto.TWhiteLabelPoints;
import ru.yandex.travel.orders.proto.TWhiteLabelPointsLinguistics;
import ru.yandex.travel.orders.proto.TWhiteLabelPromoCampaignInfo;
import ru.yandex.travel.orders.repository.OrderAggregateStateRepository;
import ru.yandex.travel.orders.services.admin.AdminUserCapabilitiesManager;
import ru.yandex.travel.orders.services.hotels.BusinessTripDocService;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.services.orders.OrderAggregateStateMapper;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.services.promo.campaigns.PromoCampaigns;
import ru.yandex.travel.orders.services.promo.campaigns.PromoCampaignsService;
import ru.yandex.travel.orders.services.train.OrderItemPayloadService;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.orders.workflows.invoice.trust.handlers.TrustUserInfoHelper;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.entities.Workflow;

@Service
@Slf4j
@RequiredArgsConstructor
public class OrderInfoMapper {
    private static final EOrderTaxi2020PromoStatusEnum TAXI_2020_PROMO_DEFAULT_STATUS =
            EOrderTaxi2020PromoStatusEnum.OTPS_UNKNOWN;
    private static final ImmutableMap<Taxi2020PromoOrderStatus, EOrderTaxi2020PromoStatusEnum>
            TAXI_2020_PROMO_STATUS_MAPPING =
            ImmutableMap.<Taxi2020PromoOrderStatus, EOrderTaxi2020PromoStatusEnum>builder()
                    .put(Taxi2020PromoOrderStatus.NOT_ELIGIBLE, EOrderTaxi2020PromoStatusEnum.OTPS_NOT_ELIGIBLE)
                    .put(Taxi2020PromoOrderStatus.ELIGIBLE, EOrderTaxi2020PromoStatusEnum.OTPS_ELIGIBLE)
                    .put(Taxi2020PromoOrderStatus.PROMO_CODE_CAN_BE_ASSIGNED,
                            EOrderTaxi2020PromoStatusEnum.OTPS_SENDING_EMAIL)
                    .put(Taxi2020PromoOrderStatus.EMAIL_SCHEDULED, EOrderTaxi2020PromoStatusEnum.OTPS_SENDING_EMAIL)
                    .put(Taxi2020PromoOrderStatus.EMAIL_SENT, EOrderTaxi2020PromoStatusEnum.OTPS_EMAIL_SENT)
                    .build();
    private static final ImmutableMap<MirPromoOrderEligibility, EMir2020PromoEligibility>
            MIR_2020_PROMO_ELIGIBILITY_MAPPING =
            ImmutableMap.<MirPromoOrderEligibility, EMir2020PromoEligibility>builder()
                    .put(MirPromoOrderEligibility.ELIGIBLE, EMir2020PromoEligibility.MIR_ELIGIBILE)
                    .put(MirPromoOrderEligibility.BLACKLISTED, EMir2020PromoEligibility.MIR_BLACKLISTED)
                    .put(MirPromoOrderEligibility.WRONG_LOS, EMir2020PromoEligibility.MIR_WRONG_LOS)
                    .put(MirPromoOrderEligibility.WRONG_STAY_DATES, EMir2020PromoEligibility.MIR_WRONG_STAY_DATES)
                    .put(MirPromoOrderEligibility.PROMO_DISABLED, EMir2020PromoEligibility.MIR_PROMO_DISABLED)
                    .build();

    private final OrderItemPayloadService orderItemPayloadService;
    private final TokenEncrypter tokenEncrypter;
    private final CheckMoneyRefundsService checkMoneyRefundsService;
    private final RefundPartsService refundPartsService;
    private final PromoCampaignsService promoCampaignsService;
    private final OrderAggregateStateRepository orderAggregateStateRepository;
    private final OrderAggregateStateMapper orderAggregateStateMapper;
    private final YandexPlusPromoService plusPromoService;
    private final OrdersAdminEnumMapper ordersAdminEnumMapper;
    private final BusinessTripDocService businessTripDocService;
    private final TrustClientProvider trustClientProvider;

    private static TAdminOrderItemInfo.Builder buildAdminOrderItemInfo(OrderItem orderItem) {
        var orderItemInfoBuilder = TAdminOrderItemInfo.newBuilder()
                .setOrderItemId(orderItem.getId().toString())
                .setOrderItemType(orderItem.getPublicType())
                .setItemState(orderItem.getItemState().toString())
                .setCreatedAt(ProtoUtils.fromInstant(orderItem.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(orderItem.getUpdatedAt()))
                .setHasTestContext(orderItem.getTestContext() != null)
                .setIsPostPayEligible(orderItem.isPostPayEligible())
                .setIsPostPaid(orderItem.isPostPaid());
        if (orderItem.getConfirmedAt() != null) {
            orderItemInfoBuilder.setConfirmedAt(ProtoUtils.fromInstant(orderItem.getConfirmedAt()));
        }
        if (orderItem.getRefundedAt() != null) {
            orderItemInfoBuilder.setRefundedAt(ProtoUtils.fromInstant(orderItem.getRefundedAt()));
        }

        if (orderItem.getExpiresAt() != null) {
            orderItemInfoBuilder.setExpiresAt(ProtoUtils.fromInstant(orderItem.getExpiresAt()));
        }

        if (orderItem.getPayload() != null) {
            orderItemInfoBuilder.setPayload(ProtoUtils.toTJson(orderItem.getPayload()));
        }

        Workflow workflow = orderItem.getWorkflow();
        orderItemInfoBuilder.setWorkflowId(workflow.getId().toString());
        orderItemInfoBuilder.setWorkflowType(workflow.getEntityType());
        return orderItemInfoBuilder;
    }

    private TAdminOrderInvoiceInfo.Builder buildAdminOrderInvoiceInfo(Invoice invoice,
                                                                             ProtoCurrencyUnit orderCurrency,
                                                                             List<TrustRefund> refunds,
                                                                             boolean needReceiptData) {
        var invoiceInfoBuilder = TAdminOrderInvoiceInfo.newBuilder()
                .setInvoiceId(invoice.getId().toString())
                .setInvoiceType(invoice.getPublicType());
        switch (invoice.getPublicType()) {
            case IT_TRUST:
                invoiceInfoBuilder.setTrustInvoiceState((ETrustInvoiceState) invoice.getInvoiceState());
                TrustInvoice trustInvoice = (TrustInvoice) invoice;
                invoiceInfoBuilder.setRrn(Strings.nullToEmpty(trustInvoice.getRrn()));
                break;
            case IT_AVIA_AEROFLOT:
                invoiceInfoBuilder.setAeroflotInvoiceState((EAeroflotInvoiceState) invoice.getInvoiceState());
                break;
        }
        if (invoice.getPaymentStartTs() != null) {
            invoiceInfoBuilder.setPaymentStartTs(ProtoUtils.fromInstant(invoice.getPaymentStartTs()));
        }
        if (invoice.getPaymentCancelTs() != null) {
            invoiceInfoBuilder.setPaymentCancelTs(ProtoUtils.fromInstant(invoice.getPaymentCancelTs()));
        }
        if (invoice.getTrustPaymentTs() != null) {
            invoiceInfoBuilder.setPaymentTs(ProtoUtils.fromInstant(invoice.getTrustPaymentTs()));
        }
        invoiceInfoBuilder.setPaymentUrl(Strings.nullToEmpty(invoice.getPaymentUrl()));
        invoiceInfoBuilder.setPurchaseToken(Strings.nullToEmpty(invoice.getPurchaseToken()));
        invoiceInfoBuilder.setTrustPaymentId(Strings.nullToEmpty(invoice.getTrustPaymentId()));
        invoiceInfoBuilder.setAuthorizationErrorCode(Strings.nullToEmpty(invoice.getAuthorizationErrorCode()));
        invoiceInfoBuilder.setAuthorizationErrorDesc(Strings.nullToEmpty(invoice.getAuthorizationErrorDesc()));
        invoiceInfoBuilder.setUserAccount(Strings.nullToEmpty(invoice.getUserAccount()));
        invoiceInfoBuilder.setApprovalCode(Strings.nullToEmpty(invoice.getApprovalCode()));

        if (invoice.getFiscalReceipts() != null) {
            for (FiscalReceipt receipt : invoice.getFiscalReceipts()) {
                var fiscalReceiptBuilder = invoiceInfoBuilder.addFiscalReceiptBuilder();
                fiscalReceiptBuilder.setType(receipt.getReceiptType().getProtoValue());
                fiscalReceiptBuilder.setUrl(Strings.nullToEmpty(receipt.getReceiptUrl()).replace("mode=mobile", "mode=pdf"));

                if (needReceiptData) {
                    // ходим синхронно в Траст, договоренность здесь
                    // https://st.yandex-team.ru/HOTELS-6312#62d446078fe6b319f6f0e8de
                    TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());
                    TrustPaymentReceiptResponse receiptResp = trustClient.getReceipt(
                            invoice.getPurchaseToken(),
                            TrustUserInfoHelper.createUserInfo(invoice));
                    fiscalReceiptBuilder.setDocumentIndex(receiptResp.getDocumentIndex());
                    fiscalReceiptBuilder.setShiftNumber(receiptResp.getShiftNumber());
                }
            }
        }

        Workflow workflow = invoice.getWorkflow();
        invoiceInfoBuilder.setWorkflowId(workflow.getId().toString());
        invoiceInfoBuilder.setWorkflowType(workflow.getEntityType());

        invoice.getInvoiceItems().forEach(invoiceItem ->
                invoiceInfoBuilder.addInvoiceItemBuilder()
                        .setInvoiceItemId(invoiceItem.getId())
                        .setFiscalItemId(invoiceItem.getFiscalItemId())
                        .setFiscalItemType(invoiceItem.getFiscalItemType().toString())
                        .setMoneyAmount(ProtoUtils.toTPrice(Money.of(invoiceItem.getPrice(), orderCurrency)))
                        .setOriginalMoneyAmount(ProtoUtils.toTPrice(Money.of(invoiceItem.getOriginalPrice(),
                                orderCurrency))));
        if (refunds != null) {
            refunds.forEach(trustRefund -> {
                var trustRefundBuilder = invoiceInfoBuilder.addTrustRefundBuilder()
                        .setId(trustRefund.getId().toString())
                        .setTrustRefundId(Strings.nullToEmpty(trustRefund.getTrustRefundId()))
                        .setState(trustRefund.getItemState().toString())
                        .setType(trustRefund.getLogEntityType())
                        .setCreatedAt(ProtoUtils.fromInstant(trustRefund.getCreatedAt()))
                        .addAllTrustRefundItem(trustRefund.getRefundItems().stream()
                                .map(item -> TAdminTrustRefundItemInfo.newBuilder()
                                        .setId(item.getId())
                                        .setOriginalAmount(ProtoUtils.toTPrice(Money.of(item.getOriginalAmount(),
                                                orderCurrency)))
                                        .setTargetAmount(ProtoUtils.toTPrice(Money.of(item.getTargetAmount(),
                                                orderCurrency)))
                                        .build())
                                .collect(Collectors.toList()));
                if (trustRefund.getTrustConfirmTs() != null) {
                    trustRefundBuilder.setConfirmedAt(ProtoUtils.fromInstant(trustRefund.getTrustConfirmTs()));
                }
                if (trustRefund.getOrderRefundId() != null) {
                    trustRefundBuilder.setOrderRefundId(trustRefund.getOrderRefundId().toString());
                }
            });
        }

        // provider-specific data
        if (invoice instanceof AeroflotInvoice) {
            AeroflotInvoice aeroflotInvoice = (AeroflotInvoice) invoice;
            invoiceInfoBuilder.setConfirmationUrl(Strings.nullToEmpty(aeroflotInvoice.getConfirmationUrl()));
        }

        return invoiceInfoBuilder;
    }

    private TAdminPayment buildAdminPayment(Payment payment, ProtoCurrencyUnit currency, Map<UUID,
            List<TrustRefund>> refundMap) {
        if (payment instanceof PaymentSchedule) {
            return TAdminPayment.newBuilder()
                    .setPaymentSchedule(buildPaymentSchedule((PaymentSchedule) payment, currency, refundMap))
                    .build();
        }
        if (payment instanceof PendingInvoice) {
            return TAdminPayment.newBuilder()
                    .setPendingInvoice(buildPendingInvoice((PendingInvoice) payment, currency, refundMap))
                    .build();
        }
        throw new IllegalStateException("Unsupported payment type " + payment.getPaymentType());
    }

    private TAdminPendingInvoice buildPendingInvoice(PendingInvoice pendingInvoice,
                                                            ProtoCurrencyUnit currency,
                                                            Map<UUID, List<TrustRefund>> refundMap) {
        var builder = TAdminPendingInvoice.newBuilder()
                .setId(pendingInvoice.getId().toString())
                .setState(pendingInvoice.getState())
                .setTotalAmount(ProtoUtils.toTPrice(pendingInvoice.getTotalAmount()))
                .setPendingAmount(ProtoUtils.toTPrice(pendingInvoice.getTotalAmount().subtract(pendingInvoice.getPaidAmount())))
                .setCreatedAt(ProtoUtils.fromInstant(pendingInvoice.getCreatedAt()))
                .setWorkflowId(pendingInvoice.getWorkflow().getId().toString());
        for (Invoice attempt : pendingInvoice.getPaymentAttempts()) {
            builder.addAttempts(buildAdminOrderInvoiceInfo(attempt, currency, refundMap.get(attempt.getId()), false));
        }
        if (pendingInvoice.getState() == EPaymentState.PS_FULLY_PAID) {
            builder.setPaidAt(ProtoUtils.fromInstant(pendingInvoice.getClosedAt()));
        }
        return builder.build();
    }

    private TAdminPaymentSchedule buildPaymentSchedule(PaymentSchedule schedule,
                                                              ProtoCurrencyUnit currency,
                                                              Map<UUID, List<TrustRefund>> refundMap) {
        return TAdminPaymentSchedule.newBuilder()
                .setId(schedule.getId().toString())
                .setState(schedule.getState())
                .setWorkflowId(schedule.getWorkflow().getId().toString())
                .setInitial(buildPendingInvoice(schedule.getInitialPendingInvoice(), currency, refundMap))
                .addAllItems(schedule.getItems().stream().map(i ->
                        TAdminPaymentScheduleItem.newBuilder()
                                .setId(i.getId().toString())
                                .setInvoice(buildPendingInvoice(i.getPendingInvoice(), currency, refundMap))
                                .setCreatedAt(ProtoUtils.fromInstant(i.getCreatedAt()))
                                .setPaymentEndsAt(ProtoUtils.fromInstant(i.getPaymentEndsAt()))
                                .setPenaltyIfUnpaid(ProtoUtils.toTPrice(i.getPenaltyIfUnpaid()))
                                .setNotificationEmailSent(i.isReminderEmailSent())
                                .setNotificationTicketCreated(i.getReminderTicket() != null)
                                .build()).collect(Collectors.toList()))
                .build();
    }

    public TOrderInfo.Builder buildOrderInfoFor(Order order, AuthorizedUser owner, boolean updateOrderOnTheFly,
                                                boolean addPromoCampaignsInfo,
                                                AggregateStateConstructMode aggregateStateConstructMode) {
        TOrderInfo.Builder orderInfoBuilder = TOrderInfo.newBuilder()
                .setOrderId(order.getId().toString())
                .setPrettyId(order.getPrettyId())
                .setOrderType(order.getPublicType())
                .setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()))
                .setDisplayOrderState(EDisplayOrderStateMapper.fromOrder(order));

        addOrderAggregateState(aggregateStateConstructMode, order, orderInfoBuilder);
        if (order.getStateContext() != null && order.getStateContext().getCancellationReason() != null) {
            orderInfoBuilder.setCancellationReason(order.getStateContext().getCancellationReason());
        }
        if (order.getExpiresAt() != null) {
            orderInfoBuilder.setExpiresAt(ProtoUtils.fromInstant(order.getExpiresAt()));
        }

        orderInfoBuilder.setDocumentUrl(Strings.nullToEmpty(order.getDocumentUrl()));
        orderInfoBuilder.setBusinessTripDocUrl(Strings.nullToEmpty(order.getBusinessTripDocUrl()));
        orderInfoBuilder.setCanGenerateBusinessTripDoc(
                businessTripDocService.canGenerateBusinessTripDoc(order).isCanGenerate());

        orderInfoBuilder.setOwner(buildUserInfo(order, owner));

        if (order.getServicedAt() != null) {
            orderInfoBuilder.setServicedAt(ProtoUtils.fromLocalDateTime(order.getServicedAt()));
        }
        orderInfoBuilder.setLabel(Strings.nullToEmpty(order.getLabel()));

        orderInfoBuilder.setDisplayOrderType(order.getDisplayType());
        Map<UUID, Object> updatedOrderItemsPayloads = new HashMap<>();
        for (OrderItem orderItem : order.getOrderItems()) {
            Object payload = orderItemPayloadService.getPayload(orderItem, updateOrderOnTheFly);
            updatedOrderItemsPayloads.put(orderItem.getId(), payload);
        }

        switch (order.getPublicType()) {
            case OT_HOTEL_EXPEDIA:
                orderInfoBuilder.setPaymentRetryEnabled(((HotelOrder) order).getPaymentRetryEnabled());
                orderInfoBuilder.setHotelOrderState((EHotelOrderState) order.getEntityState());
                break;
            case OT_AVIA_AEROFLOT:
                orderInfoBuilder.setPaymentRetryEnabled(false);
                orderInfoBuilder.setAeroflotOrderState((EAeroflotOrderState) order.getEntityState());
                break;
            case OT_TRAIN:
                orderInfoBuilder.setPaymentRetryEnabled(true);
                orderInfoBuilder.setTrainOrderState((ETrainOrderState) order.getEntityState());
                break;
            case OT_GENERIC:
                orderInfoBuilder.setPaymentRetryEnabled(((GenericOrder) order).getPaymentRetryEnabled() == Boolean.TRUE);
                orderInfoBuilder.setGenericOrderState((EOrderState) order.getEntityState());
                orderInfoBuilder.addAllRefundParts(refundPartsService.getRefundParts((GenericOrder) order,
                        updatedOrderItemsPayloads));
                break;
            default:
                throw new IllegalArgumentException("Unsupported order type: " + order.getPublicType());
        }
        orderInfoBuilder.setWorkflowState(order.getWorkflow().getState());
        orderInfoBuilder.setUserActionScheduled(order.isUserActionScheduled());

        for (OrderItem orderItem : order.getOrderItems()) {
            TOrderServiceInfo.Builder serviceInfoBuilder = orderInfoBuilder.addServiceBuilder()
                    .setServiceId(orderItem.getId().toString())
                    .setServiceType(orderItem.getPublicType())
                    .setCreatedAt(ProtoUtils.fromInstant(orderItem.getCreatedAt()))
                    .setUpdatedAt(ProtoUtils.fromInstant(orderItem.getUpdatedAt()));

            if (orderItem.getExpiresAt() != null) {
                serviceInfoBuilder.setExpiresAt(ProtoUtils.fromInstant(orderItem.getExpiresAt()));
            }
            if (orderItem.getConfirmedAt() != null) {
                serviceInfoBuilder.setConfirmedAt(ProtoUtils.fromInstant(orderItem.getConfirmedAt()));
            }

            TServiceInfo.Builder serviceInfoBuilder2 = serviceInfoBuilder.getServiceInfoBuilder()
                    .setServiceId(orderItem.getId().toString());
            switch (orderItem.getPublicType()) {
                case PT_EXPEDIA_HOTEL:
                    serviceInfoBuilder2.setExpediaItemState((EExpediaItemState) orderItem.getItemState());
                    break;
                case PT_DOLPHIN_HOTEL:
                    serviceInfoBuilder2.setDolphinItemState((EDolphinItemState) orderItem.getItemState());
                    break;
                case PT_TRAVELLINE_HOTEL:
                    serviceInfoBuilder2.setTravellineItemState((ETravellineItemState) orderItem.getItemState());
                    break;
                case PT_BNOVO_HOTEL:
                    serviceInfoBuilder2.setBNovoItemState((EBNovoItemState) orderItem.getItemState());
                    break;
                case PT_BRONEVIK_HOTEL:
                    serviceInfoBuilder2.setBronevikItemState((EBronevikItemState) orderItem.getItemState());
                    break;
                case PT_FLIGHT:
                    serviceInfoBuilder2.setAeroflotItemState((EAeroflotItemState) orderItem.getItemState());
                    break;
                case PT_TRAIN:
                case PT_BUS:
                case PT_SUBURBAN:
                    serviceInfoBuilder2.setGenericOrderItemState((EOrderItemState) orderItem.getItemState());
                    break;
                default:
                    throw new UnsupportedOperationException("Unknown order item type: " + orderItem.getPublicType());
            }

            var payload = updatedOrderItemsPayloads.get(orderItem.getId());
            if (payload != null) {
                serviceInfoBuilder2.setPayload(ProtoUtils.toTJson(payload));
            }

            TPaymentBreakdown.Builder paymentBreakdownBuilder = serviceInfoBuilder2.getPaymentBreakdownBuilder();

            for (FiscalItem fiscalItem : orderItem.getFiscalItems()) {
                TPaymentBreakdown.TFiscalItem.Builder fiscalItemBuilder =
                        paymentBreakdownBuilder.addFiscalItemsBuilder();
                if (fiscalItem.getType() != null) {
                    //TODO (mbobrov): maybe use proto enum for that
                    fiscalItemBuilder.setType(fiscalItem.getType().getValue());
                }
                fiscalItemBuilder.setTitle(Strings.nullToEmpty(fiscalItem.getTitle()));
                if (fiscalItem.getMoneyAmount() != null) {
                    fiscalItemBuilder.setPrice(ProtoUtils.toTPrice(fiscalItem.getMoneyAmount()));
                }
                if (fiscalItem.getVatType() != null) {
                    fiscalItemBuilder.setVat(OrderProtoUtils.toEVat(fiscalItem.getVatType()));
                }
            }

            Workflow workflow = orderItem.getWorkflow();
            serviceInfoBuilder.setWorkflowId(workflow.getId().toString());
            serviceInfoBuilder.setWorkflowType(workflow.getEntityType());
        }

        for (Invoice invoice : order.getInvoices()) {
            orderInfoBuilder.addInvoice(createInvoiceBuilder(invoice));
        }

        if (order.getCurrentInvoice() != null) {
            orderInfoBuilder.setCurrentInvoice(createInvoiceBuilder(order.getCurrentInvoice()));
        }
        // TODO(ganintsev): разобраться в рамках TRAINS-6367 нужно ли это поле
        for (OrderRefund refund : order.getOrderRefunds()) {
            orderInfoBuilder.addAllRefund(createTrainOrderRefunds(refund));
        }

        var priceInfoBuilder = TOrderPriceInfo.newBuilder();
        priceInfoBuilder.setPrice(ProtoUtils.toTPrice(order.calculateTotalCost()));
        priceInfoBuilder.setOriginalPrice(ProtoUtils.toTPrice(order.calculateOriginalTotalCost()));
        priceInfoBuilder.setDiscountAmount(ProtoUtils.toTPrice(order.calculateDiscountAmount()));
        if (!order.getPromoCodeApplications().isEmpty()) {
            for (PromoCodeApplication pCode : order.getPromoCodeApplications()) {
                TPromoCodeApplicationResult.Builder builder = TPromoCodeApplicationResult.newBuilder()
                        .setCode(pCode.getPromoCodeActivation().getPromoCode().getCode());
                if (pCode.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS) {
                    builder.setDiscountAmount(ProtoUtils.toTPrice(pCode.getDiscount()));
                }
                builder.setType(pCode.getApplicationResultType());
                priceInfoBuilder.addPromoCodeApplicationResults(builder);
            }
        }

        if (order.getGeneratedPromoCodes() != null && order.getGeneratedPromoCodes().getPromoCodes().size() > 0) {
            var generatedPromoCodesInfoBuilder = TGeneratedPromoCodesInfo.newBuilder();
            var promoCode = order.getGeneratedPromoCodes().getPromoCodes().get(0);
            generatedPromoCodesInfoBuilder.setId(promoCode.getId().toString());
            generatedPromoCodesInfoBuilder.setCode(promoCode.getCode());
            generatedPromoCodesInfoBuilder.setValidFrom(ProtoUtils.fromInstant(PromoCodeHelpers.getPromoCodeValidFrom(promoCode)));
            generatedPromoCodesInfoBuilder.setValidTill(ProtoUtils.fromInstant(PromoCodeHelpers.getPromoCodeValidTill(promoCode)));
            generatedPromoCodesInfoBuilder.setDiscountAmount(
                    ProtoUtils.toTPrice(Money.of(promoCode.getNominal(), ProtoCurrencyUnit.RUB)));
            if (promoCode.getPromoAction().getDiscountApplicationConfig() != null &&
                    promoCode.getPromoAction().getDiscountApplicationConfig().getMinTotalCost() != null) {
                generatedPromoCodesInfoBuilder.setMinTotalCost(ProtoUtils.toTPrice(promoCode.getPromoAction().getDiscountApplicationConfig().getMinTotalCost()));
            }

            orderInfoBuilder.setGeneratedPromoCodes(generatedPromoCodesInfoBuilder);
        }

        if (addPromoCampaignsInfo) {
            PromoCampaigns promoCampaigns = promoCampaignsService.getOrderPromoCampaignsParticipationInfo(order);
            var builder = TPromoCampaignsInfo.newBuilder();
            if (promoCampaigns.getTaxi2020() != null) {
                builder.setTaxi2020(TTaxi2020PromoCampaignInfo.newBuilder()
                        .setStatus(promoCampaigns.getTaxi2020().getStatus() == Taxi2020PromoOrderStatus.NOT_ELIGIBLE ?
                                ETaxi2020PromoStatusEnum.OTPS_NOT_ELIGIBLE :
                                ETaxi2020PromoStatusEnum.OTPS_ELIGIBLE)
                        .build());
            }
            if (promoCampaigns.getMir2020() != null &&
                    promoCampaigns.getMir2020().getEligibility() == MirPromoOrderEligibility.ELIGIBLE &&
                    promoCampaigns.getMir2020().getPaidWithMir() != null && promoCampaigns.getMir2020().getPaidWithMir()) {
                // here we output as eligible==true only fully eligible offers (i.e. the ones paid with mir)
                builder.setMir2020(TMir2020PromoCampaignInfo.newBuilder()
                        .setEligible(true)
                        .setCashbackAmount(UInt32Value.newBuilder().setValue(promoCampaigns.getMir2020().getCashbackAmount()).build())
                        .build());
            } else {
                builder.setMir2020(TMir2020PromoCampaignInfo.newBuilder()
                        .setEligible(false).build());
            }
            WhiteLabelPromoCampaign whiteLabel = promoCampaigns.getWhiteLabel();
            if (whiteLabel != null) {
                TWhiteLabelPromoCampaignInfo.Builder whiteLabelBuilder = TWhiteLabelPromoCampaignInfo.newBuilder();
                boolean eligible = whiteLabel.getEligible() == EWhiteLabelEligibility.WLE_ELIGIBLE;
                whiteLabelBuilder.setEligible(eligible);
                if (eligible) {
                    whiteLabelBuilder.setPoints(TWhiteLabelPoints.newBuilder()
                            .setAmount(whiteLabel.getPoints().getAmount())
                            .setPointsType(whiteLabel.getPoints().getPointsType())
                            .build());
                    whiteLabelBuilder.setPointsLinguistics(TWhiteLabelPointsLinguistics.newBuilder()
                            .setNameForNumeralNominative(whiteLabel.getPointsLinguistics().getNameForNumeralNominative())
                            .build());
                }
                builder.setWhiteLabel(whiteLabelBuilder.build());
            }
            priceInfoBuilder.setPromoCampaignsInfo(builder.build());
        }

        orderInfoBuilder.setPriceInfo(priceInfoBuilder);

        orderInfoBuilder.setUsesDeferredPayment(order.getUseDeferredPayment());
        orderInfoBuilder.setZeroFirstPayment(order.getUseDeferredPayment() && order.getCurrentInvoice() == null);
        orderInfoBuilder.addAllPayments(order.getPayments().stream().map(this::buildPayment).collect(Collectors.toList()));

        Workflow workflow = order.getWorkflow();
        orderInfoBuilder.setWorkflowId(workflow.getId().toString());
        orderInfoBuilder.setWorkflowType(workflow.getEntityType());

        return orderInfoBuilder;
    }


    public TOrderAggregateState.Builder getOrderAggregateStateBuilder(OrderAggregateState orderAggregateState) {
        var builder = TOrderAggregateState.newBuilder();
        builder.setOrderId(orderAggregateState.getId().toString())
                .setOrderPrettyId(orderAggregateState.getOrderPrettyId())
                .setHotelOrderAggregateState(orderAggregateState.getHotelOrderAggregateState())
                .setTrainOrderAggregateState(orderAggregateState.getTrainOrderAggregateState())
                .setPaymentUrl(Strings.nullToEmpty(orderAggregateState.getPaymentUrl()))
                .setPaymentError(orderAggregateState.getPaymentErrorDisplayState());

        if (orderAggregateState.getOrderDisplayState() != null) {
            builder.setDisplayOrderState(orderAggregateState.getOrderDisplayState());
        }
        if (orderAggregateState.getGenericOrderAggregateState() != null) {
            builder.setGenericOrderAggregateState(orderAggregateState.getGenericOrderAggregateState());
        }
        if (orderAggregateState.getVersionHash() != null) {
            builder.setVersionHash(orderAggregateState.getVersionHash());
        }
        if (orderAggregateState.getReservedTo() != null) {
            builder.setReservedTo(ProtoUtils.fromInstant(orderAggregateState.getReservedTo()));
        }

        var trainAggregateStateInfoBuilder = TTrainOrderAggregateStateExtInfo.newBuilder();
        if (orderAggregateState.getTrainOrderInsuranceAggregateState() != null) {
            trainAggregateStateInfoBuilder.setInsuranceState(orderAggregateState.getTrainOrderInsuranceAggregateState());
        }
        if (orderAggregateState.getMaxPendingTill() != null) {
            trainAggregateStateInfoBuilder.setMaxPendingTill(ProtoUtils.fromInstant(orderAggregateState.getMaxPendingTill()));
        }

        if (orderAggregateState.isHasTrainError()) {
            TTrainOrderAggregateErrorInfo.Builder errorInfoBuilder =
                    TTrainOrderAggregateErrorInfo.newBuilder()
                            .setErrorType(orderAggregateState.getTrainErrorType())
                            .setMessage(orderAggregateState.getTrainErrorMessage())
                            .setMessageCode(orderAggregateState.getTrainErrorMessageCode());
            if (orderAggregateState.getTrainImErrorInfo() != null &&
                    orderAggregateState.getTrainImErrorInfo().getMessageParams() != null) {
                errorInfoBuilder.addAllMessageParam(orderAggregateState.getTrainImErrorInfo().getMessageParams());
            }
            trainAggregateStateInfoBuilder.setErrorInfo(errorInfoBuilder);
        }
        builder.setTrainAggregateExtInfo(trainAggregateStateInfoBuilder);
        return builder;
    }

    private void addOrderAggregateState(AggregateStateConstructMode mode, Order order,
                                        TOrderInfo.Builder orderInfoBuilder) {
        if (mode == AggregateStateConstructMode.BYPASS) {
            return;
        }

        Optional<OrderAggregateState> maybeAggregateState = orderAggregateStateRepository.findById(order.getId());

        switch (mode) {
            case CALCULATE_ON_THE_FLY:
                TOrderAggregateState aggregateState = orderAggregateStateMapper.mapAggregateStateFromOrder(order);
                orderInfoBuilder.setOrderAggregateState(aggregateState);
                break;
            case GET_FROM_DB:
                maybeAggregateState.ifPresent(agState -> orderInfoBuilder.setOrderAggregateState(getOrderAggregateStateBuilder(agState)));
                break;
            default:
                throw new IllegalArgumentException("Unknown mode: " + mode);
        }
    }

    public List<TOrderRefund> createBusOrderRefunds(OrderRefund orderRefund) {
        GenericOrderUserRefund refund = (GenericOrderUserRefund) orderRefund;
        List<BusTicketRefund> ticketRefunds = refund.getBusTicketRefunds().stream()
                .filter(x -> x.getState() == EBusTicketRefundState.RS_REFUNDED)
                .collect(Collectors.toList());
        if (ticketRefunds.size() == 0) {
            return List.of(createOrderRefund(refund, null));
        }
        return ticketRefunds.stream().map(x -> createOrderRefund(refund, x.getPayload())).collect(Collectors.toList());
    }

    public List<TOrderRefund> createTrainOrderRefunds(OrderRefund refund) {
        List<TrainTicketRefund> ticketRefunds = OrderUtils.getSuccessfulTicketRefunds(refund);
        if (ticketRefunds.size() == 0) {
            return List.of(createOrderRefund(refund, null));
        }
        return ticketRefunds.stream().map(x -> createOrderRefund(refund, x.getPayload())).collect(Collectors.toList());
    }

    public TOrderRefund createOrderRefund(OrderRefund refund, Object payload) {
        TOrderRefund.Builder refundBuilder = TOrderRefund.newBuilder()
                .setId(ProtoUtils.toStringOrEmpty(refund.getId()))
                .setCreatedAt(ProtoUtils.fromInstant(refund.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(refund.getUpdatedAt()))
                .setRefundType(refund.getRefundType())
                .setState(refund.getState());
        if (payload != null) {
            refundBuilder.setPayload(ProtoUtils.toTJson(payload));
        }
        return refundBuilder.build();
    }

    public TOrderInvoiceInfo.Builder createInvoiceBuilder(Invoice invoice) {
        TOrderInvoiceInfo.Builder invoiceInfoBuilder = TOrderInvoiceInfo.newBuilder();
        invoiceInfoBuilder.setInvoiceId(invoice.getId().toString()).setInvoiceType(invoice.getPublicType());

        switch (invoice.getPublicType()) {
            case IT_TRUST:
                invoiceInfoBuilder.setTrustInvoiceState((ETrustInvoiceState) invoice.getInvoiceState());
                break;
            case IT_AVIA_AEROFLOT:
                invoiceInfoBuilder.setAeroflotInvoiceState((EAeroflotInvoiceState) invoice.getInvoiceState());
                break;
        }
        invoiceInfoBuilder.setPurchaseToken(Strings.nullToEmpty(invoice.getPurchaseToken()));
        invoiceInfoBuilder.setPaymentUrl(Strings.nullToEmpty(invoice.getPaymentUrl()));
        invoiceInfoBuilder.setAuthorizationErrorCode(Strings.nullToEmpty(invoice.getAuthorizationErrorCode()));

        if (invoice.getFiscalReceipts() != null) {
            for (FiscalReceipt receipt : invoice.getFiscalReceipts()) {
                if (!Strings.isNullOrEmpty(receipt.getReceiptUrl())) {
                    invoiceInfoBuilder.addFiscalReceiptBuilder()
                            .setType(receipt.getReceiptType().getProtoValue())
                            .setOrderRefundId(ProtoUtils.toStringOrEmpty(receipt.getOrderRefundId()))
                            .setUrl(receipt.getReceiptUrl());
                }
            }
        }

        Workflow workflow = invoice.getWorkflow();
        invoiceInfoBuilder.setWorkflowId(workflow.getId().toString());
        invoiceInfoBuilder.setWorkflowType(workflow.getEntityType());

        // provider-specific data
        if (invoice instanceof AeroflotInvoice) {
            AeroflotInvoice aeroflotInvoice = (AeroflotInvoice) invoice;
            invoiceInfoBuilder.setConfirmationUrl(Strings.nullToEmpty(aeroflotInvoice.getConfirmationUrl()));
        }
        return invoiceInfoBuilder;
    }

    public TAdminOrderInfo.Builder buildAdminOrderInfoFor(Order order,
                                                          Map<UUID, List<TrustRefund>> refundMap,
                                                          AuthorizedUser owner,
                                                          List<ESnippet> snippets,
                                                          AdminUserCapabilitiesManager adminUserCapabilities,
                                                          TAdminActionToken adminActionToken,
                                                          boolean needReceiptData) {
        TAdminOrderInfo.Builder orderInfoBuilder = TAdminOrderInfo.newBuilder()
                .setOrderId(order.getId().toString())
                .setPrettyId(order.getPrettyId())
                .setOrderType(order.getPublicType())
                .setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()))
                .setDisplayOrderType(order.getDisplayType())
                .setDisplayOrderState(EDisplayOrderStateMapper.fromOrder(order))
                .setAdminActionToken(tokenEncrypter.toAdminActionToken(adminActionToken))
                .setIsPostPayEligible(order.isPostPayEligible())
                .setIsPostPaid(order.isFullyPostPaid());

        if (order.getExpiresAt() != null) {
            orderInfoBuilder.setExpiresAt(ProtoUtils.fromInstant(order.getExpiresAt()));
        }

        orderInfoBuilder.setLabel(Strings.nullToEmpty(order.getLabel()));

        if (snippets.contains(ESnippet.S_PRIVATE_INFO)) {
            orderInfoBuilder.setOwner(buildUserInfo(order, owner));
            orderInfoBuilder.setDocumentUrl(Strings.nullToEmpty(order.getDocumentUrl()));
        }

        var flagsBuilder = TAdminOrderFlags.newBuilder();
        flagsBuilder.setHasMaskedInfo(!snippets.contains(ESnippet.S_PRIVATE_INFO));
        flagsBuilder.setCanRefundAllMoney(checkMoneyRefundsService.checkAllMoneyRefundAllowed(order));
        flagsBuilder.setCanRefundByFiscalItem(checkMoneyRefundsService.checkByFiscalItemManualRefundAllowed(order));

        switch (order.getPublicType()) {
            case OT_HOTEL_EXPEDIA:
                EHotelOrderState hotelState = (EHotelOrderState) order.getEntityState();
                orderInfoBuilder.setHotelOrderState(hotelState);
                if (hotelState == EHotelOrderState.OS_CONFIRMED) {
                    flagsBuilder.setCanSendSuccessfulMail(true);
                } else if (hotelState == EHotelOrderState.OS_REFUNDED) {
                    flagsBuilder.setCanSendRefundMail(true);
                }
                List<OrderItem> orderItems = order.getOrderItems();
                flagsBuilder.setCanRestoreDolphinOrder(hotelState == EHotelOrderState.OS_MANUAL_PROCESSING
                        && orderItems.size() == 1
                        && orderItems.get(0).getItemState() == EDolphinItemState.IS_MANUAL_CONFIRMATION);
                flagsBuilder.setCanRegenerateVouchers(hotelState == EHotelOrderState.OS_CONFIRMED);
                boolean isRefundAvailable =
                        (hotelState == EHotelOrderState.OS_CONFIRMED || hotelState == EHotelOrderState.OS_CANCELLED
                                || hotelState == EHotelOrderState.OS_REFUNDED)
                                && checkMoneyRefundsService.checkByFiscalItemManualHotelRefundAllowed(order);
                flagsBuilder.setCanRefundHotelOrder(adminUserCapabilities.canDoAction(EAdminAction.AA_REFUND_HOTEL_ORDER, order.getDisplayType())
                        && isRefundAvailable);

                if (order.calculateDiscountAmount().isZero()) {
                    flagsBuilder.setCanAmountRefundHotelOrder(flagsBuilder.getCanRefundHotelOrder());
                } else {
                    flagsBuilder.setCanAmountRefundHotelOrder(isRefundAvailable
                            && adminUserCapabilities.canDoAction(EAdminAction.AA_PARTIAL_HOTEL_REFUND, order.getDisplayType()));
                }

                flagsBuilder.setCanOnlyFullRefundHotelOrder(adminUserCapabilities.canDoAction(EAdminAction.AA_ONLY_FULL_REFUND_HOTEL_ORDER, order.getDisplayType())
                        && isRefundAvailable);
                flagsBuilder.setCanRefundHotelMoneyOnly(adminUserCapabilities.canDoAction(EAdminAction.AA_MANUAL_REFUND_MONEY_ONLY, order.getDisplayType())
                        && isRefundAvailable);
                flagsBuilder.setCanModifyHotelOrderDetails(adminUserCapabilities.canDoAction(EAdminAction.AA_MODIFY_HOTEL_ORDER_DETAILS, order.getDisplayType()));
                break;
            case OT_AVIA_AEROFLOT:
                orderInfoBuilder.setAeroflotOrderState((EAeroflotOrderState) order.getEntityState());
                break;
            case OT_TRAIN:
                ETrainOrderState trainState = (ETrainOrderState) order.getEntityState();
                orderInfoBuilder.setTrainOrderState(trainState);
                flagsBuilder.setCanSendSuccessfulMail(trainState == ETrainOrderState.OS_CONFIRMED);
                flagsBuilder.setCanRefundYandexFee(checkMoneyRefundsService.checkFeeRefundAllowed(order));
                flagsBuilder.setCanSendRefundMail(order.getOrderRefunds().stream()
                        .anyMatch(refund ->
                                refund.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND &&
                                        refund.getState() == EOrderRefundState.RS_REFUNDED));
                break;
            case OT_GENERIC:
                //noinspection SwitchStatementWithTooFewBranches
                switch (order.getDisplayType()) {
                    case DT_TRAIN:
                        EOrderState genericTrainState = ((GenericOrder) order).getEntityState();
                        orderInfoBuilder.setGenericOrderState(genericTrainState);
                        flagsBuilder.setCanSendSuccessfulMail(genericTrainState == EOrderState.OS_CONFIRMED);
                        flagsBuilder.setCanRefundYandexFee(checkMoneyRefundsService.checkFeeRefundAllowed(order));
                        flagsBuilder.setCanSendRefundMail(order.getOrderRefunds().stream()
                                .anyMatch(refund ->
                                        // todo(tlg-13,ganintsev): TRAVELBACK-1828: migrate old train refunds
                                        // todo(maxim-sazonov): combine cases for train and bus after ^
                                        (refund.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND
                                                || refund.getRefundType() == EOrderRefundType.RT_GENERIC_USER_REFUND
                                        ) && refund.getState() == EOrderRefundState.RS_REFUNDED));
                        break;
                    case DT_BUS:
                        EOrderState genericBusState = ((GenericOrder) order).getEntityState();
                        orderInfoBuilder.setGenericOrderState(genericBusState);
                        flagsBuilder.setCanRegenerateVouchers(genericBusState == EOrderState.OS_CONFIRMED);
                        flagsBuilder.setCanSendSuccessfulMail(genericBusState == EOrderState.OS_CONFIRMED);
                        flagsBuilder.setCanRefundYandexFee(checkMoneyRefundsService.checkFeeRefundAllowed(order));
                        flagsBuilder.setCanSendRefundMail(order.getOrderRefunds().stream()
                                .anyMatch(refund ->
                                        refund.getRefundType() == EOrderRefundType.RT_GENERIC_USER_REFUND &&
                                                refund.getState() == EOrderRefundState.RS_REFUNDED));
                        break;
                    default:
                        break;
                }
                var invoice = order.getCurrentInvoice();
                if (invoice != null) {
                    flagsBuilder.setCanRetryMoneyRefund(invoice.getWorkflow().getState() == EWorkflowState.WS_RUNNING &&
                            invoice.getTrustRefunds().size() > 0);
                }
                break;
            default:
                break;
        }
        orderInfoBuilder.setOrderFlags(flagsBuilder);

        orderInfoBuilder.setWorkflowState(order.getWorkflow().getState());
        orderInfoBuilder.setUserActionScheduled(order.isUserActionScheduled());
        orderInfoBuilder.setBroken(order.isBroken());

        List<TAdminOrderInvoiceInfo.Builder> invoices =
                order.getInvoices().stream().map(invoice -> buildAdminOrderInvoiceInfo(
                        invoice,
                        order.getCurrency(),
                        refundMap.get(invoice.getId()),
                        needReceiptData
                )).collect(Collectors.toList());

        Money currentTotalPaymentsSum = invoices.stream()
                .map(TAdminOrderInvoiceInfo.Builder::build)
                .filter((invoice) -> invoice.getTrustInvoiceState() != ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)
                .map((invoice) -> invoice.getInvoiceItemList().stream()
                        .map(invoiceItem -> ProtoUtils.fromTPrice(invoiceItem.getMoneyAmount()))
                        .reduce(Money::add)
                        .orElse(Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB)))
                .reduce(Money::add)
                .orElse(Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB));

        TAdminOrderPriceInfo.Builder priceInfoBuilder = TAdminOrderPriceInfo.newBuilder();
        priceInfoBuilder.setOriginalPrice(ProtoUtils.toTPrice(order.calculateOriginalTotalCost()));
        priceInfoBuilder.setPrice(ProtoUtils.toTPrice(order.calculateTotalCost()));
        priceInfoBuilder.setDiscountAmount(ProtoUtils.toTPrice(order.calculateDiscountAmount()));
        priceInfoBuilder.setCurrentTotalPaymentsSum(ProtoUtils.toTPrice(currentTotalPaymentsSum));
        for (PromoCodeApplication pca : order.getPromoCodeApplications()) {
            TAdminPromoCodeApplicationResult.Builder r = TAdminPromoCodeApplicationResult.newBuilder();
            r.setCode(pca.getPromoCodeActivation().getPromoCode().getCode());
            r.setPromoCodeActivationId(pca.getPromoCodeActivation().getId().toString());
            r.setPromoCodeId(pca.getPromoCodeActivation().getPromoCode().getId().toString());
            r.setType(pca.getApplicationResultType());
            if (pca.getDiscount() != null) {
                r.setDiscountAmount(ProtoUtils.toTPrice(pca.getDiscount()));
            }
            priceInfoBuilder.addPromoCodeApplicationResults(r);
        }

        orderInfoBuilder.setOrderPriceInfo(priceInfoBuilder);

        if (order.getGeneratedPromoCodes() != null && order.getGeneratedPromoCodes().getPromoCodes().size() > 0) {
            var generatedPromoCodesInfoBuilder = TAdminGeneratedPromoCodesInfo.newBuilder();
            var promoCode = order.getGeneratedPromoCodes().getPromoCodes().get(0);
            generatedPromoCodesInfoBuilder.setId(promoCode.getId().toString());
            generatedPromoCodesInfoBuilder.setCode(promoCode.getCode());
            generatedPromoCodesInfoBuilder.setValidFrom(ProtoUtils.fromInstant(PromoCodeHelpers.getPromoCodeValidFrom(promoCode)));
            generatedPromoCodesInfoBuilder.setValidTill(ProtoUtils.fromInstant(PromoCodeHelpers.getPromoCodeValidTill(promoCode)));
            generatedPromoCodesInfoBuilder.setDiscountAmount(
                    ProtoUtils.toTPrice(Money.of(promoCode.getNominal(), ProtoCurrencyUnit.RUB)));
            if (promoCode.getPromoAction().getDiscountApplicationConfig() != null &&
                    promoCode.getPromoAction().getDiscountApplicationConfig().getMinTotalCost() != null) {
                generatedPromoCodesInfoBuilder.setMinTotalCost(ProtoUtils.toTPrice(promoCode.getPromoAction().getDiscountApplicationConfig().getMinTotalCost()));
            }
            if (promoCode.getPromoAction().getPromoCodeGenerationType() != null) {
                generatedPromoCodesInfoBuilder.setGenerationStrategy(promoCode.getPromoAction().getPromoCodeGenerationType().getValue());
            }
            generatedPromoCodesInfoBuilder.setActivationsStrategy(promoCode.getActivationsStrategy().getValue());
            if (promoCode.getActivationsStrategy() == PromoCodeActivationsStrategy.LIMITED_ACTIVATIONS) {
                var activationsLeft = promoCode.getAllowedActivationsTotal() - promoCode.getAllowedActivationsCount();
                generatedPromoCodesInfoBuilder.setAllowedActivationsLeft(activationsLeft);
            }

            orderInfoBuilder.setGeneratedPromoCodes(generatedPromoCodesInfoBuilder);
        }

        order.getOrderItems().forEach(orderItem -> orderInfoBuilder.addOrderItem(buildAdminOrderItemInfo(orderItem)));
        invoices.forEach(orderInfoBuilder::addInvoice);
        order.getPayments().forEach(p ->
                orderInfoBuilder.addPayments(buildAdminPayment(p, order.getCurrency(), refundMap)));
        order.getOrderRefunds().forEach(orderRefund -> orderInfoBuilder
                .addOrderRefund(buildAdminOrderRefundInfo(orderRefund)));

        Workflow workflow = order.getWorkflow();
        orderInfoBuilder.setWorkflowId(workflow.getId().toString());
        orderInfoBuilder.setWorkflowType(workflow.getEntityType());
        order.getOrderItems().stream()
                .map(ordersAdminEnumMapper::getPartnerByItem)
                .forEach(orderInfoBuilder::addPartners);

        return orderInfoBuilder;
    }

    private TAdminOrderRefundInfo.Builder buildAdminOrderRefundInfo(OrderRefund orderRefund) {
        var orderRefundInfoBuilder = TAdminOrderRefundInfo.newBuilder()
                .setId(orderRefund.getId().toString())
                .setState(orderRefund.getState().toString())
                .setType(orderRefund.getRefundType().toString())
                .setCreatedAt(ProtoUtils.fromInstant(orderRefund.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(orderRefund.getUpdatedAt()));
        Money refundedAmount = orderRefund.calculateRefundedAmount();
        if (refundedAmount != null) {
            orderRefundInfoBuilder.setRefundedAmount(ProtoUtils.toTPrice(refundedAmount));
        }
        return orderRefundInfoBuilder;
    }

    public TUserInfo.Builder buildUserInfo(Order order, AuthorizedUser owner) {
        var ownerBuilder = buildUserInfoFromOrder(order);
        if (owner != null) {
            ownerBuilder.setLogin(Strings.nullToEmpty(owner.getLogin()));
            ownerBuilder.setYandexUid(Strings.nullToEmpty(owner.getYandexUid()));
            ownerBuilder.setPassportId(Strings.nullToEmpty(owner.getPassportId()));
        }
        return ownerBuilder;
    }

    public TUserInfo.Builder buildUserInfoFromOrder(Order order) {
        var ownerBuilder = TUserInfo.newBuilder();
        ownerBuilder.setEmail(Strings.nullToEmpty(order.getEmail()));
        ownerBuilder.setPhone(Strings.nullToEmpty(order.getPhone()));
        ownerBuilder.setIp(Strings.nullToEmpty(order.getIp()));
        return ownerBuilder;
    }

    public TOrderPromoCampaignsInfo buildPromoCampaignsFor(PromoCampaigns promoCampaigns, Order order) {
        TOrderPromoCampaignsInfo.Builder builder = TOrderPromoCampaignsInfo.newBuilder();
        if (promoCampaigns == null) {
            return builder.build();
        }
        if (promoCampaigns.getTaxi2020() != null) {
            Taxi2020PromoOrder taxi2020Promo = promoCampaigns.getTaxi2020();
            builder.setTaxi2020(TOrderTaxi2020PromoCampaignInfo.newBuilder()
                    .setStatus(TAXI_2020_PROMO_STATUS_MAPPING.getOrDefault(
                            taxi2020Promo.getStatus(), TAXI_2020_PROMO_DEFAULT_STATUS))
                    .setEmail(Strings.nullToEmpty(taxi2020Promo.getEmail()))
                    .setEmailScheduledAt(ProtoUtils.fromInstantSafe(taxi2020Promo.getEmailScheduledAt()))
                    .setEmailSentAt(ProtoUtils.fromInstantSafe(taxi2020Promo.getSentAt()))
                    .build());
        }
        if (promoCampaigns.getMir2020() != null) {
            Mir2020PromoOrder mir2020PromoOrder = promoCampaigns.getMir2020();
            var mirBuilder = ru.yandex.travel.orders.admin.proto.TMir2020PromoCampaignInfo.newBuilder()
                    .setOfferEligibility(MIR_2020_PROMO_ELIGIBILITY_MAPPING.getOrDefault(
                            mir2020PromoOrder.getEligibility(), EMir2020PromoEligibility.MIR_PROMO_DISABLED));
            if (mir2020PromoOrder.getPaidWithMir() != null) {
                mirBuilder.setPaidWithMir(mir2020PromoOrder.getPaidWithMir());
            }
            if (mir2020PromoOrder.getCashbackAmount() != null) {
                mirBuilder.setCashbackAmount(UInt32Value.newBuilder().setValue(mir2020PromoOrder.getCashbackAmount()).build());
            }
            builder.setMir2020(mirBuilder.build());
        } else {
            builder.setMir2020(ru.yandex.travel.orders.admin.proto.TMir2020PromoCampaignInfo.newBuilder()
                    .setOfferEligibility(EMir2020PromoEligibility.MIR_PROMO_DISABLED)
                    .build());
        }
        if (order instanceof HotelOrder) {
            TYandexPlusInfo yandexPlusInfoForOrder = plusPromoService.getYandexPlusInfoForOrder((HotelOrder) order);
            if (yandexPlusInfoForOrder != null) {
                builder.setYandexPlus(yandexPlusInfoForOrder);
            }
        }
        return builder.build();
    }

    public TPaymentInfo buildPayment(Payment payment) {
        var paymentBuilder = TPaymentInfo.newBuilder()
                .setId(payment.getId().toString())
                .setTotalAmount(ProtoUtils.toTPrice(payment.getTotalAmount()))
                .setTotalAmountMarkup(InvoiceUtils.toProtoPaymentMarkup(payment.getTotalAmountMarkup()))
                .setPaidAmount(ProtoUtils.toTPrice(payment.getPaidAmount()))
                .setState(payment.getState())
                .setType(payment.getPaymentType())
                .addAllNextPayments(payment.getNextPayments().stream().map(this::buildPayment).collect(Collectors.toList()));

        if (payment.getPaymentEndsAt() != null) {
            paymentBuilder.setPaymentEndsAt(ProtoUtils.fromInstant(payment.getPaymentEndsAt()));
        }
        if (payment.getClosedAt() != null) {
            paymentBuilder.setClosedAt(ProtoUtils.fromInstant(payment.getClosedAt()));
        }
        Invoice lastAttempt = payment.getLastAttempt();
        if (lastAttempt != null) {
            if (lastAttempt.getPaymentUrl() != null &&
                    lastAttempt.getInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT) {
                paymentBuilder.setActivePaymentUrl(lastAttempt.getPaymentUrl());
            }
            if (lastAttempt.getAuthorizationErrorCode() != null &&
                    lastAttempt.getInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                paymentBuilder.setActivePaymentErrorCode(lastAttempt.getAuthorizationErrorCode());
            }
        }
        paymentBuilder.addAllFiscalReceipts(payment.getReceipts().stream()
                .filter(r -> r.getReceiptUrl() != null)
                .map(r -> TFiscalReceipt.newBuilder()
                        .setType(r.getReceiptType().getProtoValue())
                        .setOrderRefundId(ProtoUtils.toStringOrEmpty(r.getOrderRefundId()))
                        .setUrl(r.getReceiptUrl())
                        .build())
                .collect(Collectors.toList()));
        return paymentBuilder.build();
    }

    public enum AggregateStateConstructMode {
        BYPASS, GET_FROM_DB, CALCULATE_ON_THE_FLY
    }
}
