package ru.yandex.travel.orders.services;

import java.util.Map;
import java.util.NoSuchElementException;

import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.bus.model.BusPartner;
import ru.yandex.travel.orders.admin.proto.EPartnerType;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.services.buses.BusPartnersService;
import ru.yandex.travel.suburban.partners.SuburbanProvider;

import static ru.yandex.travel.commons.map.MappingUtils.getMappingOrThrow;

@Slf4j
@Service
@RequiredArgsConstructor
public class OrdersAdminEnumMapper {
    private final BusPartnersService busPartnersService;

    private final static ImmutableBiMap<EDisplayOrderType, EOrderType> MAP_DISPLAY_ORDER_TYPE=
            new ImmutableBiMap.Builder<EDisplayOrderType, EOrderType>()
                    .put(EDisplayOrderType.DT_AVIA, EOrderType.OT_AVIA_AEROFLOT)
                    .put(EDisplayOrderType.DT_BUS, EOrderType.OT_BUS)
                    .put(EDisplayOrderType.DT_HOTEL, EOrderType.OT_HOTEL_EXPEDIA)
                    .put(EDisplayOrderType.DT_TRAIN, EOrderType.OT_TRAIN)
                    .put(EDisplayOrderType.DT_SUBURBAN, EOrderType.OT_GENERIC)
                    .build();

    private static final Map<EServiceType, EPartnerType> MAP_SERVICE_TO_PARTNER =
            new ImmutableMap.Builder<EServiceType, EPartnerType>()
                    .put(EServiceType.PT_EXPEDIA_HOTEL, EPartnerType.PT_HOTEL_EXPEDIA)
                    .put(EServiceType.PT_DOLPHIN_HOTEL, EPartnerType.PT_HOTEL_DOLPHIN)
                    .put(EServiceType.PT_TRAVELLINE_HOTEL, EPartnerType.PT_HOTEL_TRAVELLINE)
                    .put(EServiceType.PT_BNOVO_HOTEL, EPartnerType.PT_HOTEL_BNOVO)
                    .put(EServiceType.PT_BRONEVIK_HOTEL, EPartnerType.PT_HOTEL_BRONEVIK)
                    .put(EServiceType.PT_FLIGHT, EPartnerType.PT_AVIA_AEROFLOT)
                    .put(EServiceType.PT_TRAIN, EPartnerType.PT_TRAIN_RZHD)
                    .build();

    private final static ImmutableMap<SuburbanProvider, EPartnerType> MAP_SUBURBAN_PROVIDER_TO_PARTNER =
            new ImmutableMap.Builder<SuburbanProvider, EPartnerType>()
                    .put(SuburbanProvider.MOVISTA, EPartnerType.PT_MOVISTA)
                    .put(SuburbanProvider.IM, EPartnerType.PT_IM_SUBURBAN)
                    .put(SuburbanProvider.AEROEXPRESS, EPartnerType.PT_AEROEXPRESS)
                    .build();

    private final static ImmutableMap<BusPartner, EPartnerType> MAP_BUS_PARTNER_TO_PARTNER =
            new ImmutableMap.Builder<BusPartner, EPartnerType>()
                    .put(BusPartner.BUSFOR, EPartnerType.PT_BUS_BUSFOR)
                    .put(BusPartner.ECOLINES, EPartnerType.PT_BUS_ECOLINES)
                    .put(BusPartner.ETRAFFIC, EPartnerType.PT_BUS_ETRAFFIC)
                    .put(BusPartner.NOY, EPartnerType.PT_BUS_NOY)
                    .put(BusPartner.OK, EPartnerType.PT_BUS_OK)
                    .put(BusPartner.RUSET, EPartnerType.PT_BUS_RUSET)
                    .put(BusPartner.SKS, EPartnerType.PT_BUS_SKS)
                    .put(BusPartner.UNITIKI_NEW, EPartnerType.PT_BUS_UNITIKI)
                    .put(BusPartner.YUGAVTOTRANS, EPartnerType.PT_BUS_YUGAVTOTRANS)
                    .build();

    public EOrderType getOrderTypeFromDisplayOrderType(EDisplayOrderType displayOrderType){
        return getMappingOrThrow(MAP_DISPLAY_ORDER_TYPE, displayOrderType);
    }

    public EDisplayOrderType getDisplayOrderTypeFromOrderType(EOrderType orderType){
        return getMappingOrThrow(MAP_DISPLAY_ORDER_TYPE.inverse(), orderType);
    }

    public EPartnerType getPartnerByItem(OrderItem orderItem) {
        try {
            switch (orderItem.getPublicType()) {
                case PT_SUBURBAN:
                    return getMappingOrThrow(MAP_SUBURBAN_PROVIDER_TO_PARTNER,
                            ((SuburbanOrderItem) orderItem).getPayload().getProviderCode());
                case PT_BUS:
                    return getMappingOrThrow(MAP_BUS_PARTNER_TO_PARTNER,
                            busPartnersService.getPartnerById(
                                    ((BusOrderItem) orderItem).getReservation().getRide().getSupplierId()));
                default:
                    return getMappingOrThrow(MAP_SERVICE_TO_PARTNER, orderItem.getPublicType());
            }
        } catch (NoSuchElementException ignored) {
            log.warn("Unknown partner mapping for order item: {}", orderItem);
            return EPartnerType.PT_UNKNOWN;
        }
    }
}
