package ru.yandex.travel.orders.services;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Strings;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.admin.proto.EPartnerType;
import ru.yandex.travel.orders.admin.proto.TAdminListAviaInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListBusInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListHotelInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListItem;
import ru.yandex.travel.orders.admin.proto.TAdminListItemInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListTrainInfo;
import ru.yandex.travel.orders.admin.proto.TEntityStateTransition;
import ru.yandex.travel.orders.admin.proto.TEventInfo;
import ru.yandex.travel.orders.admin.proto.TGetWorkflowRsp;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
public class OrdersAdminMapper {
    private final WorkflowRepository workflowRepository;
    private final WorkflowEventRepository workflowEventRepository;
    private final JsonFormat.Printer messagePrinter;
    private final OrdersAdminEnumMapper ordersAdminEnumMapper;

    public OrdersAdminMapper(WorkflowRepository workflowRepository,
                             WorkflowEventRepository workflowEventRepository,
                             OrdersAdminEnumMapper ordersAdminEnumMapper) {
        this.workflowRepository = workflowRepository;
        this.workflowEventRepository = workflowEventRepository;
        this.messagePrinter = JsonFormat.printer().omittingInsignificantWhitespace();
        this.ordersAdminEnumMapper = ordersAdminEnumMapper;
    }

    public TAdminListItem mapAdminListItemFromOrder(Order order) {
        TAdminListItem.Builder itemBuilder = TAdminListItem.newBuilder()
                .setOrderId(order.getId().toString())
                .setPrettyId(order.getPrettyId())
                .setWorkflowId(order.getWorkflow().getId().toString())
                .setBroken(order.isBroken())
                .setOrderType(order.getPublicType())
                .setDisplayOrderType(order.getDisplayType())
                .setDisplayOrderState(EDisplayOrderStateMapper.fromOrder(order))
                .setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()))
                .setDeferred(order.getUseDeferredPayment());
        if (order.getExpiresAt() != null) {
            itemBuilder.setExpiresAt(ProtoUtils.fromInstant(order.getExpiresAt()));
        }
        order.getOrderItems().forEach(orderItem -> itemBuilder.addPartnerInfo(mapPartnerInfoFromOrderItem(orderItem)));

        return itemBuilder.build();
    }

    private TAdminListItemInfo mapPartnerInfoFromOrderItem(OrderItem orderItem) {
        EPartnerType partnerType = ordersAdminEnumMapper.getPartnerByItem(orderItem);
        TAdminListItemInfo.Builder infoBuilder = TAdminListItemInfo.newBuilder()
                .setItemId(orderItem.getId().toString())
                .setPartnerType(partnerType);

        switch (orderItem.getPublicType()) {
            case PT_EXPEDIA_HOTEL:
            case PT_DOLPHIN_HOTEL:
            case PT_TRAVELLINE_HOTEL:
            case PT_BRONEVIK_HOTEL:
            case PT_BNOVO_HOTEL:
                HotelItinerary hotelItinerary = ((HotelOrderItem) orderItem).getHotelItinerary();
                if (hotelItinerary.getFiscalPrice() != null) {
                    infoBuilder.setPrice(ProtoUtils.toTPrice(hotelItinerary.getFiscalPrice()));
                }
                infoBuilder.setHotelInfo(mapAdminListHotelInfo(hotelItinerary));
                break;
            case PT_TRAIN:
                TrainReservation reservation = ((TrainOrderItem) orderItem).getReservation();
                Money totalCost = reservation.getPassengers().stream()
                        .filter(p -> p.getTicket() != null)
                        .map(passenger -> TrainModelHelpers.calculateTotalCostForPassenger(reservation, passenger))
                        .reduce(Money::add)
                        .orElse(Money.zero(ProtoCurrencyUnit.RUB));
                infoBuilder.setPrice(ProtoUtils.toTPrice(totalCost));
                infoBuilder.setTrainInfo(mapAdminListTrainInfo(reservation).build());
                break;
            case PT_FLIGHT:
                AeroflotServicePayload payload = ((AeroflotOrderItem) orderItem).getPayload();
                if (payload.getVariant().getOffer().getTotalPrice() != null) {
                    infoBuilder.setPrice(ProtoUtils.toTPrice(payload.getVariant().getOffer().getTotalPrice()));
                }
                infoBuilder.setAviaInfo(mapAdminListAviaInfo(payload).build());
                break;
            case PT_BUS:
                BusReservation busReservation = ((BusOrderItem) orderItem).getReservation();
                if (busReservation.getOrder() != null && !busReservation.getOrder().getTickets().isEmpty()) {
                    Money total = busReservation.getOrder().getTickets().stream()
                            .map(BusesTicket::getTotalPrice)
                            .reduce(Money::add)
                            .orElse(Money.zero(ProtoCurrencyUnit.RUB));
                    infoBuilder.setPrice(ProtoUtils.toTPrice(total));
                }
                infoBuilder.setBusInfo(mapAdminListBusInfo(busReservation).build());
                break;
            default:
                break;
        }
        return infoBuilder.build();
    }

    private TAdminListHotelInfo mapAdminListHotelInfo(HotelItinerary hotelItinerary) {
        OrderDetails orderDetails = hotelItinerary.getOrderDetails();
        return TAdminListHotelInfo.newBuilder()
                .setHotelName(Strings.nullToEmpty(orderDetails.getHotelName()))
                .setHotelAddress(Strings.nullToEmpty(orderDetails.getAddressByYandex()))
                .setHotelPhone(Strings.nullToEmpty(orderDetails.getHotelPhone()))
                .setCheckIn(ProtoUtils.fromLocalDate(orderDetails.getCheckinDate()))
                .setCheckOut(ProtoUtils.fromLocalDate(orderDetails.getCheckoutDate()))
                .addAllTraveller(hotelItinerary.getGuests().stream()
                        .filter(Objects::nonNull)
                        .filter(Guest::hasFilledName)
                        .map(guest -> TAdminListHotelInfo.TGuest.newBuilder()
                                .setFio(guest.getFullNameReversed())
                                .build())
                        .collect(Collectors.toList()))
                .build();
    }

    private TAdminListTrainInfo.Builder mapAdminListTrainInfo(TrainReservation reservation) {
        TAdminListTrainInfo.Builder trainInfoBuilder = TAdminListTrainInfo.newBuilder()
                .setTrainNumber(Strings.nullToEmpty(reservation.getReservationRequestData().getTrainTicketNumber()))
                .setCarNumber(Strings.nullToEmpty(reservation.getCarNumber()))
                .setDepartureStation(Strings.nullToEmpty(reservation.getUiData().getStationFromTitle()))
                .setArrivalStation(Strings.nullToEmpty(reservation.getUiData().getStationToTitle()));
        if (reservation.getReservationRequestData().getDepartureTime() != null) {
            trainInfoBuilder.setDeparture(ProtoUtils.fromInstant(reservation.getReservationRequestData().getDepartureTime()));
        }
        reservation.getPassengers().forEach(passenger -> {
            var passengerInfo = TAdminListTrainInfo.TPassenger.newBuilder()
                    .setFio(Stream.of(passenger.getLastName(), passenger.getFirstName(), passenger.getPatronymic())
                            .filter(Objects::nonNull)
                            .collect(Collectors.joining(" ")))
                    .setAge(countPassengerAge(passenger.getBirthday()));
            if (passenger.getTicket() != null) {
                passenger.getTicket().getPlaces().forEach(trainPlace -> passengerInfo.addPlaceNumbers(trainPlace.getNumber()));
            }
            trainInfoBuilder.addTraveller(passengerInfo);
        });
        return trainInfoBuilder;
    }

    private TAdminListBusInfo.Builder mapAdminListBusInfo(BusReservation reservation) {
        var ride = reservation.getRide();
        TAdminListBusInfo.Builder infoBuilder = TAdminListBusInfo.newBuilder()
                .setName(ride.getRouteName());
        if (ride.getDepartureTime() != null) {
            infoBuilder.setDeparture(ProtoUtils.fromInstant(ride.getDepartureTime()));
            infoBuilder.setDepartureTimeZone(ride.getTitlePointFrom().getTimezone());
        }
        if (ride.getArrivalTime() != null) {
            infoBuilder.setArrival(ProtoUtils.fromInstant(ride.getArrivalTime()));
            infoBuilder.setArrivalTimeZone(ride.getTitlePointTo().getTimezone());
        }
        if (reservation.getOrder() != null) {
            reservation.getOrder().getTickets().forEach(ticket -> {
                var passengerInfo = TAdminListBusInfo.TPassenger.newBuilder();
                if (!Strings.isNullOrEmpty(ticket.getPassenger().getSeatId())) {
                    passengerInfo.setSeat(ticket.getPassenger().getSeatId());
                }
                infoBuilder.addPassenger(passengerInfo);
            });
        }
        return infoBuilder;
    }

    private TAdminListAviaInfo.Builder mapAdminListAviaInfo(AeroflotServicePayload payload) {
        TAdminListAviaInfo.Builder aviaBuilder = TAdminListAviaInfo.newBuilder();
        payload.getVariant().getSegments()
                .forEach(segment -> aviaBuilder.addSegment(TAdminListAviaInfo.TSegment.newBuilder()
                        .setFlightNumber(segment.getMarketingCarrier().getFlightNumber())
                        .setDepartureAirportCode(segment.getDeparture().getAirportCode())
                        .setArrivalAirportCode(segment.getArrival().getAirportCode())
                        .setDeparture(ProtoUtils.fromLocalDateTime(segment.getDeparture().getDateTime()))));
        payload.getTravellers().forEach(traveller -> aviaBuilder.addTraveller(TAdminListAviaInfo.TPassenger.newBuilder()
                .setFio(Stream.of(traveller.getLastName(), traveller.getFirstName(), traveller.getMiddleName())
                        .filter(Objects::nonNull)
                        .collect(Collectors.joining(" ")))
                .build()));
        return aviaBuilder;
    }

    private int countPassengerAge(LocalDate passengerBirthday) {
        LocalDate today = LocalDate.now();
        int age = today.getYear() - passengerBirthday.getYear();
        if (today.getDayOfYear() < passengerBirthday.getDayOfYear()) {
            age -= 1;
        }
        return age;
    }

    public TGetWorkflowRsp.Builder mapSingleWorkflowInfo(Workflow workflow, List<WorkflowEvent> events) {
        TGetWorkflowRsp.Builder wfBuilder = TGetWorkflowRsp.newBuilder();
        wfBuilder.setWorkflowId(workflow.getId().toString());
        wfBuilder.setWorkflowState(workflow.getState());
        wfBuilder.setVersion(workflow.getVersion());
        wfBuilder.setWorkflowVersion(workflow.getWorkflowVersion());
        if (workflow.getEntityType() != null) {
            wfBuilder.setEntityType(workflow.getEntityType());
        }
        if (workflow.getEntityId() != null) {
            wfBuilder.setEntityId(workflow.getEntityId().toString());
        }
        if (workflow.getSupervisorId() != null) {
            wfBuilder.setSupervisorId(workflow.getSupervisorId().toString());
        }
        wfBuilder.addAllWorkflowStateTransition(workflow.getStateTransitions().stream()
                .map(transition -> {
                    TEntityStateTransition.Builder transitionBuilder = TEntityStateTransition.newBuilder();
                    transitionBuilder.setId(transition.getId());
                    transitionBuilder.setWorkflowId(transition.getWorkflow().getId().toString());
                    transitionBuilder.setFromState(transition.getFromState().toString());
                    transitionBuilder.setToState(transition.getToState().toString());
                    transitionBuilder.setTransitionAt(ProtoUtils.fromInstant(transition.getCreatedAt()));
                    return transitionBuilder.build();
                }).collect(Collectors.toList()));
        wfBuilder.addAllEventInfo(events.stream()
                .map(event -> {
                    var resultEvent = TEventInfo.newBuilder()
                            .setId(event.getId())
                            .setEventType(event.getData().getClass().getTypeName())
                            .setState(event.getState())
                            .setCreatedAt(ProtoUtils.fromInstant(event.getCreatedAt()))
                            .setTimesTried(event.getTimesTried());
                    if (event.getProcessedAt() != null) {
                        resultEvent.setProcessedAt(ProtoUtils.fromInstant(event.getProcessedAt()));
                    }
                    if (event.getData() != null) {
                        try {
                            resultEvent.setData(messagePrinter.print(event.getData()));
                        } catch (InvalidProtocolBufferException e) {
                            log.error("Could not serialize data for event " + event.toString(), e);
                        }
                    }
                    return resultEvent.build();
                })
                .collect(Collectors.toList()));
        return wfBuilder;
    }

    @TransactionMandatory
    public TGetWorkflowRsp mapSingleWorkflowInfo(Workflow workflow) {
        return mapSingleWorkflowInfo(workflow, workflowEventRepository.findAllByWorkflowId(workflow.getId()))
                .build();
    }

    @TransactionMandatory
    public TGetWorkflowRsp mapSingleWorkflowInfo(UUID workflowId) {
        Workflow workflow = workflowRepository.findById(workflowId).orElseThrow(() ->
                Error.with(EErrorCode.EC_NOT_FOUND, "Workflow not found").toEx());
        return mapSingleWorkflowInfo(workflow);
    }

    @TransactionMandatory
    public TGetWorkflowRsp mapWorkflowWithSupervised(Workflow workflow) {
        TGetWorkflowRsp.Builder rspBuilder = mapSingleWorkflowInfo(workflow,
                workflowEventRepository.findAllByWorkflowId(workflow.getId()));
        Stream.of(
                        workflowRepository.findSupervisedWorkflowsWithState(workflow.getId(),
                                EWorkflowState.WS_RUNNING),
                        workflowRepository.findSupervisedWorkflowsWithState(workflow.getId(),
                                EWorkflowState.WS_CRASHED),
                        workflowRepository.findSupervisedWorkflowsWithState(workflow.getId(), EWorkflowState.WS_PAUSED),
                        workflowRepository.findSupervisedWorkflowsWithState(workflow.getId(), EWorkflowState.WS_STOPPED)
                ).flatMap(Collection::stream)
                .forEach(workflowId -> rspBuilder.addSupervisedWorkflows(mapSingleWorkflowInfo(workflowId)));

        return rspBuilder.build();
    }
}
