package ru.yandex.travel.orders.services;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.travel.clients.promo_service_booking_flow.PromoServiceBookingFlowUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.hotels.proto.EYandexPlusEligibility;
import ru.yandex.travel.hotels.proto.TDeterminePromosForOfferRsp;
import ru.yandex.travel.hotels.proto.TExperimentInfo;
import ru.yandex.travel.hotels.proto.TGetWhiteLabelPointsPropsRsp;
import ru.yandex.travel.hotels.proto.TOfferInfo;
import ru.yandex.travel.hotels.proto.TUserInfo;
import ru.yandex.travel.hotels.proto.TWhiteLabelInfo;
import ru.yandex.travel.hotels.proto.TYandexPlusStatus;
import ru.yandex.travel.hotels.services.promoservice.PromoServiceClient;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.promo.PromoCodeApplication;
import ru.yandex.travel.orders.proto.TPromoForOffer;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationResult;
import ru.yandex.travel.orders.services.promo.ServiceDescription;
import ru.yandex.travel.proto.BaseTypes;
import ru.yandex.travel.white_label.proto.EWhiteLabelPartnerId;
import ru.yandex.travel.white_label.proto.EWhiteLabelPointsType;

@Service
@RequiredArgsConstructor
@Slf4j
public class PromoServiceHelperImpl implements PromoServiceHelper {
    private final PromoServiceClient promoServiceClient;

    @Override
    public CompletableFuture<List<TPromoForOffer>> determinePromosForOffers(
            List<ServiceDescription> serviceDescriptions,
            PromoCodeApplicationResult promoCodeApplicationResult,
            List<EDisplayOrderType> userExistingOrderTypes,
            List<BaseTypes.TStringPair> kVExperiments,
            EWhiteLabelPartnerId whiteLabelPartnerId
    ) {
        if (serviceDescriptions.size() != 1 && serviceDescriptions.get(0).getPayload() instanceof HotelItinerary) {
            // at the moment promo service only works with hotel orders
            // otherwise `promoCodeApplicationResult` might contain wrong cost
            throw new IllegalArgumentException("Can only support a single HotelItinerary");
        }
        return CompletableFutures.allOf(
                        serviceDescriptions
                                .stream()
                                .filter(sd -> sd.getPayload() instanceof HotelItinerary)
                                .map(sd ->
                                        determinePromosForOffer(
                                                (HotelItinerary) sd.getPayload(),
                                                UserCredentials.get(),
                                                sd.getFiscalPrice(),
                                                promoCodeApplicationResult.getOriginalAmount(),
                                                promoCodeApplicationResult.getDiscountedAmount(),
                                                userExistingOrderTypes,
                                                kVExperiments,
                                                whiteLabelPartnerId
                                        ).thenApply(rsp -> {
                                            var builder = TPromoForOffer.newBuilder();
                                            if (sd.getId() != null) {
                                                builder.setId(sd.getId());
                                            }
                                            return builder.setPromoInfo(rsp)
                                                    .build();
                                        }))
                                .collect(Collectors.toList())

                )
                // needed to cast generics
                .thenApply(list -> list);
    }

    private CompletableFuture<TDeterminePromosForOfferRsp> determinePromosForOffer(
            HotelItinerary itinerary,
            UserCredentials userCredentials,
            Money originalCost,
            Money priceBeforePromoCodes,
            Money priceAfterPromoCodes,
            List<EDisplayOrderType> userExistingOrderTypes,
            List<BaseTypes.TStringPair> kVExperiments,
            EWhiteLabelPartnerId whiteLabelPartnerId
    ) {
        TOfferInfo offerInfo = PromoServiceBookingFlowUtils.buildOfferInfo(
                itinerary.getPartnerId(),
                itinerary.getOrderDetails().getOriginalId(),
                itinerary.getOrderDetails().getCheckinDate().toString(),
                itinerary.getOrderDetails().getCheckoutDate().toString(),
                originalCost,
                priceBeforePromoCodes,
                priceAfterPromoCodes);

        TUserInfo userInfo = PromoServiceBookingFlowUtils.buildUserInfo(
                userCredentials.getPassportId(),
                userCredentials.isUserIsPlus(),
                PromoServiceBookingFlowUtils.mapDisplayOrderTypesToOrderTypes(userExistingOrderTypes),
                true);

        TExperimentInfo experimentInfo = PromoServiceBookingFlowUtils.buildExperimentInfo(kVExperiments);

        TWhiteLabelInfo whiteLabelInfo = PromoServiceBookingFlowUtils.buildWhiteLabelInfo(
                whiteLabelPartnerId);

        return promoServiceClient.determinePromosForOffer(offerInfo, userInfo, experimentInfo, whiteLabelInfo);
    }

    @Override
    public CompletableFuture<Void> validateAppliedPromoCampaigns(Order order,
                                                                 AuthorizedUser authorizedUser,
                                                                 List<EDisplayOrderType> userExistingOrderTypes
    ) {
        if (!(order instanceof HotelOrder)) {
            return CompletableFuture.completedFuture(null);
        }

        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        HotelItinerary itinerary = orderItem.getHotelItinerary();

        Money promoCodesDiscount = order.getPromoCodeApplications().stream()
                .map(PromoCodeApplication::getDiscount)
                // when promocode is not applied, discount is null
                .filter(Objects::nonNull)
                .reduce(Money::add)
                .orElse(Money.zero(order.getCurrency()));
        Money priceAfterPromocodes = itinerary.getPriceAfterDiscount().subtract(promoCodesDiscount);

        TOfferInfo offerInfo = PromoServiceBookingFlowUtils.buildOfferInfo(
                orderItem.getPartnerId(),
                itinerary.getOrderDetails().getOriginalId(),
                itinerary.getOrderDetails().getCheckinDate().toString(),
                itinerary.getOrderDetails().getCheckoutDate().toString(),
                orderItem.preliminaryTotalCost(),
                itinerary.getPriceAfterDiscount(),
                priceAfterPromocodes);

        TUserInfo userInfo = PromoServiceBookingFlowUtils.buildUserInfo(
                authorizedUser.getPassportId(),
                authorizedUser.getUserIsPlus(),
                PromoServiceBookingFlowUtils.mapDisplayOrderTypesToOrderTypes(userExistingOrderTypes),
                true);

        var kVExperiments = order.getKVExperiments().stream()
                .map(item -> BaseTypes.TStringPair.newBuilder()
                        .setKey(item.getKey())
                        .setValue(item.getValue())
                        .build())
                .collect(Collectors.toList());
        TExperimentInfo experimentInfo = PromoServiceBookingFlowUtils.buildExperimentInfo(kVExperiments);

        // TODO: Получать whiteLabelPartnerId из базы, будет делаться в TRAVELBACK-3506
        TWhiteLabelInfo whiteLabelInfo = PromoServiceBookingFlowUtils.buildWhiteLabelInfo(
                EWhiteLabelPartnerId.WL_UNKNOWN);

        return promoServiceClient.determinePromosForOffer(offerInfo, userInfo, experimentInfo, whiteLabelInfo)
                .thenApply(rsp -> {
                    validatePlusCampaign(rsp.getPlus(), itinerary, priceAfterPromocodes);
                    return null;
                });
    }

    private void validatePlusCampaign(@Nonnull TYandexPlusStatus rsp,
                                      HotelItinerary itinerary,
                                      Money priceAfterPromocodes
    ) {
        if (itinerary.getAppliedPromoCampaigns() != null && itinerary.getAppliedPromoCampaigns().getYandexPlus() != null) {
            YandexPlusApplication yandexPlus = itinerary.getAppliedPromoCampaigns().getYandexPlus();
            if (yandexPlus.getMode() == null) {
                return;
            }

            Preconditions.checkArgument(rsp.getEligibility() == EYandexPlusEligibility.YPE_ELIGIBLE);

            switch (yandexPlus.getMode()) {
                case TOPUP:
                    int pointsFromPromoService = rsp.getPoints().getValue();
                    Preconditions.checkArgument(yandexPlus.getPoints() <= pointsFromPromoService,
                            "User expects more yandex plus points than it's eligible, failing the order. " +
                                    yandexPlus.getPoints() + " > " + pointsFromPromoService);
                    if (yandexPlus.getPoints() < pointsFromPromoService) {
                        // ATM it's there's a legit flow when we receive fewer points from frontend (see HOTELS-5915)
                        log.warn("User expected {} points, but according to promo service {} should be paid." +
                                " Increasing automatically.", yandexPlus.getPoints(), pointsFromPromoService);
                        yandexPlus.setPoints(pointsFromPromoService);
                    }
                    break;
                case WITHDRAW:
                    Preconditions.checkArgument(yandexPlus.getPoints() < priceAfterPromocodes.getNumber().intValue(),
                            "Withdraw points are higher than the actual order price. {} >= {}",
                            yandexPlus.getPoints(), priceAfterPromocodes.getNumber().intValue());
                    break;
                default:
                    throw new IllegalArgumentException("Unknown yandex plus mode: " + yandexPlus);
            }
        } else {
            if (rsp.getEligibility() != EYandexPlusEligibility.YPE_ELIGIBLE) {
                return;
            }

            YandexPlusApplication ypa = YandexPlusApplication.builder()
                    .mode(YandexPlusApplication.Mode.TOPUP)
                    .points(rsp.getPoints().getValue())
                    .build();
            if (itinerary.getAppliedPromoCampaigns() == null) {
                itinerary.setAppliedPromoCampaigns(AppliedPromoCampaigns.builder().yandexPlus(ypa).build());
            } else {
                itinerary.getAppliedPromoCampaigns().setYandexPlus(ypa);
            }
        }
    }

    @Override
    public CompletableFuture<TGetWhiteLabelPointsPropsRsp> getWhiteLabelPointsProps(EWhiteLabelPointsType pointsType,
                                                                                    int amount) {
        return promoServiceClient.getWhiteLabelPointsProps(pointsType, amount);
    }
}
