package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.GeneratedMessageV3;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceException;
import ru.yandex.travel.bus.service.BusesServiceRetryableException;
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoRequest;
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoResponse;
import ru.yandex.travel.buses.backend.proto.worker.TRequestHeader;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.proto.TRefundCalculation;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.services.buses.BusesServiceProvider;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.TGenericRefundToken;
import ru.yandex.travel.orders.workflow.order.generic.proto.TServiceRefundInfo;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TBusRefundTicket;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TBusRefundToken;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainRefundPassenger;
import ru.yandex.travel.orders.workflow.train.proto.TTrainRefundToken;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountRequest;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountRequest;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@Service
@RequiredArgsConstructor
public class RefundCalculationService {
    private final ExpediaProperties expediaProperties;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final BusProperties busProperties;
    private final ImClientProvider imClientProvider;
    private final BusesServiceProvider busesServiceProvider;
    private final Clock clock;

    private static boolean isRefundingState(HotelOrderItem orderItem) {
        switch (orderItem.getPartnerId()) {
            case PI_EXPEDIA:
                return orderItem.getItemState() == EExpediaItemState.IS_REFUNDING;
            case PI_DOLPHIN:
                return orderItem.getItemState() == EDolphinItemState.IS_REFUNDING;
            case PI_TRAVELLINE:
                return orderItem.getItemState() == ETravellineItemState.IS_REFUNDING;
            case PI_BNOVO:
                return orderItem.getItemState() == EBNovoItemState.IS_REFUNDING;
            case PI_BRONEVIK:
                return orderItem.getItemState() == EBronevikItemState.IS_REFUNDING;
            default: {
                log.warn("Can't check the IS_REFUNDING order item state; unsupported item type {}",
                        orderItem.getPartnerId());
                return false;
            }
        }
    }

    @TransactionMandatory
    public TRefundCalculation calculateRefund(Order order, TCalculateRefundReq request) {
        RefundCalculationWrapper<? extends GeneratedMessageV3> refundCalculationWrapper;
        if (order.getPublicType() == EOrderType.OT_HOTEL_EXPEDIA) {
            Error.checkState(order.getOrderItems().size() == 1, "Can't calculate refund for order with more " +
                    "than 1 order item");
            HotelOrderItem orderItem = (HotelOrderItem) order.getOrderItems().get(0);
            //TODO consider other partners
            refundCalculationWrapper = calculateRefundForHotelItemFromRules(orderItem, order.getCurrency());
        } else if (order.getPublicType() == EOrderType.OT_TRAIN) {
            Error.checkState(order.getOrderItems().size() == 1, "Can't calculate refund for order with more " +
                    "than 1 order item");
            TrainOrderItem orderItem = (TrainOrderItem) order.getOrderItems().get(0);
            refundCalculationWrapper = calculateRefundForTrainItem(orderItem,
                    Set.copyOf(request.getTrainCalculateRefundInfo().getBlankIdsList()),
                    order.getCurrency());
        } else {
            throw new UnsupportedOperationException("Refund calculation is not implemented for orders of type " + order.getPublicType());
        }
        return TRefundCalculation.newBuilder()
                .setOrderId(order.getId().toString())
                .setRefundAmount(ProtoUtils.toTPrice(refundCalculationWrapper.getRefundAmount()))
                .setPenaltyAmount(ProtoUtils.toTPrice(refundCalculationWrapper.getPenaltyAmount()))
                .setToken(refundCalculationWrapper.getRefundToken())
                .setExpiresAt(ProtoUtils.fromInstant(refundCalculationWrapper.getExpiresAt()))
                .build();
    }

    @TransactionMandatory
    public TRefundCalculation calculateRefundV2(GenericOrder order, TCalculateRefundReqV2 request) {
        Error.checkArgument(request.getContextCount() > 0, "Context list must be not empty");
        var token = TGenericRefundToken.newBuilder().addAllContext(request.getContextList());
        Map<UUID, List<TRefundPartContext>> partContextsByService = new HashMap<>();
        boolean fullOrderRefund = false;
        for (String partContextStr : request.getContextList()) {
            var partContext = RefundPartsService.partContextFromString(partContextStr);
            if (partContext.getType() == ERefundPartType.RPT_ORDER) {
                Error.checkArgument(request.getContextCount() == 1,
                        "Request must contain only one context of type ORDER");
                fullOrderRefund = true;
            } else {
                UUID id = UUID.fromString(partContext.getServiceId());
                partContextsByService.putIfAbsent(id, new ArrayList<>());
                partContextsByService.get(id).add(partContext);
            }
        }
        List<RefundCalculationWrapper<?>> refundCalculations = new ArrayList<>();
        if (order.getPublicType() == EOrderType.OT_GENERIC && OrderCompatibilityUtils.isTrainOrder(order)) {
            for (OrderItem service : order.getOrderItems()) {
                List<TRefundPartContext> partContexts = partContextsByService.get(service.getId());
                if (!fullOrderRefund && partContexts == null) {
                    continue;
                }
                boolean fullServiceRefund =
                        fullOrderRefund || partContexts.stream().anyMatch(x -> x.getType() == ERefundPartType.RPT_SERVICE);
                if (service.getPublicType() == EServiceType.PT_TRAIN) {
                    TrainOrderItem orderItem = (TrainOrderItem) service;
                    if (fullOrderRefund && orderItem.getState() == EOrderItemState.IS_REFUNDED) {
                        continue;
                    }
                    Set<Integer> blankIds;
                    if (fullServiceRefund) {
                        blankIds = orderItem.getPayload().getPassengers().stream()
                                .filter(x -> x.getTicket().getRefundStatus() == null)
                                .map(x -> x.getTicket().getBlankId())
                                .collect(Collectors.toSet());
                    } else {
                        blankIds = partContexts.stream().map(x -> x.getTrainTicketPartContext().getBlankId())
                                .collect(Collectors.toSet());
                    }
                    RefundCalculationWrapper<TTrainRefundToken> serviceCalculation = calculateRefundForTrainItem(
                            orderItem, blankIds, order.getCurrency());
                    token.addService(TServiceRefundInfo.newBuilder()
                            .setServiceId(service.getId().toString())
                            .setTrainRefundToken(serviceCalculation.refundCalculation)
                            .build());
                    refundCalculations.add(serviceCalculation);
                } else {
                    throw new UnsupportedOperationException("Refund calculation is not implemented for service of " +
                            "type " + service.getPublicType());
                }
            }
        } else if (order.getPublicType() == EOrderType.OT_GENERIC && OrderCompatibilityUtils.isBusOrder(order)) {
            for (OrderItem service : order.getOrderItems()) {
                List<TRefundPartContext> partContexts = partContextsByService.get(service.getId());
                if (!fullOrderRefund && partContexts == null) {
                    continue;
                }
                boolean fullServiceRefund =
                        fullOrderRefund || partContexts.stream().anyMatch(x -> x.getType() == ERefundPartType.RPT_SERVICE);
                if (service.getPublicType() == EServiceType.PT_BUS) {
                    BusOrderItem orderItem = (BusOrderItem) service;
                    if (fullOrderRefund && orderItem.getState() == EOrderItemState.IS_REFUNDED) {
                        continue;
                    }
                    Set<String> ticketIds;
                    if (fullServiceRefund) {
                        ticketIds = orderItem.getPayload().getOrder().getTickets().stream()
                                .filter(x -> x.getStatus() == BusTicketStatus.SOLD)
                                .map(BusesTicket::getId)
                                .collect(Collectors.toSet());
                    } else {
                        ticketIds = partContexts.stream().map(x -> x.getBusRefundPartContext().getTicketId())
                                .collect(Collectors.toSet());
                    }
                    RefundCalculationWrapper<TBusRefundToken> serviceCalculation = calculateRefundForBusItem(
                            orderItem, ticketIds, order.getCurrency());
                    token.addService(TServiceRefundInfo.newBuilder()
                            .setServiceId(service.getId().toString())
                            .setBusRefundToken(serviceCalculation.refundCalculation)
                            .build());
                    refundCalculations.add(serviceCalculation);
                } else {
                    throw new UnsupportedOperationException("Refund calculation is not implemented for service of " +
                            "type " + service.getPublicType());
                }
            }
        } else {
            throw new UnsupportedOperationException("Refund calculation is not implemented for orders of type " + order.getPublicType());
        }
        Money refundAmount =
                refundCalculations.stream().map(RefundCalculationWrapper::getRefundAmount).reduce(Money::add).orElseThrow();
        Money penaltyAmount =
                refundCalculations.stream().map(RefundCalculationWrapper::getPenaltyAmount).reduce(Money::add).orElseThrow();
        Instant expiresAt =
                refundCalculations.stream().map(RefundCalculationWrapper::getExpiresAt).reduce((a, b) -> a.isBefore(b) ? a : b).orElseThrow();
        token.setExpiresAt(ProtoUtils.fromInstant(expiresAt));
        return TRefundCalculation.newBuilder()
                .setOrderId(order.getId().toString())
                .setRefundAmount(ProtoUtils.toTPrice(refundAmount))
                .setPenaltyAmount(ProtoUtils.toTPrice(penaltyAmount))
                .setExpiresAt(ProtoUtils.fromInstant(expiresAt))
                .setToken(BaseEncoding.base64Url().encode(token.build().toByteArray()))
                .build();
    }

    private RefundCalculationWrapper<TBusRefundToken> calculateRefundForBusItem(
            BusOrderItem orderItem, Set<String> ticketIds, ProtoCurrencyUnit currency) {
        if (orderItem.getState() == EOrderItemState.IS_REFUNDED) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Order is already refunded").toEx();
        }
        if (orderItem.getState() == EOrderItemState.IS_REFUNDING) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Some tickets are being refunded at this moment").toEx();
        }
        Error.checkArgument(orderItem.getState().equals(EOrderItemState.IS_CONFIRMED), "Order item is not confirmed");
        Error.checkArgument(ticketIds.size() > 0, "BlankIds must be not empty");
        List<BusesTicket> ticketsToReturn = orderItem.getPayload().getOrder().getTickets().stream()
                .filter(x -> ticketIds.contains(x.getId()))
                .collect(Collectors.toList());
        var tokenBuilder = TBusRefundToken.newBuilder();
        Money refundAmount = Money.zero(currency);
        Money totalAmount = Money.zero(currency);
        Map<String, Money> refundPriceByTicketId = null;
        try {
            refundPriceByTicketId = getBusReturnInfoResponses(orderItem, ticketsToReturn);
        } catch (BusesServiceRetryableException e) {
            log.error("Retryable bus worker exception", e);
            Error.with(EErrorCode.EC_UNAVAILABLE, e.getMessage()).withCause(e).andThrow();
        } catch (BusesServiceException e) {
            log.error("Non retryable bus worker exception", e);
            Error.with(EErrorCode.EC_FAILED_PRECONDITION, e.getMessage()).withCause(e).andThrow();
        }
        Error.checkState(refundPriceByTicketId != null, "Refund prices must be initialized");
        for (BusesTicket ticket : ticketsToReturn) {
            totalAmount = totalAmount.add(ticket.getTotalPrice());
            Money ticketRefundAmount = refundPriceByTicketId.get(ticket.getId());
            refundAmount = refundAmount.add(ticketRefundAmount);
            tokenBuilder.addTickets(TBusRefundTicket.newBuilder()
                    .setRefundAmount(ProtoUtils.toTPrice(ticketRefundAmount))
                    .setTicketId(ticket.getId())
                    .build());
        }
        Money penaltyAmount = totalAmount.subtract(refundAmount);
        TBusRefundToken token = tokenBuilder.build();
        return new RefundCalculationWrapper<>(token,
                BaseEncoding.base64Url().encode(token.toByteArray()),
                refundAmount,
                penaltyAmount,
                Instant.now().plus(busProperties.getRefundCalculationExpireTime()));
    }

    @TransactionMandatory
    public boolean mayInitiateRefundForOrder(Order order) {
        if (order.getPublicType() == EOrderType.OT_HOTEL_EXPEDIA) {
            Preconditions.checkState(order.getOrderItems().size() == 1, "Can't initiate refund for order with more " +
                    "than 1 order item");
            HotelOrderItem orderItem = (HotelOrderItem) order.getOrderItems().get(0);
            return (!order.isUserActionScheduled() && orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CONFIRMED);
        } else if (order.getPublicType() == EOrderType.OT_TRAIN) {
            Preconditions.checkState(order.getOrderItems().size() == 1, "Can't initiate refund for order with more " +
                    "than 1 order item");
            TrainOrderItem orderItem = (TrainOrderItem) order.getOrderItems().get(0);
            boolean orderIsEligible = !order.isUserActionScheduled()
                    && orderItem.getState() == EOrderItemState.IS_CONFIRMED;
            if (orderItem.getTestContext() != null) {
                boolean testContextIsEligible =
                        ((TTrainTestContext) orderItem.getTestContext()).getRefundCheckoutOutcome() == ETrainRefundCheckoutOutcome.RCO_SUCCESS;
                return orderIsEligible && testContextIsEligible;
            } else {
                return orderIsEligible;
            }
        } else if (order.getPublicType() == EOrderType.OT_GENERIC && OrderCompatibilityUtils.isTrainOrder(order)) {
            return !order.isUserActionScheduled() && order.getEntityState() == EOrderState.OS_CONFIRMED;
        } else if (order.getPublicType() == EOrderType.OT_GENERIC && OrderCompatibilityUtils.isBusOrder(order)) {
            return !order.isUserActionScheduled() && order.getEntityState() == EOrderState.OS_CONFIRMED;
        } else {
            throw new UnsupportedOperationException("Refund  is not implemented for orders of type " + order.getPublicType());
        }
    }

    @TransactionMandatory
    public boolean isCurrentlyFullyRefundable(Order order) {
        if (order.getPublicType() == EOrderType.OT_HOTEL_EXPEDIA) {
            Preconditions.checkState(order.getOrderItems().size() == 1, "Can't calculate refund for order with more " +
                    "than 1 order item");
            HotelOrderItem orderItem = (HotelOrderItem) order.getOrderItems().get(0);
            return orderItem.getHotelItinerary().getRefundRules().isFullyRefundableAt(Instant.now());
        } else {
            throw new UnsupportedOperationException("Refund calculation is not implemented for orders of type " + order.getPublicType());
        }
    }

    public RefundCalculationWrapper<TTrainRefundToken> calculateRefundForTrainItem(
            TrainOrderItem orderItem, Set<Integer> blankIds, ProtoCurrencyUnit currency) {
        if (orderItem.getState() == EOrderItemState.IS_REFUNDED) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Order is already refunded").toEx();
        }
        if (orderItem.getState() == EOrderItemState.IS_REFUNDING) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Some tickets are being refunded at this moment").toEx();
        }
        Error.checkArgument(orderItem.getState().equals(EOrderItemState.IS_CONFIRMED),
                "Order item is not confirmed");
        Error.checkArgument(blankIds.size() > 0, "BlankIds must be not empty");
        Map<Integer, List<TrainPassenger>> passengersByBlankId = orderItem.getPayload().getPassengers().stream()
                .collect(Collectors.groupingBy(x -> x.getTicket().getBlankId(), Collectors.toList()));
        if (orderItem.getPayload().isOnlyFullReturnPossible()) {
            Error.checkArgument(blankIds.equals(passengersByBlankId.keySet()), "Only full return allowed");
        }
        if (!orderItem.getPayload().isFullRefundPossible()) {
            Error.checkArgument(blankIds.size() == 1, "Full return not allowed");
        }

        var tokenBuilder = TTrainRefundToken.newBuilder();
        boolean insuranceCheckedOut = orderItem.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT;
        boolean refundFee = orderItem.getConfirmedAt().plus(trainWorkflowProperties.getRefund().getReturnFeeTime())
                .isAfter(Instant.now());
        Money refundAmount = Money.zero(currency);
        Money totalAmount = Money.zero(currency);
        List<RailwayReturnBlankResponse> imReturnBlanks = null;
        try {
            imReturnBlanks = getRailwayReturnBlankResponses(orderItem, blankIds);
        } catch (ImClientRetryableException e) {
            log.error("Retryable IM exception", e);
            Error.with(EErrorCode.EC_UNAVAILABLE, e.getMessage()).withCause(e).andThrow();
        } catch (ImClientException e) {
            log.error("Non retrayble IM exception", e);
            Error.with(EErrorCode.EC_FAILED_PRECONDITION, e.getMessage()).withCause(e).andThrow();
        }
        Error.checkState(imReturnBlanks != null, "IM return blanks must be initialized");
        for (RailwayReturnBlankResponse imReturnBlank : imReturnBlanks) {
            for (TrainPassenger passenger : passengersByBlankId.get(imReturnBlank.getPurchaseOrderItemBlankId())) {
                Money ticketRefundAmount = Money.zero(currency);
                Money feeRefundAmount = Money.zero(currency);
                Money insuranceRefundAmount = Money.zero(currency);
                if (passenger.getCategory() != PassengerCategory.BABY) {
                    TrainTicket ticket = passenger.getTicket();
                    ticketRefundAmount = Money.of(imReturnBlank.getAmount(), currency);
                    if (refundFee) {
                        feeRefundAmount = ticket.calculateRefundFeeAmount();
                        Money zeroAmount = Money.zero(currency);
                        feeRefundAmount = feeRefundAmount.isGreaterThan(zeroAmount) ? feeRefundAmount : zeroAmount;
                    }
                    totalAmount = totalAmount.add(ticket.calculateTotalCost());
                }
                if (insuranceCheckedOut) {
                    insuranceRefundAmount = passenger.getInsurance().getAmount();
                    totalAmount = totalAmount.add(passenger.getInsurance().getAmount());
                }
                refundAmount = refundAmount.add(ticketRefundAmount).add(feeRefundAmount).add(insuranceRefundAmount);

                var refundBlankBuilder = TTrainRefundPassenger.newBuilder()
                        .setBlankId(imReturnBlank.getPurchaseOrderItemBlankId())
                        .setCustomerId(passenger.getCustomerId())
                        .setTicketRefundAmount(ProtoUtils.toTPrice(ticketRefundAmount))
                        .setFeeRefundAmount(ProtoUtils.toTPrice(feeRefundAmount))
                        .setInsuranceRefundAmount(ProtoUtils.toTPrice(insuranceRefundAmount));
                tokenBuilder.addPassenger(refundBlankBuilder.build());
            }
        }
        Money penaltyAmount = totalAmount.subtract(refundAmount);
        TTrainRefundToken token = tokenBuilder.build();
        return new RefundCalculationWrapper<>(token,
                BaseEncoding.base64Url().encode(token.toByteArray()),
                refundAmount,
                penaltyAmount,
                Instant.now().plus(trainWorkflowProperties.getRefund().getCalculationExpireTime()));
    }

    private List<RailwayReturnBlankResponse> getRailwayReturnBlankResponses(
            TrainOrderItem orderItem, Set<Integer> blankIds) {
        ImClient imClient = imClientProvider.getImClientForOrderItem(orderItem);
        List<RailwayReturnBlankResponse> imReturnBlanks = new ArrayList<>();
        Map<Integer, Set<Integer>> blankIdsByImOrderItemId = orderItem.getPayload().getBlankIdsByImOrderItemId();
        for (Integer imOrderItemId : blankIdsByImOrderItemId.keySet()) {
            Set<Integer> allItemBlankIds = blankIdsByImOrderItemId.get(imOrderItemId);
            Set<Integer> itemBlankIdsToReturn = allItemBlankIds.stream().filter(blankIds::contains).collect(Collectors.toSet());
            if (itemBlankIdsToReturn.size() == 0) {
                continue;
            }
            String checkDocumentNumber = orderItem.getPayload().getPassengers().stream()
                    .filter(p -> imOrderItemId.equals(p.getTicket().getPartnerBuyOperationId()))
                    .map(TrainPassenger::getDocumentNumber)
                    .findFirst().orElse(orderItem.getPayload().getPassengers().get(0).getDocumentNumber());
            if (allItemBlankIds.equals(itemBlankIdsToReturn)) {
                var imReq = new ReturnAmountRequest();
                imReq.setServiceReturnAmountRequest(new RailwayReturnAmountRequest());
                imReq.getServiceReturnAmountRequest().setOrderItemId(imOrderItemId);
                imReq.getServiceReturnAmountRequest().setCheckDocumentNumber(checkDocumentNumber);
                ReturnAmountResponse imRsp = imClient.getReturnAmount(imReq);
                imReturnBlanks.addAll(imRsp.getServiceReturnResponse().getBlanks());
            } else {
                for (Integer blankId : itemBlankIdsToReturn) {
                    var imReq = new ReturnAmountRequest();
                    imReq.setServiceReturnAmountRequest(new RailwayReturnAmountRequest());
                    imReq.getServiceReturnAmountRequest().setOrderItemId(imOrderItemId);
                    imReq.getServiceReturnAmountRequest().setCheckDocumentNumber(checkDocumentNumber);
                    imReq.getServiceReturnAmountRequest().setOrderItemBlankIds(List.of(blankId));
                    ReturnAmountResponse imRsp = imClient.getReturnAmount(imReq);
                    imReturnBlanks.addAll(imRsp.getServiceReturnResponse().getBlanks());
                }
            }
        }
        return imReturnBlanks;
    }

    private Map<String, Money> getBusReturnInfoResponses(BusOrderItem orderItem, List<BusesTicket> ticketsById) {
        Map<String, Money> result = new HashMap<>();
        BusesService client = busesServiceProvider.getBusesServiceByOrderItem(orderItem);
        for (var ticket : ticketsById) {
            TRefundInfoResponse rsp = client.refundInfo(TRefundInfoRequest.newBuilder()
                    .setHeader(TRequestHeader.newBuilder().build())
                    .setSupplierId(orderItem.getPayload().getRide().getSupplierId())
                    .setOrderId(orderItem.getPayload().getOrder().getId())
                    .setTicketId(ticket.getId())
                    .build());
            if (!rsp.getRefundInfo().getAvailable()) {
                throw new BusesServiceException(EErrorCode.EC_UNAVAILABLE, "Ticket refund is not available");
            }
            result.put(ticket.getId(), ProtoUtils.fromTPrice(rsp.getRefundInfo().getPrice()));
        }
        return result;
    }

    public RefundCalculationWrapper<THotelRefundToken> calculateRefundForHotelItemFromRules(HotelOrderItem orderItem,
                                                                                            ProtoCurrencyUnit currency) {
        //TODO (mbobrov): that's a one time operation and we take account into consideration here
        Money totalPaid;
        // todo(tivelkov): TRAVELBACK-1585 - refactor the logic to use directly use the order's data
        if (orderItem.getOrder() != null && !orderItem.getOrder().getPayments().isEmpty()) {
            totalPaid = orderItem.getOrder().getPayments().stream()
                    .map(Payment::getPaidAmount)
                    .reduce(Money::add)
                    .orElse(Money.zero(orderItem.getOrder().getCurrency()));
        } else if (orderItem.isPostPaid()) {
            totalPaid = Money.zero(orderItem.preliminaryTotalCost().getCurrency());
        } else {
            totalPaid = orderItem.totalCostAfterReservation();
        }

        Money newInvoiceAmount = null;
        Money refundAmount = null;
        Money penaltyAmount = null;

        Error.checkState(orderItem.getPublicState() != HotelOrderItem.HotelOrderItemState.REFUNDED,
                "Order is already refunded");
        Error.checkState(!isRefundingState(orderItem), "Order is being refunded at this moment");
        Error.checkState(orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CONFIRMED,
                "Order item is not confirmed");

        Instant now = Instant.now(clock);
        RefundRule refundRule = orderItem.getHotelItinerary().getRefundRules().getRuleAtInstant(now);
        int penaltyIndex = orderItem.getHotelItinerary().getRefundRules().getRuleIndexAtInstant(now);
        Preconditions.checkNotNull(refundRule, "Unable to get refund rule for order");
        switch (refundRule.getType()) {
            case FULLY_REFUNDABLE:
                penaltyAmount = Money.of(BigDecimal.ZERO, currency);
                newInvoiceAmount = penaltyAmount;
                refundAmount = totalPaid;
                break;
            case REFUNDABLE_WITH_PENALTY:
                Error.checkArgument(refundRule.getPenalty().getCurrency().getCurrencyCode().equals(currency.getCurrencyCode()),
                        "Cancellation penalty currency %s does not match order currency %s",
                        refundRule.getPenalty().getCurrency().getCurrencyCode(), currency.getCurrencyCode());
                penaltyAmount = refundRule.getPenalty();
                newInvoiceAmount = penaltyAmount;

                if (penaltyAmount.compareTo(totalPaid) > 0) {
                    newInvoiceAmount = totalPaid;
                    refundAmount = Money.zero(currency);
                } else {
                    refundAmount = totalPaid.subtract(penaltyAmount);
                }
                break;
            case NON_REFUNDABLE:
                throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Order is not refundable").toEx();
        }
        return createRefundWrapperForHotelItem(newInvoiceAmount, penaltyAmount, refundAmount, penaltyIndex);
    }

    public RefundCalculationWrapper<THotelRefundToken> calculateRefundForHotelItemFromRefundAmount(HotelOrderItem orderItem,
                                                                                                   Money refundAmount) {
        Preconditions.checkState(orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CONFIRMED,
                "Order item is not confirmed");
        Money totalPaid = getTotalPaidByOrderItem(orderItem);
        Money penaltyAmount = totalPaid.subtract(refundAmount);
        //noinspection UnnecessaryLocalVariable
        Money newInvoiceAmount = penaltyAmount;
        Preconditions.checkArgument(penaltyAmount.isPositiveOrZero(),
                "Penalty amount must be non-negative: got %s", penaltyAmount);

        return createRefundWrapperForHotelItem(newInvoiceAmount, penaltyAmount, refundAmount, null);
    }

    private Money getTotalPaidByOrderItem(HotelOrderItem orderItem) {
        if (orderItem.getOrder() != null && !orderItem.getOrder().getPayments().isEmpty()) {
            return orderItem.getOrder().calculatePaidAmount();
        } else {
            return orderItem.totalCostAfterReservation();
        }
    }

    private RefundCalculationWrapper<THotelRefundToken> createRefundWrapperForHotelItem(
            Money newInvoiceAmount, Money penaltyAmount, Money refundAmount, Integer penaltyIndex) {
        THotelRefundToken.Builder tokenBuilder = THotelRefundToken.newBuilder()
                .setRequestedAt(ProtoUtils.fromInstant(Instant.now(clock)))
                .setNewInvoiceAmount(ProtoUtils.toTPrice(newInvoiceAmount))
                .setPenaltyAmount(ProtoUtils.toTPrice(penaltyAmount))
                .setRefundAmount(ProtoUtils.toTPrice(refundAmount));
        if (penaltyIndex != null) {
            tokenBuilder.setPenaltyIndex(penaltyIndex);
        }
        THotelRefundToken refundCalculation = tokenBuilder.build();

        return new RefundCalculationWrapper<>(refundCalculation,
                BaseEncoding.base64Url().encode(refundCalculation.toByteArray()),
                refundAmount,
                penaltyAmount,
                Instant.now(clock).plus(expediaProperties.getReservation().getRefundDuration()));
        //TODO extract refundDuration to common properties
    }

    @Getter
    @RequiredArgsConstructor
    public static final class RefundCalculationWrapper<R> {
        private final R refundCalculation;
        private final String refundToken;
        private final Money refundAmount;
        private final Money penaltyAmount;
        private final Instant expiresAt;
    }
}
