package ru.yandex.travel.orders.services;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.orders.admin.proto.TAdminActionToken;
import ru.yandex.travel.orders.proto.TRefundCalculation;

@Slf4j
@RequiredArgsConstructor
@Service
@EnableConfigurationProperties(TokenEncrypterProperties.class)
public class TokenEncrypter {
    private static final String HMAC_SHA256_ALGORITHM = "HmacSHA256";

    private final TokenEncrypterProperties tokenEncrypterProperties;

    public String toRefundToken(TRefundCalculation refundCalculation) {
        return toToken(refundCalculation);
    }

    public String toAdminActionToken(TAdminActionToken adminActionToken) {
        return toToken(adminActionToken);
    }

    public TRefundCalculation fromRefundToken(String refundToken) {
        var dataPart = validateTokenAndExtractData(refundToken);
        try {
            return TRefundCalculation.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw Error.with(EErrorCode.EC_ABORTED, "Corrupted refund token").withCause(e).toEx();
        }
    }

    public TAdminActionToken fromAdminActionToken(String adminActionToken) {
        var dataPart = validateTokenAndExtractData(adminActionToken);
        try {
            return TAdminActionToken.parseFrom(BaseEncoding.base64Url().decode(dataPart));
        } catch (InvalidProtocolBufferException e) {
            throw Error.with(EErrorCode.EC_ABORTED, "Corrupted admin action token").withCause(e).toEx();
        }
    }

    private <T extends Message> String toToken(T token) {
        String dataPart = dataPartToString(token);
        String dataHmac = calculateHmac(dataPart);
        return dataPart + ":" + dataHmac;
    }

    private String validateTokenAndExtractData(String token) {
        String[] stringParts = token.split(":");
        Preconditions.checkState(stringParts.length == 2, "Token must have exactly 2 parts");

        String dataPart = stringParts[0];
        String messageHmac = stringParts[1];

        String calculatedHmac = calculateHmac(dataPart);
        if (!calculatedHmac.equals(messageHmac)) {
            throw Error.with(EErrorCode.EC_ABORTED, "Corrupted token").toEx();
        }
        return dataPart;
    }

    private String dataPartToString(Message message) {
        return BaseEncoding.base64Url().encode(message.toByteArray());
    }

    private String calculateHmac(String data) {
        try {
            SecretKeySpec signingKey = new SecretKeySpec(tokenEncrypterProperties.getSecretKey().getBytes(), HMAC_SHA256_ALGORITHM);
            Mac mac = Mac.getInstance(HMAC_SHA256_ALGORITHM);
            mac.init(signingKey);
            return BaseEncoding.base64Url().encode(mac.doFinal(data.getBytes()));
        } catch (Exception e) {
            throw Error.with(EErrorCode.EC_GENERAL_ERROR, "Error encoding token").withCause(e).toEx();
        }
    }
}
