package ru.yandex.travel.orders.services;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.notifications.trains.GenericSegmentArgs;
import ru.yandex.travel.orders.entities.notifications.trains.GenericTrainBaseMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.trains.GenericTrainConfirmedMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.trains.GenericTrainPassengerArgs;
import ru.yandex.travel.orders.entities.notifications.trains.GenericTrainRefundMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.trains.OrderRegistrationStatus;
import ru.yandex.travel.orders.entities.notifications.trains.TrainBaseMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.trains.TrainConfirmedMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.trains.TrainPassengerArgs;
import ru.yandex.travel.orders.entities.notifications.trains.TrainRefundMailSenderArgs;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.RoutePolicy;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainPlace;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;

import static ru.yandex.travel.orders.services.NotificationHelper.formatMoney;
import static ru.yandex.travel.orders.services.NotificationHelper.humanDate;
import static ru.yandex.travel.orders.services.NotificationHelper.humanTime;

@Service
@RequiredArgsConstructor
@Slf4j
public class TrainNotificationHelper {
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final UrlShortenerService urlShortenerService;
    private static final Integer MAIN_PASSENGER_MIN_AGE = 18;
    private static final Pattern PLACE_NUMBER_PATTERN = Pattern.compile("([0-9]+).*");
    private static final String DASH = " \u2014 ";
    private static final String COMMA = ", ";

    public TrainConfirmedMailSenderArgs prepareConfirmedMailSenderArgs(Order order, TrainOrderItem orderItem) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Not a train order: %s", order.getClass().getName());
        var payload = orderItem.getPayload();
        var args = new TrainConfirmedMailSenderArgs();
        fillBaseMailArgs(args, order, orderItem);

        var registrationStatus = OrderRegistrationStatus.MIXED;
        if (payload.getPassengers().stream().map(p -> p.getTicket().getImBlankStatus())
                .allMatch(x -> x == ImBlankStatus.REMOTE_CHECK_IN)) {
            registrationStatus = OrderRegistrationStatus.ENABLED;
        } else if (payload.getPassengers().stream().map(p -> p.getTicket().getImBlankStatus())
                .allMatch(x -> x == ImBlankStatus.NO_REMOTE_CHECK_IN)) {
            registrationStatus = OrderRegistrationStatus.DISABLED;
        }
        args.setRegistrationStatus(registrationStatus);
        args.setInsuranceStatus(payload.getInsuranceStatus());

        var insurancePrice = orderItem.getFiscalItems().stream()
                .filter(x -> x.getType() == FiscalItemType.TRAIN_INSURANCE)
                .map(x -> x.getMoneyAmount().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        var initOrderPrice = orderItem.getFiscalItems().stream()
                .map(x -> x.getMoneyAmount().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        args.setInsurancePrice(formatMoney(insurancePrice));
        args.setInitialOrderPrice(formatMoney(initOrderPrice));
        args.setOrderPrice(formatMoney(orderItem.totalCostAfterReservation().getNumberStripped()));
        return args;
    }

    public GenericTrainConfirmedMailSenderArgs prepareGenericConfirmedMailSenderArgs(GenericOrder order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Not a train order: %s", order.getClass().getName());
        List<TrainOrderItem> orderItems = OrderCompatibilityUtils.getTrainOrderItems(order).stream()
                .sorted(Comparator.comparing(a -> a.getPayload().getDepartureTime()))
                .collect(Collectors.toList());
        var args = new GenericTrainConfirmedMailSenderArgs();
        fillGenericBaseMailArgs(args, order, orderItems);
        args.setPassengers(new ArrayList<>());
        for (TrainPassenger p : orderItems.get(0).getPayload().getPassengers()) {
            args.getPassengers().add(new GenericTrainPassengerArgs(getFullName(p)));
        }
        args.setManyPassengers(args.getPassengers().size() > 1);
        args.setManyTickets(orderItems.stream()
                .flatMap(x -> x.getPayload().getPassengers().stream())
                .map(x -> x.getTicket().getBlankId())
                .distinct().count() > 1);
        args.setSegments(new ArrayList<>());
        for (TrainOrderItem orderItem : orderItems) {
            args.getSegments().add(getGenericSegmentArgs(orderItem, orderItem.getPayload().getPassengers()));
        }
        List<TrainPassenger> passengers = orderItems.stream()
                .flatMap(x -> x.getPayload().getPassengers().stream())
                .collect(Collectors.toList());

        var registrationStatus = OrderRegistrationStatus.MIXED;
        if (passengers.stream().map(p -> p.getTicket().getImBlankStatus())
                .allMatch(x -> x == ImBlankStatus.REMOTE_CHECK_IN)) {
            registrationStatus = OrderRegistrationStatus.ENABLED;
        } else if (passengers.stream().map(p -> p.getTicket().getImBlankStatus())
                .allMatch(x -> x == ImBlankStatus.NO_REMOTE_CHECK_IN)) {
            registrationStatus = OrderRegistrationStatus.DISABLED;
        }
        args.setRegistrationStatus(registrationStatus);
        args.setInsuranceAutoReturn(orderItems.stream().anyMatch(x ->
                x.getPayload().getInsuranceStatus() == InsuranceStatus.AUTO_RETURN));

        var insurancePrice = orderItems.stream()
                .filter(x -> x.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT)
                .flatMap(x -> x.getFiscalItems().stream())
                .filter(x -> x.getType() == FiscalItemType.TRAIN_INSURANCE)
                .map(x -> x.getMoneyAmount().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        var initOrderPrice = orderItems.stream()
                .flatMap(x -> x.getFiscalItems().stream())
                .map(x -> x.getMoneyAmount().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        args.setInsurancePrice(formatMoney(insurancePrice));
        args.setInitialOrderPrice(formatMoney(initOrderPrice));
        var orderPrice = orderItems.stream()
                .map(x -> x.totalCostAfterReservation().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        args.setOrderPrice(formatMoney(orderPrice));
        return args;
    }

    public TrainRefundMailSenderArgs prepareRefundMailSenderArgs(Order order, TrainOrderItem orderItem,
                                                                 TrainTicketRefund refund, boolean isResize) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Not a train order: %s", order.getClass().getName());
        var args = new TrainRefundMailSenderArgs();
        args.setIsResize(isResize);
        fillBaseMailArgs(args, order, orderItem);
        BigDecimal refundSum = refund.getPayload().calculateActualRefundSum().getNumberStripped();
        args.setRefundSum(formatMoney(refundSum));
        args.setPassengers(new ArrayList<>());
        Set<Integer> blankIds = refund.getPayload().getItems().stream()
                .map(PassengerRefundInfo::getBlankId).collect(Collectors.toSet());
        Money paymentSum = Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB);
        for (TrainPassenger p : orderItem.getPayload().getPassengers()) {
            if (!blankIds.contains(p.getTicket().getBlankId())) {
                continue;
            }
            paymentSum = paymentSum.add(p.getTicket().calculateTotalCost());
            if (p.getInsurance() != null &&
                    orderItem.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                paymentSum = paymentSum.add(p.getInsurance().getAmount());
            }
            args.getPassengers().add(createTrainPassengerArgs(p));
            if (refundSum.compareTo(BigDecimal.ZERO) == 0 && p.isNonRefundableTariff()) {
                args.setIsNonRefundableWithZeroSum(true);
            }
        }
        BigDecimal feeSum = paymentSum.getNumberStripped().subtract(refundSum);
        args.setFeeSum(formatMoney(feeSum));
        return args;
    }

    public GenericTrainRefundMailSenderArgs prepareGenericRefundMailSenderArgs(GenericOrder order,
                                                                               List<TrainTicketRefund> refunds,
                                                                               boolean isResize) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Not a train order: %s", order.getClass().getName());
        boolean refundPartFailed = refunds.stream().anyMatch(x -> x.getState() == ETrainTicketRefundState.RS_FAILED ||
                x.getPayload().getItems().stream().anyMatch(i -> i.getRefundOperationStatus() != ImOperationStatus.OK));
        refunds = refunds.stream()
                .filter(x -> x.getState() == ETrainTicketRefundState.RS_REFUNDED)
                .collect(Collectors.toList());
        Set<Integer> refundedBlankIds = refunds.stream().flatMap(x -> x.getPayload().getRefundedItems().stream())
                .map(PassengerRefundInfo::getBlankId).collect(Collectors.toSet());
        List<TrainOrderItem> trainOrderItems = OrderCompatibilityUtils.getTrainOrderItems(order);
        List<TrainOrderItem> refundedOrderItems = new ArrayList<>();
        Map<UUID, List<TrainPassenger>> refundedPassengersByItemId = new HashMap<>();
        for (TrainOrderItem orderItem : trainOrderItems) {
            List<TrainPassenger> refundedPassengers = orderItem.getPayload().getPassengers().stream()
                    .filter(x -> refundedBlankIds.contains(x.getTicket().getBlankId()))
                    .collect(Collectors.toList());
            if (refundedPassengers.size() > 0) {
                refundedOrderItems.add(orderItem);
                refundedPassengersByItemId.put(orderItem.getId(), refundedPassengers);
            }
        }
        Set<String> refundedPassengersDocumentNumbers = refundedPassengersByItemId.values().stream()
                .flatMap(Collection::stream)
                .map(TrainPassenger::getDocumentNumber)
                .collect(Collectors.toSet());
        List<TrainOrderItem> activeOrderItems = new ArrayList<>();
        Map<UUID, List<TrainPassenger>> activePassengersByItemId = new HashMap<>();
        for (TrainOrderItem orderItem : trainOrderItems) {
            List<TrainPassenger> activePassengers = orderItem.getPayload().getPassengers().stream()
                    .filter(x -> x.getTicket().getRefundStatus() == null
                            && refundedPassengersDocumentNumbers.contains(x.getDocumentNumber()))
                    .collect(Collectors.toList());
            if (activePassengers.size() > 0) {
                activeOrderItems.add(orderItem);
                activePassengersByItemId.put(orderItem.getId(), activePassengers);
            }
        }
        var args = new GenericTrainRefundMailSenderArgs();
        args.setResize(isResize);
        args.setRefundPartFailed(refundPartFailed);
        fillGenericBaseMailArgs(args, order, refundedOrderItems);
        args.setPassengers(refundedPassengersByItemId.values().stream()
                .flatMap(Collection::stream)
                .map(TrainNotificationHelper::getFullName)
                .distinct()
                .map(GenericTrainPassengerArgs::new)
                .collect(Collectors.toList()));
        args.setManyPassengers(args.getPassengers().size() > 1);
        args.setManyTickets(refundedBlankIds.size() > 1);
        args.setSegments(new ArrayList<>());
        BigDecimal refundSum = refunds.stream().map(x -> x.getPayload().calculateActualRefundSum().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        BigDecimal paymentSum = BigDecimal.ZERO;
        for (TrainOrderItem orderItem : refundedOrderItems) {
            List<TrainPassenger> refundedPassengers = refundedPassengersByItemId.get(orderItem.getId());
            args.getSegments().add(getGenericSegmentArgs(orderItem, refundedPassengers));
            for (TrainPassenger p : refundedPassengers) {
                paymentSum = paymentSum.add(p.getTicket().calculateTotalCost().getNumberStripped());
                if (p.getInsurance() != null &&
                        orderItem.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                    paymentSum = paymentSum.add(p.getInsurance().getAmount().getNumberStripped());
                }
                if (refundSum.equals(BigDecimal.ZERO) && p.isNonRefundableTariff()) {
                    args.setNonRefundableWithZeroSum(true);
                }
            }
        }
        BigDecimal feeSum = paymentSum.subtract(refundSum);
        args.setRefundSum(formatMoney(refundSum));
        args.setFeeSum(formatMoney(feeSum));

        args.setHasActiveSegments(activeOrderItems.size() > 0);
        args.setActiveTripTitle(buildGenericRouteTitle(activeOrderItems));
        args.setActiveSegments(new ArrayList<>());
        for (TrainOrderItem orderItem : activeOrderItems) {
            List<TrainPassenger> activePassengers = activePassengersByItemId.get(orderItem.getId());
            args.getActiveSegments().add(getGenericSegmentArgs(orderItem, activePassengers));
        }
        return args;
    }

    private static TrainPassengerArgs createTrainPassengerArgs(TrainPassenger p) {
        var passenger = new TrainPassengerArgs();
        passenger.setFullName(getFullName(p));
        passenger.setFirstName(p.getFirstName());
        passenger.setPlace(p.getTicket().getPlaces().stream().map(TrainPlace::getNumber)
                .map(TrainNotificationHelper::removeLeadingZeroes)
                .collect(Collectors.joining(", ")));
        return passenger;
    }

    private static String removeLeadingZeroes(String flightNum) {
        return flightNum.replaceFirst("^0+", "");
    }

    private static String getFullName(TrainPassenger p) {
        return String.format("%s %s", p.getFirstName(), p.getLastName());
    }

    public String getConfirmationSmsText(Order order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getName());
        List<TrainOrderItem> orderItems = OrderCompatibilityUtils.getTrainOrderItems(order);
        var url = String.format(
                "https://%s/my/order/%s?utm_source=sms&utm_medium=transaction&utm_campaign=buy",
                trainWorkflowProperties.getMail().getFrontUrl(),
                order.getId().toString());
        url = urlShortenerService.shorten(url, true);
        var text = trainWorkflowProperties.getSms().getOrderTicketText();
        boolean manyTickets = orderItems.stream().flatMap(x -> x.getPayload().getPassengers().stream())
                .map(x -> x.getTicket().getBlankId()).distinct().count() > 1;
        if (manyTickets) {
            text = trainWorkflowProperties.getSms().getOrderTicketsText();
        }
        String trainPartText = "";
        if (orderItems.size() == 1) {
            TrainOrderItem orderItem = orderItems.get(0);
            LocalDateTime departureLocalDt = ImHelpers.toLocalDateTime(orderItem.getPayload().getDepartureTime(),
                    orderItem.getPayload().getStationFromTimezone());
            trainPartText = String.format(" %s %s",
                    orderItem.getPayload().getReservationRequestData().getTrainTicketNumber(),
                    humanDate(departureLocalDt));
        }
        text = String.format(text, trainPartText, url);
        return text;
    }

    @VisibleForTesting
    public String buildRouteTrainTitle(TrainReservation payload) {
        var from = payload.getUiData().getStationFromTitle();
        var to = payload.getUiData().getStationToTitle();
        var isSameSettlement = payload.getUiData().getStationFromSettlementTitle() != null
                && payload.getUiData().getStationFromSettlementTitle().equals(payload.getUiData().getStationToSettlementTitle());
        if (!isSameSettlement) {
            if (!payload.getUiData().isStationFromNotGeneralize() && payload.getUiData().getStationFromSettlementTitle() != null) {
                from = payload.getUiData().getStationFromSettlementTitle();
            }
            if (!payload.getUiData().isStationToNotGeneralize() && payload.getUiData().getStationToSettlementTitle() != null) {
                to = payload.getUiData().getStationToSettlementTitle();
            }
        }
        return from + " \u2014 " + to;
    }

    @VisibleForTesting
    public String buildGenericRouteTitle(List<TrainOrderItem> orderItems) {
        var points = new ArrayList<Point>();
        for (TrainOrderItem orderItem : orderItems) {
            var payload = orderItem.getPayload();
            var from = new Point(payload.getUiData().getStationFromTitle(),
                    payload.getUiData().getStationFromSettlementTitle(),
                    payload.getUiData().isStationFromNotGeneralize());
            var to = new Point(payload.getUiData().getStationToTitle(),
                    payload.getUiData().getStationToSettlementTitle(),
                    payload.getUiData().isStationToNotGeneralize());

            var isSameSettlement = from.getSettlementTitle() != null
                    && from.getSettlementTitle().equals(to.getSettlementTitle());
            if (isSameSettlement) {
                from.canUseSettlementTitle = false;
                to.canUseSettlementTitle = false;
            }
            if (points.size() > 0) {
                var lastTo = points.get(points.size() - 1);
                if (!(from.stationTitle.equals(lastTo.stationTitle) ||
                        (from.settlementTitle != null && from.settlementTitle.equals(lastTo.settlementTitle)))) {
                    lastTo.routeInterrupt = true;
                    points.add(from);
                }
                if (isSameSettlement) {
                    lastTo.canUseSettlementTitle = false;
                }
            } else {
                points.add(from);
            }
            points.add(to);
        }
        StringBuilder routeString = new StringBuilder();
        for (int i = 0; i < points.size(); i++) {
            Point p = points.get(i);
            routeString.append(p.getTitle());
            if (i == points.size() - 1) {
                // last point, do nothing
            } else if (p.routeInterrupt) {
                routeString.append(COMMA);
            } else {
                routeString.append(DASH);
            }
        }
        return routeString.toString();
    }

    @Data
    private static class Point {
        private final String stationTitle;
        private final String settlementTitle;
        private Boolean canUseSettlementTitle;
        private Boolean routeInterrupt;

        public Point(String stationTitle, String settlementTitle, Boolean notGeneralize) {
            this.stationTitle = stationTitle;
            this.settlementTitle = settlementTitle;
            canUseSettlementTitle = !notGeneralize && settlementTitle != null;
            routeInterrupt = false;
        }

        public String getTitle() {
            return canUseSettlementTitle ? settlementTitle : stationTitle;
        }
    }

    private void fillBaseMailArgs(TrainBaseMailSenderArgs args, Order order, TrainOrderItem orderItem) {
        TrainReservation payload = orderItem.getPayload();
        args.setTrainNumber(payload.getReservationRequestData().getTrainTicketNumber());
        args.setTrainTitle(buildRouteTrainTitle(payload));
        args.setCoachNumber(removeLeadingZeroes(payload.getCarNumber()));
        args.setCoachType(payload.getCarType().getTitle());
        args.setFrontUrl(trainWorkflowProperties.getMail().getFrontUrl());
        args.setOrderUid(order.getId());
        args.setTicketNumber(order.getPrettyId());
        args.setStationFromTitle(payload.getUiData().getStationFromTitle());
        args.setStationToTitle(payload.getUiData().getStationToTitle());

        args.setIsMoscowRegion(payload.isMoscowRegion());
        args.setIsInternationalRoute(payload.getRoutePolicy() == RoutePolicy.INTERNATIONAL);

        var mainPassenger = payload.getPassengers().get(0);
        for (TrainPassenger p : payload.getPassengers()) {
            if (p.getBirthday().plusYears(MAIN_PASSENGER_MIN_AGE).isBefore(LocalDate.now())) {
                mainPassenger = p;
                break;
            }
        }
        args.setMainFullName(getFullName(mainPassenger));
        args.setMainFirstName(mainPassenger.getFirstName());
        args.setPassengers(new ArrayList<>());
        for (TrainPassenger p : payload.getPassengers()) {
            args.getPassengers().add(createTrainPassengerArgs(p));
        }
        var arrivalLocalDt = ImHelpers.toLocalDateTime(payload.getArrivalTime(), payload.getStationToTimezone());
        var departureLocalDt = ImHelpers.toLocalDateTime(payload.getDepartureTime(), payload.getStationFromTimezone());
        var arrivalRwDt = ImHelpers.toLocalDateTime(payload.getArrivalTime(), payload.getStationToRailwayTimezone());
        var departureRwDt = ImHelpers.toLocalDateTime(payload.getDepartureTime(),
                payload.getStationFromRailwayTimezone());
        args.setArrivalDateLocal(humanDate(arrivalLocalDt));
        args.setArrivalTimeLocal(humanTime(arrivalLocalDt));
        args.setArrivalTzNameRw(payload.getUiData().getStationToRailwayTimeZoneText());
        args.setArrivalDateRw(humanDate(arrivalRwDt));
        args.setArrivalTimeRw(humanTime(arrivalRwDt));
        args.setDepartureDateLocal(humanDate(departureLocalDt));
        args.setDepartureTimeLocal(humanTime(departureLocalDt));
        args.setDepartureTzNameRw(payload.getUiData().getStationFromRailwayTimeZoneText());
        args.setDepartureDateRw(humanDate(departureRwDt));
        args.setDepartureTimeRw(humanTime(departureRwDt));
    }

    private void fillGenericBaseMailArgs(GenericTrainBaseMailSenderArgs args, GenericOrder order,
                                         List<TrainOrderItem> orderItems) {
        args.setFrontUrl(trainWorkflowProperties.getMail().getFrontUrl());
        args.setOrderUid(order.getId());

        args.setTicketNumber(order.getPrettyId());
        args.setTripTitle(buildGenericRouteTitle(orderItems));
        TrainOrderItem firstOrderItem = orderItems.get(0);
        args.setIsMoscowRegion(firstOrderItem.getPayload().isMoscowRegion());

        TrainPassenger mainPassenger = firstOrderItem.getPayload().getPassengers().stream()
                .filter(p -> p.getBirthday().plusYears(MAIN_PASSENGER_MIN_AGE).isBefore(LocalDate.now()))
                .findFirst().orElse(firstOrderItem.getPayload().getPassengers().get(0));
        args.setMainFullName(getFullName(mainPassenger));
        args.setMainFirstName(mainPassenger.getFirstName());
        args.setManyTrains(orderItems.size() > 1);
    }

    private GenericSegmentArgs getGenericSegmentArgs(TrainOrderItem orderItem, List<TrainPassenger> passengers) {
        var segment = new GenericSegmentArgs();
        TrainReservation payload = orderItem.getPayload();
        segment.setTrainNumber(payload.getReservationRequestData().getTrainTicketNumber());
        segment.setTrainTitle(buildRouteTrainTitle(payload));
        segment.setCoachNumber(removeLeadingZeroes(payload.getCarNumber()));
        segment.setCoachType(payload.getCarType().getTitle());
        segment.setStationFromTitle(payload.getUiData().getStationFromTitle());
        segment.setStationToTitle(payload.getUiData().getStationToTitle());

        var arrivalLocalDt = ImHelpers.toLocalDateTime(payload.getArrivalTime(), payload.getStationToTimezone());
        var departureLocalDt = ImHelpers.toLocalDateTime(payload.getDepartureTime(),
                payload.getStationFromTimezone());
        var arrivalRwDt = ImHelpers.toLocalDateTime(payload.getArrivalTime(),
                payload.getStationToRailwayTimezone());
        var departureRwDt = ImHelpers.toLocalDateTime(payload.getDepartureTime(),
                payload.getStationFromRailwayTimezone());
        segment.setArrivalDateLocal(humanDate(arrivalLocalDt));
        segment.setArrivalTimeLocal(humanTime(arrivalLocalDt));
        segment.setArrivalTzNameRw(payload.getUiData().getStationToRailwayTimeZoneText());
        segment.setArrivalDateRw(humanDate(arrivalRwDt));
        segment.setArrivalTimeRw(humanTime(arrivalRwDt));
        segment.setDepartureDateLocal(humanDate(departureLocalDt));
        segment.setDepartureTimeLocal(humanTime(departureLocalDt));
        segment.setDepartureTzNameRw(payload.getUiData().getStationFromRailwayTimeZoneText());
        segment.setDepartureDateRw(humanDate(departureRwDt));
        segment.setDepartureTimeRw(humanTime(departureRwDt));

        segment.setPlaces(getPlacesString(passengers));
        return segment;
    }

    @VisibleForTesting
    public String getPlacesString(List<TrainPassenger> passengers) {
        List<String> numbers = passengers.stream()
                .flatMap(p -> p.getTicket().getPlaces().stream())
                .map(TrainPlace::getNumber)
                .map(TrainNotificationHelper::removeLeadingZeroes)
                .distinct()
                .sorted(Comparator.comparing(x -> {
                    var m = PLACE_NUMBER_PATTERN.matcher(x);
                    if (m.matches()) {
                        return Integer.parseInt(m.group(1));
                    }
                    return Integer.MAX_VALUE;
                }))
                .collect(Collectors.toList());
        String prefix;
        if (numbers.size() == 0) {
            return "без мест";
        } else if (numbers.size() == 1) {
            prefix = "место ";
        } else {
            prefix = "места ";
        }
        return prefix + String.join(", ", numbers);
    }
}
