package ru.yandex.travel.orders.services;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Param;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.entities.notifications.UrlShortenerRetryableException;

@Slf4j
@Service
@EnableConfigurationProperties(UrlShortenerConfigurationProperties.class)
public class UrlShortenerService {
    private final UrlShortenerConfigurationProperties config;
    private final AsyncHttpClientWrapper ahcClient;

    public UrlShortenerService(UrlShortenerConfigurationProperties config,
                               @Qualifier("urlShortenerAhcClientWrapper") AsyncHttpClientWrapper ahcClient) {
        this.config = config;
        this.ahcClient = ahcClient;
    }

    public String shorten(String url, boolean booking) {
        List<Param> params = new ArrayList<>();
        if (booking) {
            params.add(new Param("type", "booking"));
        }
        params.add(new Param("url", url));
        return sync(sendRequest("GET", "/--", params));
    }

    private CompletableFuture<String> sendRequest(String method, String path, List<Param> params) {
        RequestBuilder requestBuilder = createBaseRequestBuilder(method, config.getTimeout())
                .setUrl(config.getBaseUrl() + path)
                .setQueryParams(params);
        return ahcClient.executeRequest(requestBuilder, path).thenApply(this::parseResponse);
    }

    private String parseResponse(Response response) {
        if (response.getStatusCode() >= 200 && response.getStatusCode() < 300) {
            return response.getResponseBody();
        } else {
            throw new RuntimeException(String.format("UrlShortenerService http error. Call: %s. Response body: %s",
                    response.getStatusText(), response.getResponseBody()));
        }
    }

    private RequestBuilder createBaseRequestBuilder(String method, Duration timeout) {
        RequestBuilder builder = new RequestBuilder()
                .setReadTimeout(Math.toIntExact(timeout.toMillis()))
                .setRequestTimeout(Math.toIntExact(timeout.toMillis()))
                .setMethod(method);
        return builder;
    }

    private <T> T sync(CompletableFuture<T> future) {
        try {
            return future.get();
        } catch (InterruptedException e) {
            log.error("UrlShortenerService call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new UrlShortenerRetryableException("UrlShortenerService call interrupted");
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof UrlShortenerRetryableException) {
                throw (UrlShortenerRetryableException) cause;
            } else if (cause instanceof TimeoutException) {
                throw new UrlShortenerRetryableException(e);
            } else if (cause instanceof IOException) {
                throw new UrlShortenerRetryableException(e);
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new  RuntimeException(e);
            }
        }
    }
}
