package ru.yandex.travel.orders.services;

import java.io.IOException;
import java.io.StringReader;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Param;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.integration.balance.xmlrpc.XmlUtils;
import ru.yandex.travel.orders.entities.notifications.YaSmsRetryableException;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
@Service
@EnableConfigurationProperties(YaSmsConfigurationProperties.class)
public class YaSmsService {
    private final YaSmsConfigurationProperties config;
    private final AsyncHttpClientWrapper ahcClient;
    private final TvmWrapper tvm;

    public YaSmsService(YaSmsConfigurationProperties config,
                        @Qualifier("yaSmsAhcClientWrapper") AsyncHttpClientWrapper ahcClient,
                        @Autowired(required = false) TvmWrapper tvm) {
        this.config = config;
        this.ahcClient = ahcClient;
        this.tvm = tvm;
        if (tvm != null) {
            tvm.validateAlias(config.getTvmAlias());
        }
    }

    private boolean isIgnored(String phone) {
        YaSmsConfigurationProperties.IgnoreConfig ignore = config.getIgnore();
        switch (ignore.getMode()) {
            case BLACK_LIST:
                return ignore.getPhones().contains(phone);
            case WHITE_LIST:
                return !ignore.getPhones().contains(phone);
        }
        return false;
    }

    public String sendSms(String text, String phone) {
        if (isIgnored(phone)) {
            log.info("Not sending sms to {} as it is ignored according to the config.", phone);
            return null;
        }
        List<Param> params = new ArrayList<>();
        params.add(new Param("utf8", "1"));
        params.add(new Param("text", text));
        params.add(new Param("phone", phone));
        var document = sync(sendRequest("GET", "/sendsms", params));
        var resultId = document.valueOf("doc/message-sent/@id");
        Preconditions.checkState(!Strings.isNullOrEmpty(resultId), "Message id does not exists");
        return resultId;
    }

    private CompletableFuture<Document> sendRequest(String method, String path, List<Param> params) {
        var timeout = config.getTimeout();
        params.add(new Param("sender", config.getSender()));
        RequestBuilder requestBuilder = createBaseRequestBuilder(method, timeout)
                .setUrl(config.getBaseUrl() + path)
                .setQueryParams(params);
        return ahcClient.executeRequest(requestBuilder, path).thenApply(this::parseResponse);
    }

    private static Document parseXml(String content) {
        try {
            return XmlUtils.createSaxReader().read(new StringReader(content));
        } catch (DocumentException e) {
            log.debug("Failed to parse the document: {}", content);
            throw new YaSmsServiceException("Failed to parse the document", e);
        }
    }

    private Document parseResponse(Response response) {
        if (response.getStatusCode() >= 200 && response.getStatusCode() < 300) {
            var document = parseXml(response.getResponseBody());
            var errorCode = document.valueOf("doc/errorcode");
            if (!Strings.isNullOrEmpty(errorCode)) {
                var errorMsg = document.valueOf("doc/error");
                throw new YaSmsServiceException(errorCode, errorMsg);
            }
            return document;
        } else {
            throw new YaSmsServiceException(String.format("YaSms http error. Call: %s. Response body: %s",
                    response.getStatusText(), response.getResponseBody()), response.getStatusCode());
        }
    }

    private RequestBuilder createBaseRequestBuilder(String method, Duration timeout) {
        RequestBuilder builder = new RequestBuilder()
                .setReadTimeout(Math.toIntExact(timeout.toMillis()))
                .setRequestTimeout(Math.toIntExact(timeout.toMillis()))
                .setMethod(method);
        if (tvm != null) {
            String serviceTicket = tvm.getServiceTicket(config.getTvmAlias());
            builder.setHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), serviceTicket);
        }
        return builder;
    }

    private <T> T sync(CompletableFuture<T> future) {
        try {
            return future.get();
        } catch (InterruptedException e) {
            log.error("YaSms call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new YaSmsRetryableException("YaSms call interrupted");
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof YaSmsRetryableException) {
                throw (YaSmsRetryableException) cause;
            } else if (cause instanceof TimeoutException) {
                throw new YaSmsRetryableException("YaSms call timeout", e);
            } else if (cause instanceof IOException) {
                throw new YaSmsRetryableException("YaSms call io error", e);
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e);
            }
        }
    }

}
