package ru.yandex.travel.orders.services.admin;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.orders.admin.proto.TAdminActionToken;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.services.TokenEncrypter;

@Service
@RequiredArgsConstructor
public class AdminActionTokenService {
    private final TokenEncrypter tokenEncrypter;

    public TAdminActionToken buildOrderToken(Order order) {
        var adminActionToken = TAdminActionToken.newBuilder()
                .setOrderId(order.getId().toString())
                .setOrderVersion(order.getVersion());
        order.getOrderItems().forEach(item -> adminActionToken.addOrderItemSnapshot(
                TAdminActionToken.TItemSnapshot.newBuilder()
                        .setItemId(item.getId().toString())
                        .setItemVersion(item.getVersion()))
        );
        order.getInvoices().forEach(invoice -> adminActionToken.addInvoiceItemSnapshot(
                TAdminActionToken.TItemSnapshot.newBuilder()
                        .setItemId(invoice.getId().toString())
                        .setItemVersion(invoice.getVersion()))
        );

        return adminActionToken.build();
    }

    public void checkOrderToken(Order order, String adminActionToken) {
        TAdminActionToken parsedAdminActionToken = tokenEncrypter.fromAdminActionToken(adminActionToken);

        boolean valid = order.getId().equals(UUID.fromString(parsedAdminActionToken.getOrderId()))
                && order.getVersion().equals(parsedAdminActionToken.getOrderVersion())
                && parsedAdminActionToken.getOrderItemSnapshotList().stream().allMatch((orderItemSnapshot) ->
                    order.getOrderItems().stream().anyMatch((orderItem) ->
                        orderItem.getId().equals(UUID.fromString(orderItemSnapshot.getItemId()))
                                && orderItemSnapshot.getItemVersion() == orderItem.getVersion())
                )
                && parsedAdminActionToken.getInvoiceItemSnapshotList().stream().allMatch((orderInvoiceSnapshot) ->
                order.getInvoices().stream().anyMatch((orderInvoice) ->
                        orderInvoice.getId().equals(UUID.fromString(orderInvoiceSnapshot.getItemId()))
                                && orderInvoice.getVersion() == orderInvoiceSnapshot.getItemVersion())
        );

        if (!valid) {
            Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Action admin token is invalid").andThrow();
        }
    }
}
