package ru.yandex.travel.orders.services.admin

import io.grpc.Status
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo
import ru.yandex.travel.orders.admin.proto.TGetPaymentsStateRsp
import ru.yandex.travel.orders.admin.proto.TResumePaymentsRsp
import ru.yandex.travel.orders.admin.proto.TStopPaymentsRsp
import ru.yandex.travel.orders.cache.BalanceContractDictionary
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository


@Service
class AdminPartnerPaymentsService @Autowired constructor(
    private val balanceContractDictionary: BalanceContractDictionary,
    private val billingPartnerConfigRepository: BillingPartnerConfigRepository,
) {
    fun getPaymentsState(billingClientId: Long, billingContractId: Long): TGetPaymentsStateRsp {
        val billingPartnerConfig = getBillingPartnerConfig(billingClientId, billingContractId)
        return TGetPaymentsStateRsp.newBuilder()
            .setPaymentsEnabled(isPaymentsEnabled(billingPartnerConfig)).build()
    }

    fun stopPayments(billingClientId: Long, billingContractId: Long): TStopPaymentsRsp {
        val billingPartnerConfig = getBillingPartnerConfig(billingClientId, billingContractId)
        val clientId = billingPartnerConfig.billingClientId
        if (!isPaymentsEnabled(billingPartnerConfig)) {
            throw Status.FAILED_PRECONDITION.withDescription(
                "Для клиента $clientId платежи уже были остановлены").asRuntimeException()
        }

        billingPartnerConfig.exportToYt = false
        billingPartnerConfig.generateTransactions = false
        return TStopPaymentsRsp.newBuilder()
            .setBillingClientId(clientId)
            .setStopResult("Платежи успешно остановлены для клиента $clientId")
            .setPaymentsEnabled(isPaymentsEnabled(billingPartnerConfig))
            .build()
    }

    fun resumePayments(billingClientId: Long, billingContractId: Long): TResumePaymentsRsp {
        val billingPartnerConfig = getBillingPartnerConfig(billingClientId, billingContractId)
        val clientId = billingPartnerConfig.billingClientId
        if (isPaymentsEnabled(billingPartnerConfig)) {
            throw Status.FAILED_PRECONDITION.withDescription(
                "Для клиента $clientId платежи уже были включены").asRuntimeException()
        }
        billingPartnerConfig.exportToYt = true
        billingPartnerConfig.generateTransactions = true
        return TResumePaymentsRsp.newBuilder()
            .setBillingClientId(clientId)
            .setResumeResult("Платежи успешно возобновлены для клиента $clientId")
            .setPaymentsEnabled(isPaymentsEnabled(billingPartnerConfig))
            .build()
    }

    private fun isPaymentsEnabled(billingPartnerConfig: BillingPartnerConfig): Boolean {
        return billingPartnerConfig.exportToYt && billingPartnerConfig.generateTransactions
    }

    private fun getBillingPartnerConfig(billingClientId: Long, billingContractId: Long): BillingPartnerConfig {
        val clientId: Long = getClientId(billingClientId, billingContractId)
        return billingPartnerConfigRepository.findById(clientId).orElseThrow {
            Status.NOT_FOUND.withDescription(
                "Billing config not found for clientId $billingClientId, contractId $billingContractId")
                .asRuntimeException()
        }
    }

    private fun getClientId(clientId: Long, contractId: Long): Long {
        if (clientId != 0L) {
            return clientId
        }

        val contractInfo: ContractInfo = balanceContractDictionary.findContractInfoByContractId(contractId)
            ?: throw Status.NOT_FOUND.withDescription(
                "Client is not found for contract $contractId").asRuntimeException()
        return contractInfo.clientId
    }
}
