package ru.yandex.travel.orders.services.admin;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.orders.commons.proto.EAdminAction;
import ru.yandex.travel.orders.commons.proto.EAdminRole;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;

public class AdminUserCapabilitiesManager {
    private final static Set<EAdminAction> basicActions = Set.of(
            EAdminAction.AA_GET_INFO, EAdminAction.AA_GET_PRIVATE_INFO,
            EAdminAction.AA_CHANGE_PHONE, EAdminAction.AA_CHANGE_EMAIL,
            EAdminAction.AA_SEND_USER_NOTIFICATION, EAdminAction.AA_UPDATE_TRAIN_TICKET_STATUS_WITH_IM,
            EAdminAction.AA_LINK_ISSUE
    );

    private final static Set<EAdminAction> advancedActions = Set.of(
            EAdminAction.AA_MODIFY, EAdminAction.AA_GET_ORDER_LOGS, EAdminAction.AA_GET_WORKFLOW_INFO,
            EAdminAction.AA_PAUSE_WORKFLOW, EAdminAction.AA_UNPAUSE_WORKFLOW, EAdminAction.AA_STOP_WORKFLOW,
            EAdminAction.AA_REFUND_USER_MONEY, EAdminAction.AA_RETRY_REFUND_MONEY,
            EAdminAction.AA_CHANGE_EVENT_STATE, EAdminAction.AA_SEND_EVENT,
            EAdminAction.AA_REGENERATE_VOUCHERS
    );

    private final static Set<EAdminAction> advancedHotelActions = Set.of(
            EAdminAction.AA_RESTORE_DOLPHIN_ORDER,
            EAdminAction.AA_CALCULATE_HOTEL_ORDER_REFUND,
            EAdminAction.AA_REFUND_HOTEL_ORDER,
            EAdminAction.AA_CALCULATE_HOTEL_MONEY_ONLY_REFUND,
            EAdminAction.AA_MANUAL_REFUND_MONEY_ONLY,
            EAdminAction.AA_MODIFY_HOTEL_ORDER_DETAILS,
            EAdminAction.AA_ONLY_FULL_REFUND_HOTEL_ORDER
    );

    private static <A, B> Set<Tuple2<A, B>> crossJoin(Set<A> actions, Set<B> types) {
        return actions.stream()
                .flatMap(a -> types.stream().map(b -> Tuple2.tuple(a, b)))
                .collect(Collectors.toUnmodifiableSet());
    }

    private static <T> Set<T> merge(Set<T> a, Set<T> b) {
        return Stream.concat(a.stream(), b.stream()).collect(Collectors.toUnmodifiableSet());
    }

    private final static Map<EAdminRole, Set<Tuple2<EAdminAction, EDisplayOrderType>>> allowedActions = Map.ofEntries(
            Map.entry(
                    EAdminRole.AR_OPERATOR,
                    crossJoin(basicActions, Set.of(EDisplayOrderType.values()))
            ),
            Map.entry(
                    EAdminRole.AR_ADVANCED_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(merge(advancedActions, advancedHotelActions), Set.of(EDisplayOrderType.values()))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_ADV_HOTEL_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(merge(advancedActions, advancedHotelActions),
                                    Set.of(EDisplayOrderType.DT_HOTEL, EDisplayOrderType.DT_UNKNOWN))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_ADV_AVIA_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(advancedActions, Set.of(EDisplayOrderType.DT_AVIA, EDisplayOrderType.DT_UNKNOWN))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_ADV_BUS_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(advancedActions, Set.of(EDisplayOrderType.DT_BUS, EDisplayOrderType.DT_UNKNOWN))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_ADV_TRAINS_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(advancedActions, Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_UNKNOWN))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_ADV_SUBURBAN_OPERATOR,
                    merge(
                            crossJoin(basicActions, Set.of(EDisplayOrderType.values())),
                            crossJoin(advancedActions, Set.of(EDisplayOrderType.DT_SUBURBAN, EDisplayOrderType.DT_UNKNOWN))
                    )
            ),
            Map.entry(
                    EAdminRole.AR_DEVELOPER,
                    crossJoin(Set.of(EAdminAction.values()), Set.of(EDisplayOrderType.values()))
            ),
            Map.entry(
                    EAdminRole.AR_SUPERUSER,
                    crossJoin(Set.of(EAdminAction.values()), Set.of(EDisplayOrderType.values()))
            ),
            Map.entry(
                    EAdminRole.AR_BIZDEV,
                    merge(
                        crossJoin(Set.of(EAdminAction.AA_PARTNER_PAYMENTS_ENABLING),
                                Set.of(EDisplayOrderType.DT_UNKNOWN)),
                        crossJoin(Set.of(EAdminAction.AA_MOVE_ORDER_TO_NEW_CONTRACT),
                                Set.of(EDisplayOrderType.DT_HOTEL, EDisplayOrderType.DT_UNKNOWN))
                    )

            )
    );

    private final Set<EAdminRole> roles;
    private final boolean rolesEnabled;

    public AdminUserCapabilitiesManager(Set<EAdminRole> roles, boolean rolesEnabled) {
        this.roles = roles;
        this.rolesEnabled = rolesEnabled;
    }

    public boolean canDoAction(EAdminAction action, EDisplayOrderType type) {
        return !rolesEnabled || roles.stream().anyMatch(
                role -> allowedActions.getOrDefault(role, Set.of()).contains(Tuple2.tuple(action, type))
        );
    }
}
