package ru.yandex.travel.orders.services.admin;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.ChronoUnit;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItineraryOriginalInfo;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderGuestsParams;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;

@Slf4j
public class HotelOrderDetailsModifier {
    private final HotelOrder order;
    private final HotelItinerary itinerary;
    private Offer uiPayload;
    private boolean isModified;
    private HotelItineraryOriginalInfo originalInfo;

    public HotelOrderDetailsModifier(HotelOrder o, HotelOrderItem oi) {
        order = o;
        itinerary = oi.getHotelItinerary();
        uiPayload = Offer.fromJsonNode(itinerary.getUiPayload());
        isModified = false;
        originalInfo = itinerary.getOriginalInfo();
        if (originalInfo == null) {
            originalInfo = new HotelItineraryOriginalInfo();
            originalInfo.setCheckinDate(itinerary.getOrderDetails().getCheckinDate());
            originalInfo.setCheckoutDate(itinerary.getOrderDetails().getCheckoutDate());
            originalInfo.setGuests(itinerary.getGuests());
            originalInfo.setRefundRules(itinerary.getRefundRules());
        }
    }

    public void finish() {
        if (isModified) {
            itinerary.setUiPayload(uiPayload.toJsonNode());
            itinerary.setOriginalInfo(originalInfo);
        }
    }

    public void changeDates(LocalDate checkInDate, LocalDate checkOutDate) {
        Preconditions.checkArgument(checkInDate.isBefore(checkOutDate), "CheckIn Date should be before CheckOut");
        log.info("Changing dates for order {} ({}), new dates: {}...{}, current dates {}...{}, original dates {}...{}",
            order.getPrettyId(), order.getId(), checkInDate, checkOutDate,
            itinerary.getOrderDetails().getCheckinDate(), itinerary.getOrderDetails().getCheckoutDate(),
            originalInfo.getCheckinDate(), originalInfo.getCheckoutDate()
        );
        isModified = true;
        itinerary.getOrderDetails().setCheckinDate(checkInDate);
        itinerary.getOrderDetails().setCheckoutDate(checkOutDate);

        uiPayload = uiPayload.toBuilder().metaInfo(
                uiPayload.getMetaInfo().toBuilder().search(
                        uiPayload.getMetaInfo().getSearch().toBuilder()
                                .checkIn(checkInDate)
                                .checkOut(checkOutDate)
                                .build()
                ).build()
        ).build();// простите меня, пожалуйста, меня заставили

        int nights = (int) ChronoUnit.DAYS.between(checkInDate, checkOutDate);
        if (itinerary instanceof BNovoHotelItinerary) {
            var pItinerary = (BNovoHotelItinerary)itinerary;
            pItinerary.setBNovoStay(
                    pItinerary.getBNovoStay().toBuilder()
                    .checkin(checkInDate)
                    .nights(nights)
                    .build()
            );
            // Также есть bNovoStay.rates, но кажется, его нет необходимости патчить
        } else if (itinerary instanceof DolphinHotelItinerary) {
            var pItinerary = (DolphinHotelItinerary)itinerary;
            pItinerary.setCheckinDate(checkInDate);
            pItinerary.setNights(nights);
        } else if (itinerary instanceof ExpediaHotelItinerary || itinerary instanceof BronevikHotelItinerary) {
            // Nothing special here
        } else if (itinerary instanceof TravellineHotelItinerary) {
            var pItinerary = (TravellineHotelItinerary)itinerary;
            LocalTime checkInTime = pItinerary.getOffer().getStayDates().getStartDate().toLocalTime();
            LocalTime checkOutTime = pItinerary.getOffer().getStayDates().getEndDate().toLocalTime();
            pItinerary.setOffer(
                    pItinerary.getOffer().toBuilder()
                            .stayDates(pItinerary.getOffer().getStayDates().toBuilder()
                                    .startDate(LocalDateTime.of(checkInDate, checkInTime))
                                    .endDate(LocalDateTime.of(checkOutDate, checkOutTime))
                                    .build()
                            )
                    .build()
            );
        } else {
            throw new RuntimeException("Unknown itinerary class");
        }
        // При модификации RefundRules придерживаемся следующей идеологии:
        // 1. если дата checkIn стала раньше, чем оригинальный checkIn, то смещаем все даты в RefundRules на столько же дней.
        // Это "безопасно", т.к. с т.з. правила отмены становятся только строже, и партнёра это не удивит
        //  (ему же мы ничего не сообщаем)
        // 2. если дата checkIn стала позже, чем оригинальная, то даты в RefundRules не смещаем.
        // Причина - чтобы "ослаблять" правила отмены нужно, чтобы партнёр знал об этом, а мы ему не сообщаем ничего.
        // Если самовольно подвинуть правила отмены, то при попытке отмены партнёр не согласится этого сделать.

        // На сколько дней позже стал checkIn
        int checkinDaysPlus = (int) ChronoUnit.DAYS.between(originalInfo.getCheckinDate(), checkInDate);
        if (checkinDaysPlus > 0) {
            // п.2 - не смещаем назад
            checkinDaysPlus = 0;
        }
        // RefundRules меняем всегда, т.к. вдруг они уже были исправлены
        log.info("Shift refund rules for order {} ({}), subtract {} days",
            order.getPrettyId(), order.getId(), -checkinDaysPlus
        );
        // Если дата checkIn стала раньше
        var rrb = RefundRules.builder();
        Duration checkinPlus = Duration.ofDays(checkinDaysPlus);
        for (RefundRule rule: originalInfo.getRefundRules().getRules()) {
            rrb.rule(rule.shift(checkinPlus));
        }
        itinerary.setRefundRules(rrb.build());
        uiPayload = uiPayload.toBuilder()
                .refundRules(rrb.build())
                .build();
        // todo modify base_rate_breakdown ?

        // todo modify plus topup moment
    }

    public void changeGuests(TModifyHotelOrderGuestsParams guests) {
        log.info("Changing guests for order {} ({})", order.getPrettyId(), order.getId());
        itinerary.setGuests(guests.getGuestsList().stream().map(
                g -> {
                    var res = new Guest();
                    res.setFirstName(g.getFirstName());
                    res.setLastName(g.getLastName());
                    res.setChild(g.getChild());
                    if (g.hasAge()) {
                        res.setAge(g.getAge().getValue());
                    }
                    return res;
                }
        ).collect(Collectors.toList()));
        isModified = true;
    }
}
