package ru.yandex.travel.orders.services.admin;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundFlags;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundRsp;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TManualRefundMoneyOnlyRsp;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderParams;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractReq;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractRsp;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderRsp;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.repository.HotelOrderRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefresh;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TChangeAgreement;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TMoneyMarkup;
import ru.yandex.travel.orders.workflow.order.proto.TStartManualServiceRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartMoneyOnlyRefund;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.orders.workflows.order.hotel.MoneyRefundUtils;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@RequiredArgsConstructor
@Slf4j
public class OrderAdminOperationsService {
    private final WorkflowMessageSender workflowMessageSender;
    private final WorkflowMaintenanceService workflowMaintenanceService;
    private final RefundCalculationService refundCalculationService;

    private final HotelOrderRepository hotelOrderRepository;
    private final TicketRepository ticketRepository;
    private final VoucherRepository voucherRepository;
    private final BalanceContractDictionary balanceContractDictionary;
    private final OrderRepository orderRepository;
    private final Clock clock;

    public TRestoreDolphinOrderRsp restoreDolphinOrder(UUID orderId, boolean cancel) {
        HotelOrder order = hotelOrderRepository.getOne(orderId);
        try (var ignored = NestedMdc.forEntity(order)) {
            Error.checkState(order.getState() == EHotelOrderState.OS_MANUAL_PROCESSING,
                    "Order should be in manual processing state");
            Error.checkState(order.getOrderItems().size() == 1,
                    "Unexpected number of services");
            Error.checkState(order.getOrderItems().get(0) instanceof DolphinOrderItem,
                    "Unexpected service type");
            DolphinOrderItem item = (DolphinOrderItem) order.getOrderItems().get(0);
            Error.checkState(item.getState() == EDolphinItemState.IS_MANUAL_CONFIRMATION,
                    "Service should be in manual confirmation state");
            if (order.getCurrentInvoice() != null) {
                Error.checkState(order.getCurrentInvoice() instanceof TrustInvoice,
                        "Unexpected invoice type");
                TrustInvoice invoice = (TrustInvoice) order.getCurrentInvoice();
                Error.checkState(invoice.getState() == ETrustInvoiceState.IS_HOLD || invoice.getState() == ETrustInvoiceState.IS_CLEARED,
                        "Invoice should be in hold or cleared state");
                Error.checkState(order.getWorkflow().getState() == EWorkflowState.WS_RUNNING,
                        "Order workflow should be running");
                Error.checkState(invoice.getWorkflow().getState() == EWorkflowState.WS_PAUSED,
                        "Invoice workflow should be paused");
            } else {
                // 0 payment has been used
                Error.checkState(
                        order.getUseDeferredPayment(),
                        "Only in case of deferred usage current invoice is not set for an order."
                );
            }
            Error.checkState(item.getWorkflow().getState() == EWorkflowState.WS_PAUSED,
                    "Service workflow should be paused");

            order.setState(EHotelOrderState.OS_WAITING_CONFIRMATION);
            if (cancel) {
                log.info("Requested cancellation of order item in manual confirmation state");
                if (item.getDolphinOrderCode() != null) {
                    log.info("A booking exists on partner's side, will schedule its cancellation");
                    item.setState(EDolphinItemState.IS_CANCELLING);
                    workflowMessageSender.scheduleEvent(item.getWorkflow().getId(),
                            TCancellationCommit.newBuilder().build());
                } else {
                    log.info("There is no dolphin-order-code present, assuming no booking on partner's side, " +
                            "cancelling the service immediately");
                    // returning back to reserved and sending TCancellationStart to immediately cancel with
                    // comment to ticket
                    item.setState(EDolphinItemState.IS_RESERVED);
                    workflowMessageSender.scheduleEvent(item.getWorkflow().getId(),
                            TCancellationStart.newBuilder().setReason(CancellationDetails.Reason.ADMIN_INTENTION.toString()).build());
                }
            } else {
                log.info("Requested restoration of order item in manual confirmation state");
                if (item.getDolphinOrderCode() != null) {
                    log.info("A booking exists on partner's side, will move service back to polling to refetch " +
                            "its state");
                    item.setState(EDolphinItemState.IS_POLLING_FOR_STATUS);
                    workflowMessageSender.scheduleEvent(item.getWorkflow().getId(), TRefresh.newBuilder().build());
                } else {
                    log.info("There is no dolphin-order-code present, assuming no booking on partner's side, " +
                            "will attempt to recreate the booking");
                    item.setState(EDolphinItemState.IS_CONFIRMING);
                    workflowMessageSender.scheduleEvent(item.getWorkflow().getId(),
                            TConfirmationCommit.newBuilder().build());
                }
            }
            workflowMaintenanceService.resumeSupervisedPausedWorkflows(order.getWorkflow().getId());

            TRestoreDolphinOrderRsp.Builder builder = TRestoreDolphinOrderRsp.newBuilder()
                    .setOrderId(order.getId().toString())
                    .setOrderPrettyId(order.getPrettyId())
                    .setHadPartnerOrder(item.getDolphinOrderCode() != null);
            if (item.getHotelItinerary().getManualTicketId() != null) {
                builder.setTicketKey(ticketRepository.getOne(item.getHotelItinerary().getManualTicketId()).getIssueId());
            }
            return builder.build();
        }
    }

    public void regenerateOrderVouchers(UUID orderId) {
        for (var v : voucherRepository.findAllByOrderId(orderId)) {
            if (v.getState() == EVoucherState.VS_CREATED) {
                log.info("Will regenerate voucher {} of order {}", v.getId(), orderId);
                workflowMessageSender.scheduleEvent(v.getWorkflow().getId(),
                        TGenerateVoucher.newBuilder().build());
            } else if (v.getState() == EVoucherState.VS_CREATING && v.getWorkflow().getState() == EWorkflowState.WS_CRASHED) {
                v.getWorkflow().setState(EWorkflowState.WS_RUNNING);
                log.info("Will regenerate voucher {} of order {}", v.getId(), orderId);
                workflowMessageSender.scheduleEvent(v.getWorkflow().getId(),
                        TGenerateVoucher.newBuilder().build());
            } else {
                log.warn("Voucher {} of order {} is not created yet", v.getId(), orderId);
            }
        }
    }

    public TCalculateHotelOrderRefundRsp calculateHotelOrderRefund(Order order) {
        try (var ignored = NestedMdc.forEntity(order)) {
            HotelOrderItem orderItem = checkAndGetHotelOrderItem(order);
            TPrice amount;

            try {
                var wrapper = refundCalculationService.calculateRefundForHotelItemFromRules(orderItem,
                        order.getCurrency());
                amount = ProtoUtils.toTPrice(wrapper.getRefundAmount());
            } catch (ErrorException e) {
                if (e.getError().getCode() == EErrorCode.EC_FAILED_PRECONDITION) {  // non-refundable order
                    amount = TPrice.newBuilder()
                            .setAmount(0L)
                            .setPrecision(2)
                            .setCurrency(order.getCurrency().getProtoCurrency())
                            .build();
                } else {
                    throw e;
                }
            }

            TCalculateHotelOrderRefundRsp.Builder builder = TCalculateHotelOrderRefundRsp.newBuilder()
                    .setOrderId(order.getId().toString())
                    .setOrderPrettyId(order.getPrettyId())
                    .setRefundAmountByRules(amount)
                    .setFlags(TCalculateHotelOrderRefundFlags.newBuilder()
                            .setCanAmountRefund(order.calculateDiscountAmount().isZero())
                            .build());

            if (order.canCalculateTotalCost()) {
                builder.setTotalAmount(ProtoUtils.toTPrice(order.calculateTotalCost()));
                builder.setPaidAmount(ProtoUtils.toTPrice(order.calculatePaidAmount()));
            }

            return builder.build();
        }
    }

    public void manualRefundHotelOrder(Order order, TPrice refundAmount, boolean generateFinEvents,
                                       EMoneyRefundMode moneyRefundMode, String reason) {
        try (var ignored = NestedMdc.forEntity(order)) {
            HotelOrderItem orderItem = checkAndGetHotelOrderItem(order);
            log.info("Scheduling refund for hotel order {}, amount: {}", order.getId(), refundAmount);

            order.setUserActionScheduled(true);

            var refundTokenWrapper = refundCalculationService.calculateRefundForHotelItemFromRefundAmount(
                    orderItem,
                    ProtoUtils.fromTPrice(refundAmount)
            );
            TStartManualServiceRefund serviceRefundMessage = TStartManualServiceRefund.newBuilder()
                    .setToken(refundTokenWrapper.getRefundToken())
                    .setSkipFinEvents(!generateFinEvents)
                    .setMoneyRefundMode(moneyRefundMode)
                    .setRefundDescription(reason)
                    .build();
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), serviceRefundMessage);
        }
    }

    public TCalculateMoneyOnlyRefundRsp calculateMoneyOnlyRefund(Order order) {
        try (var ignored = NestedMdc.forEntity(order)) {
            var hotelOrderItem = checkAndGetHotelOrderItemForMoneyOnlyRefund(order);
            Error.checkState(order.canCalculateTotalCost(), "Can't calculate total cost");
            return TCalculateMoneyOnlyRefundRsp.newBuilder()
                    .setOrderId(order.getId().toString())
                    .setOrderPrettyId(order.getPrettyId())
                    .setTotalAmount(ProtoUtils.toTPrice(order.calculateTotalCost()))
                    .setRemainingAmount(ProtoUtils.toTPrice(MoneyRefundUtils.getCurrentOrderMoney(hotelOrderItem)))
                    .build();
        }
    }

    public TManualRefundMoneyOnlyRsp manualRefundMoneyOnly(Order order, boolean isRefundUserMoney,
                                                           boolean isGenerateFinEvents,
                                                           TPrice requestedInvoiceAmount,
                                                           TMoneyMarkup requestedInvoiceAmountMarkup,
                                                           TPrice requestedRefundAmount,
                                                           EMoneyRefundMode moneyRefundMode,
                                                           String reason) {
        try (var ignored = NestedMdc.forEntity(order)) {
            var hotelOrderItem = checkAndGetHotelOrderItemForMoneyOnlyRefund(order);
            Error.checkArgument(isRefundUserMoney || isGenerateFinEvents,
                    "At least one of RefundUserMoney or GenerateFinEvents params should be set");

            Money newInvoiceAmount = ProtoUtils.fromTPrice(requestedInvoiceAmount);
            MoneyMarkup newInvoiceAmountMarkup = MoneyRefundUtils.fromTMoneyMarkup(requestedInvoiceAmountMarkup);
            Money refundAmount = ProtoUtils.fromTPrice(requestedRefundAmount);

            log.info("Scheduling money-only refund for hotel order {}, amount: {}, newInvoiceAmount: {}",
                    order.getId(), refundAmount, newInvoiceAmount);

            var currentMoney = MoneyRefundUtils.getCurrentOrderMoney(hotelOrderItem);
            Preconditions.checkState(newInvoiceAmount.isPositiveOrZero(),
                    "newInvoiceAmount (%s) should be positive or zero", newInvoiceAmount);
            Preconditions.checkState(refundAmount.isPositive(),
                    "refundAmount (%s) should be positive", refundAmount);
            Preconditions.checkState(newInvoiceAmount.add(refundAmount).isEqualTo(currentMoney),
                    "newInvoiceAmount (%s) + refundAmount (%s) should be equal to currentMoney (%s)",
                    newInvoiceAmount, refundAmount, currentMoney);
            Preconditions.checkState(newInvoiceAmountMarkup == null || newInvoiceAmount.equals(newInvoiceAmountMarkup.getTotal()),
                    "New invoice amount doesn't match the specified markup: new total %s, markup %s",
                    newInvoiceAmount, newInvoiceAmountMarkup);

            order.setUserActionScheduled(true);
            TStartMoneyOnlyRefund.Builder moneyRefundMessageBuilder = TStartMoneyOnlyRefund.newBuilder()
                    .setNewInvoiceAmount(requestedInvoiceAmount)
                    .setNewInvoiceAmountMarkup(requestedInvoiceAmountMarkup)
                    .setRefundAmount(requestedRefundAmount)
                    .setRefundUserMoney(isRefundUserMoney)
                    .setGenerateFinEvents(isGenerateFinEvents)
                    .setMoneyRefundMode(moneyRefundMode);

            if (reason != null) {
                moneyRefundMessageBuilder.setReason(reason);
            }

            TStartMoneyOnlyRefund moneyRefundMessage = moneyRefundMessageBuilder.build();

            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), moneyRefundMessage);

            return TManualRefundMoneyOnlyRsp.newBuilder().build();
        }
    }

    public void modifyHotelOrderDetails(HotelOrder order, TModifyHotelOrderParams params) {
        try (var ignored = NestedMdc.forEntity(order)) {
            log.info("Modifying order {} ({}) details, reason {}", order.getPrettyId(), order.getId(),
                    params.getReason());
            HotelOrderItem orderItem = checkHotelOrderItem(order);
            HotelOrderDetailsModifier modifier = new HotelOrderDetailsModifier(order, orderItem);
            if (params.hasDates()) {
                LocalDate checkInDate = LocalDate.parse(params.getDates().getCheckInDate());
                LocalDate checkOutDate = LocalDate.parse(params.getDates().getCheckOutDate());
                modifier.changeDates(checkInDate, checkOutDate);
            }
            if (params.hasGuests()) {
                modifier.changeGuests(params.getGuests());
            }
            modifier.finish();
        }
    }

    private HotelOrderItem checkAndGetHotelOrderItem(Order order) {
        checkHotelOrderItem(order);
        Error.checkArgument(order.getEntityState() == EHotelOrderState.OS_CONFIRMED,
                "Hotel order should be in CONFIRMED state, got " + order.getEntityState());

        return (HotelOrderItem) order.getOrderItems().get(0);
    }

    private HotelOrderItem checkAndGetHotelOrderItemForMoneyOnlyRefund(Order order) {
        checkHotelOrderItem(order);

        HotelOrder hotelOrder = (HotelOrder) order;
        Error.checkState(hotelOrder.getState() == EHotelOrderState.OS_CONFIRMED || hotelOrder.getState() == EHotelOrderState.OS_REFUNDED,
                "Order must be confirmed or refunded for money refund");

        return (HotelOrderItem) order.getOrderItems().get(0);
    }

    private HotelOrderItem checkHotelOrderItem(Order order) {
        Error.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                "Only hotel order can refund money with this method, got " + order.getDisplayType().toString());
        Error.checkState(order.getOrderItems().size() == 1, "Only 1 order item expected");

        return (HotelOrderItem) order.getOrderItems().get(0);
    }

    public TMoveHotelOrderToNewContractRsp moveHotelOrderToNewContract(TMoveHotelOrderToNewContractReq request) {
        HotelOrder order = getHotelOrder(request.getOrderId());
        Error.checkArgument(order.getState().equals(EHotelOrderState.OS_CONFIRMED),
                "Only Confirmed orders are supported, but found: " + order.getState());
        HotelOrderItem hotelOrderItem = checkAndGetItem(order);
        Error.checkArgument(hotelOrderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                "Only direct partner HotelOrders are supported");
        DirectHotelBillingPartnerAgreementProvider agreementProvider =
                (DirectHotelBillingPartnerAgreementProvider) hotelOrderItem;
        //TODO check payoutDate

        ContractInfo contractInfo = balanceContractDictionary.findContractInfoByClientId(request.getClientId());
        Preconditions.checkNotNull(contractInfo, "Contract was not found for clientId: %s", request.getClientId());
        Preconditions.checkArgument(request.getClientId() != agreementProvider.getAgreement().getFinancialClientId(),
                "The order is already on clientId: %s", request.getClientId());
        Preconditions.checkArgument(request.getContractId() == contractInfo.getContractId(),
                "Expected contract %s; got: %s", contractInfo.getContractId(), request.getContractId());
        if (request.hasPayoutDate()) {
            Instant payoutAt = ProtoUtils.toInstant(request.getPayoutDate());
            Preconditions.checkArgument(Instant.now(clock).isBefore(payoutAt), "PayoutAt: " + payoutAt +
                    " should not be in the past.");
        }
        if (hotelOrderItem instanceof TravellineOrderItem) {
            Preconditions.checkState(((TravellineOrderItem) hotelOrderItem).getState().equals(ETravellineItemState.IS_CONFIRMED));
            TChangeAgreement.Builder changeAgreementBuilder = TChangeAgreement.newBuilder()
                    .setClientId(request.getClientId())
                    .setContractId(request.getContractId());
            if (request.hasPayoutDate()) {
                changeAgreementBuilder.setPayoutDate(request.getPayoutDate());
            }
            workflowMessageSender.scheduleEvent(hotelOrderItem.getWorkflow().getId(),
                    changeAgreementBuilder.build());
        } else if (hotelOrderItem instanceof BNovoOrderItem) {
            Preconditions.checkState(((BNovoOrderItem) hotelOrderItem).getState().equals(EBNovoItemState.IS_CONFIRMED));
            ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TChangeAgreement.Builder changeAgreementBuilder =
                    ru.yandex.travel.orders.workflow.hotels.bnovo.proto.TChangeAgreement.newBuilder()
                            .setClientId(request.getClientId())
                            .setContractId(request.getContractId());
            if (request.hasPayoutDate()) {
                changeAgreementBuilder.setPayoutDate(request.getPayoutDate());
            }
            workflowMessageSender.scheduleEvent(hotelOrderItem.getWorkflow().getId(),
                    changeAgreementBuilder.build());
        } else {
            throw new RuntimeException("Unexpected OrderItem type");
        }
        return TMoveHotelOrderToNewContractRsp.newBuilder().build();
    }

    public HotelOrder getHotelOrder(TOrderId orderId) {
        HotelOrder order = null;
        switch (orderId.getOneOfOrderIdsCase()) {
            case ORDERID:
                order = hotelOrderRepository.getOne(UUID.fromString(orderId.getOrderId()));
                break;
            case PRETTYID:
                Order o = orderRepository.getOrderByPrettyId(orderId.getPrettyId());
                Error.checkArgument(o != null, "Hotel order not found");
                order = hotelOrderRepository.getOne(o.getId());
                break;
        }
        Error.checkArgument(order != null, "Hotel order not found");
        return order;
    }

    public HotelOrderItem checkAndGetItem(Order order) {
        Error.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                "Only hotel order can refund money with this method, got " + order.getDisplayType().toString());
        Error.checkArgument(order.getEntityState() == EHotelOrderState.OS_CONFIRMED,
                "Hotel order should be in CONFIRMED state, got " + order.getEntityState());
        Error.checkState(order.getOrderItems().size() == 1, "Only 1 order item expected");
        return (HotelOrderItem) order.getOrderItems().get(0);
    }
}
