package ru.yandex.travel.orders.services.attachments;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.FiscalReceiptState;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.AttachmentProviderType;
import ru.yandex.travel.orders.entities.notifications.FiscalReceiptAttachmentProviderData;
import ru.yandex.travel.orders.entities.notifications.ImAttachmentProviderData;
import ru.yandex.travel.orders.entities.notifications.UrlAttachmentProviderData;
import ru.yandex.travel.orders.entities.notifications.suburban.SuburbanCouponAttachmentProviderData;
import ru.yandex.travel.orders.repository.FiscalReceiptRepository;
import ru.yandex.travel.orders.repository.SuburbanOrderItemRepository;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@Service
@RequiredArgsConstructor
public class AttachmentProviderService {
    private final PdfGeneratorService pdfGeneratorService;
    private final FiscalReceiptRepository fiscalReceiptRepository;
    private final ImClient imClient;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final SuburbanOrderItemRepository suburbanOrderItemRepository;
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;

    public byte[] getAttachmentData(Attachment attachment) {
        var data = attachment.getProviderData();
        if (attachment.getProvider() == AttachmentProviderType.URL) {
            return getUrlEmailAttachmentData((UrlAttachmentProviderData) data);
        } else if (attachment.getProvider() == AttachmentProviderType.IM) {
            return getImEmailAttachmentData((ImAttachmentProviderData) data);
        } else if (attachment.getProvider() == AttachmentProviderType.FISCAL_RECEIPT) {
            return getFiscalReceiptEmailAttachmentData((FiscalReceiptAttachmentProviderData) data);
        } else if (attachment.getProvider() == AttachmentProviderType.SUBURBAN_COUPON ||
                attachment.getProvider() == AttachmentProviderType.MOVISTA_COUPON
        ) {
            return getSuburbanCouponAttachmentData((SuburbanCouponAttachmentProviderData) data);
        }
        throw new RuntimeException(String.format("Unknown Attachment provider '%s'", attachment.getProvider()));
    }

    private byte[] getImEmailAttachmentData(ImAttachmentProviderData data) {
        try {
            // Very nasty hack
            //TODO(mbobrov) think of a better way to handle this
            if (data.getFakeAttachment() != null && data.getFakeAttachment()) {
                return MockImClient.getStaticBlankPdf();
            } else {
                return imClient.orderReservationBlank(data.getBlankRequest(),
                        trainWorkflowProperties.getDownloadBlankTimeout());
            }
        } catch (ImClientRetryableException ex) {
            throw new RetryableException(ex);
        }
    }

    private byte[] getUrlEmailAttachmentData(UrlAttachmentProviderData data) {
        return pdfGeneratorService.downloadDocumentAsBytesSync(data.getUrl());
    }

    private byte[] getFiscalReceiptEmailAttachmentData(FiscalReceiptAttachmentProviderData data) {
        FiscalReceipt receipt = fiscalReceiptRepository.getOne(data.getFiscalReceiptId());
        if (receipt.getState() == FiscalReceiptState.CREATED) {
            throw new RetryableException("Assigned fiscal receipt is not fetched");
        } else if (receipt.getState() == FiscalReceiptState.ERROR) {
            throw new RuntimeException("Assigned fiscal receipt is in error state");
        }
        var url = receipt.getReceiptUrl().replace("mode=mobile", "mode=pdf");
        return pdfGeneratorService.downloadDocumentAsBytesSync(url);
    }

    private byte[] getSuburbanCouponAttachmentData(SuburbanCouponAttachmentProviderData data) {
        SuburbanOrderItem orderItem = suburbanOrderItemRepository.getOne(data.getOrderItemId());
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        return provider.getCouponAttachmentData();
    }
}
