package ru.yandex.travel.orders.services.avia;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.jackson.MoneySerializersModule;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.tvm.TvmWrapper;
import ru.yandex.travel.workflow.exceptions.RetryableException;

// inspired by DefaultTrustClient
@Service
@EnableConfigurationProperties(AviaApiProperties.class)
@Slf4j
public class AviaApiProxy extends HttpApiClientBase {
    private final TvmWrapper tvm;
    private final String dstTvmAlias;

    // in fact tvm should be enabled in all environments, made optional only for tests
    public AviaApiProxy(@Qualifier(value = "aviaApiAhcClientWrapper") AsyncHttpClientWrapper asyncHttpClient,
                        AviaApiProperties config, @Autowired(required = false) TvmWrapper tvm) {
        super(asyncHttpClient, config, new ObjectMapper()
                .registerModule(new MoneySerializersModule())
                .registerModule(new JavaTimeModule()));
        this.tvm = tvm;
        this.dstTvmAlias = config.getTvmAlias();
        if (tvm != null) {
            tvm.validateAlias(dstTvmAlias);
        }
    }

    public void refreshVariant(String variantId) {
        sync(sendRequest("GET","/v1/variants/refresh?id=" + variantId, null, JsonNode.class, "refreshVariant"));
    }

    @Override
    protected Throwable translateRpcError(Throwable t) {
        Throwable tr = super.translateRpcError(t);
        if (tr instanceof HttpApiRetryableException) {
            return new RetryableException(tr);
        }
        return tr;
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        UserCredentials userCredentials = UserCredentials.get();
        RequestBuilder builder = super.createBaseRequestBuilder(method, path, body);
        if (tvm != null) {
            String serviceTicket = tvm.getServiceTicket(dstTvmAlias);
            builder.setHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), serviceTicket);
        }
        if (userCredentials != null) {
            if (!Strings.isNullOrEmpty(userCredentials.getUserTicket())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.USER_TICKET.getHeader(), userCredentials.getUserTicket());
            }
            if (!Strings.isNullOrEmpty(userCredentials.getYandexUid())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.YANDEX_UID.getHeader(), userCredentials.getYandexUid());
            }
            if (!Strings.isNullOrEmpty(userCredentials.getPassportId())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.PASSPORT_ID.getHeader(), userCredentials.getPassportId());
            }
            if (!Strings.isNullOrEmpty(userCredentials.getLogin())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.USER_LOGIN.getHeader(), userCredentials.getLogin());
            }
            if (!Strings.isNullOrEmpty(userCredentials.getSessionKey())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.SESSION_KEY.getHeader(), userCredentials.getSessionKey());
            }
        }
        return builder;
    }
}
