package ru.yandex.travel.orders.services.avia.aeroflot;

import java.time.Instant;

import lombok.extern.slf4j.Slf4j;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobExecutionContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.QuartzJobBean;

@DisallowConcurrentExecution
@Slf4j
public class AeroflotMqJob extends QuartzJobBean {
    @Autowired
    private AeroflotMqProperties properties;
    @Autowired
    private AeroflotMqReader reader;
    @Autowired
    private AeroflotMqService mqService;

    @Override
    protected void executeInternal(JobExecutionContext context) {
        log.debug("Synchronizing Aeroflot orders");
        boolean hasMessages;
        do {
            hasMessages = false;
            for (AeroflotMqRawData rawMessage : reader.readOrders()) {
                if (rawMessage.getSendTimestamp() != null &&
                        rawMessage.getSendTimestamp().plus(properties.getProcessingDelay()).isAfter(Instant.now())) {
                    // Some messages arrive too early, before we can match them by avia_pnr
                    // Let's delay them until the workflow handler has committed the received pnr
                    log.debug("Will delay the message processing: messageId={}, sentTimestamp={}, maxDelay={}",
                            rawMessage.getId(), rawMessage.getSendTimestamp(), properties.getProcessingDelay());
                    continue;
                }
                try {
                    hasMessages = true;
                    mqService.scheduleMessageProcessing(rawMessage);
                    reader.markProcessed(rawMessage.getHandle());
                } catch (Exception e) {
                    log.warn("Failed to process the order data: data={}", rawMessage, e);
                }
            }
        } while (hasMessages);
    }
}
