package ru.yandex.travel.orders.services.avia.aeroflot;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dom4j.Node;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.enums.PassengerCategory;
import ru.yandex.avia.booking.partners.gateways.aeroflot.parsing.XmlUtils;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqData.PassengerRef;

import static java.util.stream.Collectors.toList;
import static ru.yandex.avia.booking.partners.gateways.aeroflot.parsing.XmlUtils.extractDouble;
import static ru.yandex.avia.booking.partners.gateways.aeroflot.parsing.XmlUtils.extractText;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotMqParser {
    private static final DateTimeFormatter mqDateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private final AeroflotMqProperties properties;

    public AeroflotMqData parseMessage(String content) {
        try {
            Node mqMsgData = XmlUtils.parseXml(content);
            Node bookingData = XmlUtils.extractNode(mqMsgData, "/cepMessage/content/bookings/booking");

            AeroflotMqData data = new AeroflotMqData();
            data.setSentDate(parseDate(extractText(mqMsgData, "/cepMessage/header/dateSent"), properties.getMqTimeZoneId()));
            data.setBookingDate(parseDate(extractText(bookingData, "bookingDate"), properties.getApiTimeZoneId()));
            data.setPnr(extractText(bookingData, "rloc"));
            data.setEmail(extractText(bookingData, "IATA_Notification_Email"));
            data.setPhone(extractText(bookingData, "IATA_Notification_Phone"));
            data.setPassengers(((List<?>) bookingData.selectNodes("bookingNames/bookingName")).stream()
                    .map(passenger -> (Node) passenger)
                    .map(passenger -> PassengerRef.builder()
                            .firstName(extractText(passenger, "firstName"))
                            .lastName(extractText(passenger, "lastName"))
                            .category(parseCategory(extractText(passenger, "passengerType")))
                            .build())
                    .collect(toList()));

            data.setSourceMessage(content);

            Node price = bookingData.selectSingleNode("PGWOrders/PGWOrder");
            if (price != null) {
                data.setTotalPrice(Money.of(
                        extractDouble(price, "Amount"),
                        extractText(price, "Currency")
                ));
            }

            List<?> tickets = bookingData.selectNodes("TicketDocuments/TicketDocument");
            if (tickets != null && !tickets.isEmpty()) {
                data.setTickets(tickets.stream()
                        .map(ticket -> (Node) ticket)
                        .map(ticket -> extractText(ticket, "TicketNumber"))
                        .collect(toList())
                );
            }

            return data;
        } catch (Exception e) {
            log.warn("Parsing of the message has failed: msg={}, e.msg={}", content, e.getMessage());
            throw new RuntimeException(e);
        }
    }

    private ZonedDateTime parseDate(String value, ZoneId tzId) {
        return LocalDateTime.parse(value, mqDateTimeFormatter).atZone(tzId);
    }

    private PassengerCategory parseCategory(String category) {
        switch (category) {
            case "1":
                return PassengerCategory.ADULT;
            case "2":
                return PassengerCategory.CHILD;
            case "3":
                return PassengerCategory.INFANT;
            default: {
                log.warn("Unknown passenger category: {}", category);
                return null;
            }
        }
    }
}
