package ru.yandex.travel.orders.services.avia.aeroflot;

import java.time.Instant;
import java.util.List;

import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.model.Message;
import com.amazonaws.services.sqs.model.ReceiveMessageRequest;
import com.amazonaws.services.sqs.model.ReceiveMessageResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import static java.util.stream.Collectors.toList;

@Service
@Slf4j
public class AeroflotMqReader {
    private static final ObjectMapper mqJsonMapper = new ObjectMapper();

    private final AmazonSQS client;
    private final AeroflotMqProperties properties;
    private final String queueUrl;

    @Autowired
    public AeroflotMqReader(AmazonSQS client, AeroflotMqProperties properties) {
        this.client = client;
        this.properties = properties;
        this.queueUrl = constructQueueUrl();

        log.info("AeroflotMqReader initialised; queueUrl={}, apiTimeZone={}, mqTimeZone={}, enabled={}",
                queueUrl, properties.getApiTimeZoneId(), properties.getMqTimeZoneId(), properties.getEnabled());
    }

    private String constructQueueUrl() {
        return properties.getSqsEndpoint() + "/" + properties.getSqsUser() + "/" + properties.getSqsQueueName();
    }

    public List<AeroflotMqRawData> readOrders() {
        ReceiveMessageResult result = client.receiveMessage(new ReceiveMessageRequest()
                .withQueueUrl(queueUrl)
                .withVisibilityTimeout((int) properties.getReadVisibilityTimeout().toSeconds())
                .withMaxNumberOfMessages(properties.getReadMessagesLimit())
        );
        log.debug("{} sqs messages received", result.getMessages().size());
        return result.getMessages().stream()
                .map(this::parseMessage)
                .collect(toList());
    }

    private AeroflotMqRawData parseMessage(Message message) {
        try {
            JsonNode aviaTicketDaemonWrapper = mqJsonMapper.readTree(message.getBody());
            String aeroflotMqData = aviaTicketDaemonWrapper.get("message").textValue();
            return AeroflotMqRawData.builder()
                    .id(message.getMessageId())
                    .data(aeroflotMqData)
                    .handle(message.getReceiptHandle())
                    .sendTimestamp(extractSentTimestamp(message))
                    .build();
        } catch (Exception e) {
            throw new RuntimeException("Message wrapper parsing has failed", e);
        }
    }

    private Instant extractSentTimestamp(Message message) {
        try {
            String value = message.getAttributes().get("SentTimestamp");
            return Instant.ofEpochMilli(Long.parseLong(value));
        } catch (Exception e) {
            return null;
        }
    }

    public void markProcessed(String mqMessageHandle) {
        client.deleteMessage(queueUrl, mqMessageHandle);
    }
}
