package ru.yandex.travel.orders.services.avia.aeroflot;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTicketCoupon;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTicketCouponStatusCode;
import ru.yandex.travel.orders.entities.AeroflotLastRequest;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.repository.AeroflotLastRequestRepository;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemCouponRefresh;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTicketCoupons;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties(AeroflotOrderStateSyncProperties.class)
public class AeroflotOrderStateSync {
    private final AeroflotOrderStateSyncLimit aeroflotOrderStateSyncLimit;
    private final AeroflotServiceProvider provider;
    private final WorkflowMessageSender workflowMessageSender;
    private final AeroflotOrderStateSyncProperties config;
    private final Clock clock;
    private final AeroflotLastRequestRepository aeroflotLastRequestRepository;

    @VisibleForTesting
    ConcurrentMap<UUID, Pair<Instant, AeroflotOrderCreateResult>> lastUpdate = new ConcurrentHashMap<>();
    ConcurrentMap<UUID, Object> locks = new ConcurrentHashMap<>();

    public boolean sync(AeroflotOrder order) {
        Preconditions.checkNotNull(order, "Order is null");
        log.debug("Sync aeroflot order {}", order.getId());
        if (order.getState() == EAeroflotOrderState.OS_CANCELLED || order.getState() == EAeroflotOrderState.OS_EXTERNALLY_CANCELLED) {
            log.debug("The order {} is already cancelled", order.getId());
            return false;
        }
        var couponStatusCodes = getCouponStatusCodes(order);
        var canCancel = canCancel(couponStatusCodes);
        var orderItem = order.getAeroflotOrderItem();
        if (canCancel || (couponStatusCodes != null && !couponEquals(couponStatusCodes,
                orderItem.getPayload().getTicketCoupons()))) {
            log.debug("Cancelling aeroflot order {}", order.getId());

            var couponRefresh = TAeroflotOrderItemCouponRefresh.newBuilder();
            for (var coupon : couponStatusCodes.entrySet()) {
                var build = TAeroflotOrderItemTicketCoupons.newBuilder();
                for (var c : coupon.getValue()) {
                    build.putCoupons(c.getCouponId(),
                            AeroflotOrderCouponStateMapper.getCouponStatusCode(c.getStatusCode()));
                }
                couponRefresh.putTicketCoupons(coupon.getKey(), build.build());
            }
            couponRefresh.setCancel(canCancel);
            workflowMessageSender.scheduleEvent(orderItem.getWorkflow().getId(), couponRefresh.build());

            log.debug("Cancelled aeroflot order {}", order.getId());
            return true;
        }
        log.debug("Skip aeroflot order {} state {} couponStatusCodes {}", order.getId(), order.getState(),
                couponStatusCodes);
        return false;
    }

    public AeroflotOrderCreateResult getAeroflotOrderStateUseCache(AeroflotOrder order) {
        locks.putIfAbsent(order.getId(), new Object());
        synchronized (locks.get(order.getId())) {
            var res = getAeroflotOrderStateFromCache(order);
            if (res != null) {
                return res;
            }

            var orderId = order.getId();
            log.debug("Get aeroflot state for order {}", orderId);

            if (aeroflotOrderStateSyncLimit.need(1)) {
                log.debug("Request limit is available for order {}", orderId);
                var lastRequest = Instant.now();

                AeroflotOrderCreateResult orderStatus = null;
                try {
                    var service = AeroflotWorkflowUtils.getOnlyOrderItem(order);
                    log.debug("Preparing request to aeroflot for order {}", orderId);
                    orderStatus = provider.getAeroflotServiceForProfile(service).getOrderStatus(orderId,
                            service.getPayload());
                    log.debug("Requested from aeroflot for order {}, return new ", orderId);

                    return orderStatus;
                } catch (RuntimeException e) {
                    log.error("Can not get aeroflot order status from aeroflot service", e);
                    return null;
                } finally {
                    log.debug("Save AeroflotOrderCreateResult {}, time request {} for order {}", orderStatus,
                            lastRequest, orderId);

                    createOrUpdateEntityAndUpdateLocalCache(orderId, lastRequest, orderStatus);
                }
            }
            log.debug("Rate limit exceeded");
            return null;
        }
    }

    @VisibleForTesting
    AeroflotOrderCreateResult getAeroflotOrderStateFromCache(AeroflotOrder order) {
        var orderId = order.getId();
        log.debug("Get aeroflot state from cache for order {}", orderId);
        if (!lastUpdate.containsKey(orderId)) {
            // no data in local cache, download from DB
            getOrCreateEntityAndUpdateLocalCache(orderId);
        }
        var cacheItem = lastUpdate.get(orderId);
        if (!isLastRequestExpired(cacheItem.getLeft())) {
            // local cache is up-to-date
            log.debug("Skip request to aeroflot for order {}, return from cache", orderId);
            return cacheItem.getRight();
        }
        log.debug("Search last request to aeroflot in db, order {}", orderId);
        var optEntity = aeroflotLastRequestRepository.findById(orderId);
        if (optEntity.isEmpty()) {
            return null;
        }
        var entity = optEntity.get();
        if (isLastRequestExpired(entity.getLastRequestAt())) {
            log.debug("Local and DB caches are expired {}", orderId);
            return null;
        }
        log.debug("Update cache from db and return from cache, order {}", orderId);
        updateLocalCache(entity);
        return entity.getResult();
    }

    @VisibleForTesting
    boolean isLastRequestExpired(Instant lastRequest) {
        return config.getResponseExpiration().toSeconds() < Duration.between(lastRequest,
                Instant.now(clock)).toSeconds();
    }

    private AeroflotLastRequest getOrDefaultEntity(UUID orderId, Consumer<AeroflotLastRequest> onDefault) {
        var optEntity = aeroflotLastRequestRepository.findById(orderId);
        return optEntity.orElseGet(() -> {
            var e = new AeroflotLastRequest();
            e.setOrderId(orderId);
            e.setLastRequestAt(Instant.ofEpochSecond(0));
            e.setResult(null);
            if (onDefault != null) {
                onDefault.accept(e);
            }
            return e;
        });
    }

    void updateLocalCache(AeroflotLastRequest entity) {
        lastUpdate.put(entity.getOrderId(), Pair.of(entity.getLastRequestAt(), entity.getResult()));
    }

    @VisibleForTesting
    void getOrCreateEntityAndUpdateLocalCache(UUID orderId) {
        updateLocalCache(getOrDefaultEntity(orderId, aeroflotLastRequestRepository::saveAndFlush));
    }

    @VisibleForTesting
    synchronized void createOrUpdateEntityAndUpdateLocalCache(
            UUID orderId, Instant lastRequest, AeroflotOrderCreateResult result) {
        var entity = getOrDefaultEntity(orderId, null);
        entity.setLastRequestAt(lastRequest);
        entity.setResult(result);
        aeroflotLastRequestRepository.saveAndFlush(entity);
        updateLocalCache(entity);
    }

    public Map<String, List<AeroflotTicketCoupon>> getCouponStatusCodes(AeroflotOrder order) {
        var res = getAeroflotOrderStateUseCache(order);
        if (res != null) {
            return res.getCouponStatusCodes();
        }
        return null;
    }

    public static boolean canCancel(Map<String, List<AeroflotTicketCoupon>> ticketCouponStatus) {
        if (ticketCouponStatus == null || ticketCouponStatus.isEmpty()) {
            return false;
        }

        for (var coupons : ticketCouponStatus.entrySet()) {
            for (var coupon : coupons.getValue()) {
                if (coupon.getStatusCode() != AeroflotTicketCouponStatusCode.VOIDED && coupon.getStatusCode() != AeroflotTicketCouponStatusCode.REFUNDED) {
                    return false;
                }
            }
        }
        return true;
    }

    public static boolean couponEquals(Map<String, List<AeroflotTicketCoupon>> couponStatusCodes,
                                       Map<String, List<AeroflotTicketCoupon>> couponStatusCodes2) {
        if (couponStatusCodes == couponStatusCodes2) {
            return true;
        }
        if (couponStatusCodes == null || couponStatusCodes2 == null) {
            return false;
        }
        if (couponStatusCodes.size() != couponStatusCodes2.size()) {
            return false;
        }
        for (var couponStatus : couponStatusCodes.entrySet()) {
            if (!couponStatusCodes2.containsKey(couponStatus.getKey())) {
                return false;
            }
            var couponStatus2 = couponStatusCodes2.get(couponStatus.getKey());
            if (couponStatus.getValue().size() != couponStatus2.size()) {
                return false;
            }
            for (var coupon : couponStatus.getValue()) {
                var first =
                        couponStatus2.stream()
                                .filter(x -> x.getCouponId().equals(coupon.getCouponId()) && x.getStatusCode() == coupon.getStatusCode())
                                .findFirst()
                                .orElse(null);
                if (first == null) {
                    return false;
                }
            }
        }
        return true;
    }
}
