package ru.yandex.travel.orders.services.avia.aeroflot;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;

import com.google.common.annotations.VisibleForTesting;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

/**
 * Rate limiter с окном
 * Характеризуется определённым лимитом запросов за определённый период времени.
 * Лимит запрашивается методом need(int).
 * Если лимит превышен, то он будет сброшен по окончании временного окна.
 */
@Slf4j
@Component
@RequiredArgsConstructor
@EnableConfigurationProperties(AeroflotOrderStateSyncProperties.class)
public class AeroflotOrderStateSyncLimit {
    private final AeroflotOrderStateSyncProperties config;
    private final Clock clock;
    private Instant rateLimiterRestartedAt = null;
    private long limit = 0;

    public synchronized boolean need(int n) {
        resetIfNeeded();
        log.info("date update {}, current limit {}, need {}", rateLimiterRestartedAt, limit, n);

        if (limit >= n) {
            limit -= n;
            return true;
        }
        return false;
    }


    /**
     * Нам нужно постоянно обновлять состояния заказов. Заказы обновляются пачками по N штук (N ≅ 1000).
     * Мы знаем размер пачки и знаем доступное количество запросов до конца окна.
     * Нужно размазать процесс обновления по окну.
     * Необходимо так рассчитать интервал между заказами, чтобы в процессе выполнения он не превысил rate limit.
     * Если ожидаемое количество заказов больше, чем оставшийся лимит, то остаток нужно "перенести" в новое окно.
     *
     * @param numberOfOrders количество заказов, которые будут обработаны в рамках одной пачки
     * @return средний интервал между обновлениями каждого заказа в пачке, величина обратная rps
     */
    public Duration estimateIntervalBetweenOrderUpdates(long numberOfOrders) {
        if (numberOfOrders <= 0) {
            return config.getDefaultDelay();
        }
        // количество полных пачек заказов, которое должно влезает в оставшийся лимит
        long numBatches = this.getUnusedLimit() / numberOfOrders;
        var secondsUntilNextRateLimitPeriod = this.durationUntilNextRateLimitPeriod();

        if (numBatches > 0) {
            // Если влезает хотя бы одна пачка заказов, надо размазать их по оставшемуся окну
            Duration batchInterval = secondsUntilNextRateLimitPeriod.dividedBy(numBatches);
            return batchInterval.dividedBy(numberOfOrders);
        }
        // Если вся пачка не влезает, расчитываем средний интервал согласно конфигу
        return config.getRequestLimitPeriod().dividedBy(config.getRequestLimitCount());
    }

    @VisibleForTesting
    synchronized long getUnusedLimit() {
        resetIfNeeded();
        return limit;
    }

    @VisibleForTesting
    synchronized Duration durationUntilNextRateLimitPeriod() {
        if (rateLimiterRestartedAt == null) {
            return Duration.ZERO;
        }
        var nextRateLimiterPeriodStartsAt = rateLimiterRestartedAt.plus(config.getRequestLimitPeriod());
        return Duration.between(Instant.now(clock), nextRateLimiterPeriodStartsAt);
    }

    private synchronized void resetIfNeeded() {
        if (rateLimiterRestartedAt == null || durationUntilNextRateLimitPeriod().isNegative()) {
            limit = config.getRequestLimitCount();
            rateLimiterRestartedAt = Instant.now(clock);
        }
    }
}
