package ru.yandex.travel.orders.services.buses;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.FiscalReceiptType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.entities.notifications.SmsChannelInfo;
import ru.yandex.travel.orders.entities.notifications.bus.BusBaseMailArgs;
import ru.yandex.travel.orders.entities.notifications.bus.BusConfirmedMailArgs;
import ru.yandex.travel.orders.entities.notifications.bus.BusPassengerMailArgs;
import ru.yandex.travel.orders.entities.notifications.bus.BusRefundMailArgs;
import ru.yandex.travel.orders.entities.notifications.bus.BusRideMailArgs;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.services.NotificationFileTypes;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.UrlShortenerService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.orders.services.NotificationHelper.formatMoney;
import static ru.yandex.travel.orders.services.NotificationHelper.humanDate;
import static ru.yandex.travel.orders.services.NotificationHelper.humanTime;

@Service
@Slf4j
@RequiredArgsConstructor
public class BusNotificationHelper {
    private static final Integer MAIN_PASSENGER_MIN_AGE = 18;
    private static final String DASH = " \u2014 ";
    private static final String COMMA = ", ";
    private static final String LOCAL_TIME_TZ_TEXT = "по местному времени";

    private final BusProperties busProperties;
    private final WorkflowRepository workflowRepository;
    private final NotificationRepository notificationRepository;
    private final UrlShortenerService urlShortenerService;

    public UUID createWorkflowForConfirmedOrderEmail(GenericOrder order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isBusOrder(order),
                "Bus order is expected but got %s", order.getClass().getName());

        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(busProperties.getNotification().getOrderConfirmedCampaign());
        emailInfo.setArguments(new POJONode(prepareConfirmedMailSenderArgs(order)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(busProperties.getNotification().getPreparingAttachmentsTime()));

        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);

        List<BusOrderItem> orderItems = OrderCompatibilityUtils.getBusOrderItems(order);
        for (int i = 0, orderItemsSize = orderItems.size(); i < orderItemsSize; i++) {
            BusOrderItem orderItem = orderItems.get(i);
            var ticket = Attachment.createUrlAttachment(email,
                    "Ticket_" + email.getOrderPrettyId() + (orderItemsSize > 1 ? "_" + i : "") + ".pdf",
                    NotificationFileTypes.PDF, true, orderItem.getOrder().getDocumentUrl());
            workflowRepository.save(Workflow.createWorkflowForEntity(ticket, workflow.getId()));
        }
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        var receipt = invoice.getFiscalReceipts().stream()
                .filter(i -> i.getReceiptType() == FiscalReceiptType.ACQUIRE)
                .collect(Collectors.toList());
        Preconditions.checkState(receipt.size() == 1, "exactly 1 receipt is expected");
        var trustCheck = Attachment.createFiscalReceiptAttachment(email, "Check_" + email.getOrderPrettyId() + ".pdf",
                NotificationFileTypes.PDF, false, receipt.get(0).getId());
        workflowRepository.save(Workflow.createWorkflowForEntity(trustCheck, workflow.getId()));
        notificationRepository.save(email);
        return workflow.getId();
    }

    private BusConfirmedMailArgs prepareConfirmedMailSenderArgs(GenericOrder order) {
        List<BusOrderItem> orderItems = OrderCompatibilityUtils.getBusOrderItems(order);
        var args = new BusConfirmedMailArgs();
        fillBaseMailArgs(args, order, orderItems);
        args.setPassengers(new ArrayList<>());
        for (BusesPassenger p : orderItems.get(0).getPayload().getRequestPassengers()) {
            args.getPassengers().add(new BusPassengerMailArgs(getFullName(p)));
        }
        args.setManyPassengers(args.getPassengers().size() > 1);
        args.setManyTickets(orderItems.stream()
                .flatMap(x -> x.getPayload().getOrder().getTickets().stream())
                .count() > 1);
        args.setRides(new ArrayList<>());
        for (var orderItem : orderItems) {
            args.getRides().add(getRideArgs(orderItem));
        }
        return args;
    }

    public UUID createWorkflowForRefundEmail(GenericOrder order, GenericOrderUserRefund refund) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isBusOrder(order),
                "Bus order is expected but got %s", order.getClass().getName());
        List<BusTicketRefund> ticketRefunds = refund.getBusTicketRefunds();
        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(busProperties.getNotification().getOrderRefundCampaign());
        boolean isResize = NotificationHelper.isResize(order, refund);
        emailInfo.setArguments(new POJONode(prepareRefundMailArgs(order, ticketRefunds, isResize)));
        emailInfo.setTarget(order.getEmail());
        var email = Notification.createEmailNotification(order, emailInfo);
        email.setPreparingAttachmentsTill(Instant.now().plus(busProperties.getNotification().getPreparingAttachmentsTime()));
        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        List<FiscalReceipt> refundReceipts = invoice.getFiscalReceipts().stream()
                .filter(i -> refund.getId().equals(i.getOrderRefundId()))
                .collect(Collectors.toList());
        if (refundReceipts.size() > 0) {
            Preconditions.checkState(refundReceipts.size() == 1,
                    "exactly 1 refund receipt is expected but got %s", refundReceipts.size());
            var trustRefundCheck = Attachment.createFiscalReceiptAttachment(email,
                    "Refund_check_" + refund.getId() + ".pdf",
                    NotificationFileTypes.PDF, false, refundReceipts.get(0).getId());
            workflowRepository.save(Workflow.createWorkflowForEntity(trustRefundCheck, workflow.getId()));
        }
        notificationRepository.save(email);
        return workflow.getId();
    }

    private BusRefundMailArgs prepareRefundMailArgs(GenericOrder order, List<BusTicketRefund> refunds,
                                                    boolean isResize) {
        List<BusOrderItem> orderItems = OrderCompatibilityUtils.getBusOrderItems(order);
        var args = new BusRefundMailArgs();
        args.setIsResize(isResize);
        args.setRefundPartFailed(refunds.stream().anyMatch(x -> x.getState() == EBusTicketRefundState.RS_FAILED));
        fillBaseMailArgs(args, order, orderItems);
        Set<String> refundedTicketIds = refunds.stream().filter(x -> x.getState() == EBusTicketRefundState.RS_REFUNDED)
                .map(x -> x.getPayload().getTicketId()).collect(Collectors.toSet());
        List<BusesTicket> refundedTickets = orderItems.stream()
                .flatMap(x -> x.getPayload().getOrder().getTickets().stream())
                .filter(x -> refundedTicketIds.contains(x.getId())).collect(Collectors.toList());
        args.setPassengers(refundedTickets.stream()
                .map(x -> getFullName(x.getPassenger())).distinct()
                .map(BusPassengerMailArgs::new)
                .collect(Collectors.toList()));
        args.setManyPassengers(args.getPassengers().size() > 1);
        args.setManyTickets(refundedTicketIds.size() > 1);
        args.setRides(new ArrayList<>());
        for (var orderItem : orderItems.stream()
                .filter(i -> i.getPayload().getOrder().getTickets().stream()
                        .anyMatch(t -> refundedTicketIds.contains(t.getId())))
                .collect(Collectors.toList())) {
            args.getRides().add(getRideArgs(orderItem));
        }
        BigDecimal totalSum = refundedTickets.stream()
                .map(x -> x.getTotalPrice().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        BigDecimal refundSum = refunds.stream().filter(x -> x.getState() == EBusTicketRefundState.RS_REFUNDED)
                .map(x -> x.getPayload().getRefundAmount().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        args.setRefundSum(formatMoney(refundSum));
        args.setFeeSum(formatMoney(totalSum.subtract(refundSum)));
        return args;
    }

    private BusRideMailArgs getRideArgs(BusOrderItem orderItem) {
        var rideMailArgs = new BusRideMailArgs();
        BusRide ride = orderItem.getPayload().getRide();
        rideMailArgs.setRouteName(ride.getRouteName());
        rideMailArgs.setRouteNumber(ride.getRouteNumber());
        rideMailArgs.setFromTitle(ride.getPointFrom().getSupplierDescription());
        rideMailArgs.setToTitle(ride.getPointTo().getSupplierDescription());

        LocalDateTime departureLocalDt = ride.getLocalDepartureTime();
        rideMailArgs.setDepartureDate(humanDate(departureLocalDt));
        rideMailArgs.setDepartureTime(humanTime(departureLocalDt));
        rideMailArgs.setDepartureTzName(LOCAL_TIME_TZ_TEXT);

        LocalDateTime arrivalLocalDt = ride.getLocalArrivalTime();
        if (arrivalLocalDt != null) {
            rideMailArgs.setArrivalDate(humanDate(arrivalLocalDt));
            rideMailArgs.setArrivalTime(humanTime(arrivalLocalDt));
            rideMailArgs.setArrivalTzName(LOCAL_TIME_TZ_TEXT);
        }

        return rideMailArgs;
    }

    private void fillBaseMailArgs(BusBaseMailArgs args, GenericOrder order, List<BusOrderItem> orderItems) {
        args.setFrontUrl(busProperties.getNotification().getFrontUrl());
        args.setOrderUid(order.getId());
        args.setOrderPrettyId(order.getPrettyId());
        BigDecimal orderPrice = orderItems.stream()
                .map(x -> x.totalCostAfterReservation().getNumberStripped())
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
        args.setOrderPrice(formatMoney(orderPrice));

        args.setTripTitle(buildRouteTitle(orderItems));
        BusOrderItem firstOrderItem = orderItems.get(0);
        args.setIsMoscowRegion(firstOrderItem.getPayload().isMoscowRegion());

        BusesPassenger mainPassenger = firstOrderItem.getPayload().getRequestPassengers().stream()
                .filter(p -> p.getBirthday().plusYears(MAIN_PASSENGER_MIN_AGE).isBefore(LocalDate.now()))
                .findFirst().orElse(firstOrderItem.getPayload().getRequestPassengers().get(0));
        args.setMainFirstName(mainPassenger.getFirstName());
    }

    @VisibleForTesting
    public String buildRouteTitle(List<BusOrderItem> orderItems) {
        StringBuilder routeString = new StringBuilder();
        String fromTitle = null;
        String toTitle = null;
        for (var orderItem : orderItems) {
            BusRide ride = orderItem.getPayload().getRide();
            fromTitle = ride.getTitlePointFrom().getTitle();
            if (routeString.length() == 0) {
                routeString.append(fromTitle);
            } else if (toTitle != null && !toTitle.equals(fromTitle)) {
                routeString.append(COMMA);
                routeString.append(fromTitle);
            }
            toTitle = ride.getTitlePointTo().getTitle();
            routeString.append(DASH);
            routeString.append(toTitle);
        }
        return routeString.toString();
    }

    public UUID createWorkflowForConfirmedOrderSms(GenericOrder order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isBusOrder(order),
                "Bus order is expected but got %s", order.getClass().getName());
        Preconditions.checkState(OrderCompatibilityUtils.isConfirmed(order), "Order must be confirmed");
        var text = getConfirmationSmsText(order);
        var channelInfo = new SmsChannelInfo();
        channelInfo.setPhone(NotificationHelper.normalizeFreeFormatPhone(order.getPhone()));
        channelInfo.setText(text);
        var sms = Notification.createSmsNotification(order, channelInfo);
        Workflow workflow = Workflow.createWorkflowForEntity(sms, WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        notificationRepository.save(sms);
        return workflow.getId();
    }

    private String getConfirmationSmsText(GenericOrder order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isBusOrder(order),
                "Bus order is expected but got %s", order.getClass().getName());
        List<BusOrderItem> orderItems = OrderCompatibilityUtils.getBusOrderItems(order);
        var url = String.format("https://%s/my/order/%s", busProperties.getNotification().getFrontUrl(),
                order.getId().toString());
        url = urlShortenerService.shorten(url, true);
        String text = busProperties.getNotification().getOrderConfirmedSmsText();
        String rideText = "";
        if (orderItems.size() == 1) {
            BusRide ride = orderItems.get(0).getPayload().getRide();
            String departureDate = humanDate(ride.getLocalDepartureTime());
            rideText = " " + Stream.of(ride.getRouteNumber(), ride.getRouteName(), departureDate)
                    .filter(x -> !Strings.isNullOrEmpty(x)).collect(Collectors.joining(" "));
        }
        text = String.format(text, rideText, url);
        return text;
    }

    private static String getFullName(BusesPassenger p) {
        return String.format("%s %s", p.getFirstName(), p.getLastName());
    }
}
