package ru.yandex.travel.orders.services.buses;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.bus.model.BusPartner;
import ru.yandex.travel.commons.lang.MoneyUtils;
import ru.yandex.travel.orders.entities.FiscalItemType;

@Service
public class BusPartnersService {
    private final BusPartnersProperties busPartnersProperties;
    private final Map<Integer, BusPartner> partnersById;
    private final Map<BusPartner, BusPartnersProperties.PartnerProperties> partners;

    private static final BigDecimal PERCENT = BigDecimal.valueOf(1, 2);
    private static final Map<BusPartner, PartnerFiscalItemTypes> PARTNER_FISCAL_ITEMS_MAP = Map.ofEntries(
            Map.entry(BusPartner.ATLASBUS, new PartnerFiscalItemTypes(FiscalItemType.BUS_ATLASBUS_TICKET, FiscalItemType.BUS_ATLASBUS_PARTNER_FEE, FiscalItemType.BUS_ATLASBUS_YANDEX_FEE)),
            Map.entry(BusPartner.BUSFOR, new PartnerFiscalItemTypes(FiscalItemType.BUS_BUSFOR_TICKET, FiscalItemType.BUS_BUSFOR_PARTNER_FEE, FiscalItemType.BUS_BUSFOR_YANDEX_FEE)),
            Map.entry(BusPartner.ECOLINES, new PartnerFiscalItemTypes(FiscalItemType.BUS_ECOLINES_TICKET, FiscalItemType.BUS_ECOLINES_PARTNER_FEE, FiscalItemType.BUS_ECOLINES_YANDEX_FEE)),
            Map.entry(BusPartner.ETRAFFIC, new PartnerFiscalItemTypes(FiscalItemType.BUS_ETRAFFIC_TICKET, FiscalItemType.BUS_ETRAFFIC_PARTNER_FEE, FiscalItemType.BUS_ETRAFFIC_YANDEX_FEE)),
            Map.entry(BusPartner.NOY, new PartnerFiscalItemTypes(FiscalItemType.BUS_NOY_TICKET, FiscalItemType.BUS_NOY_PARTNER_FEE, FiscalItemType.BUS_NOY_YANDEX_FEE)),
            Map.entry(BusPartner.OK, new PartnerFiscalItemTypes(FiscalItemType.BUS_OK_TICKET, FiscalItemType.BUS_OK_PARTNER_FEE, FiscalItemType.BUS_OK_YANDEX_FEE)),
            Map.entry(BusPartner.RUSET, new PartnerFiscalItemTypes(FiscalItemType.BUS_RUSET_TICKET, FiscalItemType.BUS_RUSET_PARTNER_FEE, FiscalItemType.BUS_RUSET_YANDEX_FEE)),
            Map.entry(BusPartner.SKS, new PartnerFiscalItemTypes(FiscalItemType.BUS_SKS_TICKET, FiscalItemType.BUS_SKS_PARTNER_FEE, FiscalItemType.BUS_SKS_YANDEX_FEE)),
            Map.entry(BusPartner.UNITIKI_NEW, new PartnerFiscalItemTypes(FiscalItemType.BUS_UNITIKI_NEW_TICKET, FiscalItemType.BUS_UNITIKI_NEW_PARTNER_FEE, FiscalItemType.BUS_UNITIKI_NEW_YANDEX_FEE)),
            Map.entry(BusPartner.YUGAVTOTRANS, new PartnerFiscalItemTypes(FiscalItemType.BUS_YUGAVTOTRANS_TICKET, FiscalItemType.BUS_YUGAVTOTRANS_PARTNER_FEE, FiscalItemType.BUS_YUGAVTOTRANS_YANDEX_FEE))

    );

    public BusPartnersService(BusPartnersProperties busPartnersProperties) {
        this.busPartnersProperties = busPartnersProperties;
        partnersById = busPartnersProperties.getAllPartners().stream()
                .collect(Collectors.toMap(BusPartnersProperties.PartnerProperties::getId,
                        BusPartnersProperties.PartnerProperties::getPartner));
        partners = busPartnersProperties.getAllPartners().stream()
                .collect(Collectors.toMap(BusPartnersProperties.PartnerProperties::getPartner, x -> x));
    }

    public Money calculateYandexFee(BusPartner partner, Money price) {
        BusPartnersProperties.PartnerProperties partnerProperties = partners.get(partner);
        Preconditions.checkArgument(partnerProperties != null, "bus partner %s not found", partner);
        Money fee = price.multiply(partnerProperties.getYandexFeePercent().multiply(PERCENT));
        return MoneyUtils.roundToDecimal(fee, RoundingMode.HALF_UP);
    }

    public BusPartner getPartnerById(Integer supplierId) {
        BusPartner partner = partnersById.get(supplierId);
        if (partner == null) {
            throw new NoSuchElementException("bus partner not found by supplierId=" + supplierId);
        }
        return partner;
    }

    public FiscalItemType getTicketFiscalItemType(BusPartner partner) {
        return PARTNER_FISCAL_ITEMS_MAP.get(partner).ticket;
    }
    public FiscalItemType getPartnerFeeFiscalItemType(BusPartner partner) {
        return PARTNER_FISCAL_ITEMS_MAP.get(partner).partnerFee;
    }
    public FiscalItemType getYandexFeeFiscalItemType(BusPartner partner) {
        return PARTNER_FISCAL_ITEMS_MAP.get(partner).yandexFee;
    }

    @RequiredArgsConstructor
    private static class PartnerFiscalItemTypes {
        private final FiscalItemType ticket;
        private final FiscalItemType partnerFee;
        private final FiscalItemType yandexFee;
    }
}
