package ru.yandex.travel.orders.services.cloud.s3;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.function.Supplier;

import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.http.ContentDisposition;
import org.springframework.stereotype.Service;

import ru.yandex.travel.workflow.exceptions.RetryableException;

@Service
@RequiredArgsConstructor
public class S3ServiceImpl implements S3Service {
    private final S3ServiceProperties properties;
    private final AmazonS3 s3;

    @Override
    public boolean checkObjectExists(String id) {
        try {
            return s3.doesObjectExist(properties.getBucket(), id);
        } catch (SdkClientException e) {
            throw wrapRetryableExceptions(e, () -> "check object exists: " + id);
        }
    }

    @Override
    public void uploadObject(InMemoryS3Object object) {
        byte[] data = object.getData();
        InputStream dataStream = new ByteArrayInputStream(data);
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength(data.length);
        metadata.setContentType(object.getMimeType());
        if (!Strings.isNullOrEmpty(object.getFileName())) {
            ContentDisposition contentDisposition = ContentDisposition.builder("attachment")
                    .filename(object.getFileName(), StandardCharsets.UTF_8)
                    .build();
            metadata.setContentDisposition(contentDisposition.toString());
        }
        PutObjectRequest request = new PutObjectRequest(properties.getBucket(), object.getId(), dataStream, metadata);
        try {
            s3.putObject(request);
        } catch (SdkClientException e) {
            throw wrapRetryableExceptions(e, () -> "upload object: " + object.getId());
        }
    }

    @Override
    public InMemoryS3Object readObject(String id) {
        S3Object object;
        try {
            object = s3.getObject(properties.getBucket(), id);
        } catch (SdkClientException e) {
            throw wrapRetryableExceptions(e, () -> "read object: " + id);
        }
        ObjectMetadata metadata = object.getObjectMetadata();
        long contentLength = metadata.getContentLength();
        Preconditions.checkState(contentLength <= properties.getMaxInMemoryFileSize(),
                "The requested file's size of %s bytes exceeds the maximum allowed in-memory file size of %s bytes",
                contentLength, properties.getMaxInMemoryFileSize());
        byte[] data = new byte[(int) contentLength];
        try (InputStream dataStream = object.getObjectContent()) {
            int read = dataStream.readNBytes(data, 0, (int) contentLength);
            Preconditions.checkState(read == contentLength,
                    "Exactly %s bytes should have been read but got %s", contentLength, read);
        } catch (IOException e) {
            throw wrapRetryableNetworkExceptions(e, () -> "Failed to read the content of object " + id);
        }
        String fileName = null;
        if (!Strings.isNullOrEmpty(metadata.getContentDisposition())) {
            fileName = ContentDisposition.parse(metadata.getContentDisposition()).getFilename();
        }
        return InMemoryS3Object.builder()
                .id(id)
                .mimeType(metadata.getContentType())
                .fileName(fileName)
                .data(data)
                .build();
    }

    private RuntimeException wrapRetryableExceptions(SdkClientException e, Supplier<String> errorMessage) {
        return e.isRetryable() ? new RetryableException(errorMessage.get() + "; " + e.getMessage(), e) : e;
    }

    private RuntimeException wrapRetryableNetworkExceptions(IOException e, Supplier<String> errorMessage) {
        // expect most exceptions to be retryable, need to exclude non-retryable ones when they are found
        return new RetryableException(errorMessage.get() + "; " + e.getMessage(), e);
    }
}
