package ru.yandex.travel.orders.services.cpa;

import java.time.Instant;
import java.util.Comparator;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.protobuf.BoolValue;
import lombok.RequiredArgsConstructor;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.cpa.ECpaOrderStatus;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.cpa.TPersonalUserData;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.grpc.CpaProperties;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;

@RequiredArgsConstructor
abstract class AbstractCpaSnapshotProvider implements CpaSnapshotProvider {

    private final CpaProperties cpaProperties;
    private final AuthorizationService authService;

    protected abstract boolean supports(EOrderType orderType);

    protected boolean supports(EOrderType orderType, EServiceType serviceType) {
        return supports(orderType);
    }

    @Override
    public boolean supports(Order order) {
        return supports(order.getPublicType());
    }

    @Override
    public boolean supports(TListSnapshotsReqV2 request) {
        return supports(request.getOrderType(), request.getServiceType());
    }

    public static Instant getUpdatedAtToOrNow(TListSnapshotsReqV2 request) {
        Instant updatedAtTo;
        if (request.hasUpdatedAtTo()) {
            updatedAtTo = ProtoUtils.toInstant(request.getUpdatedAtTo());
        } else {
            updatedAtTo = Instant.now();
        }
        return updatedAtTo;
    }

    protected TOrderSnapshot.Builder mapBasicOrderInfo(Order order, int itemNumber, boolean addPersonalData) {
        TOrderSnapshot.Builder builder = TOrderSnapshot.newBuilder();
        builder.setAmount(ProtoUtils.toTPrice(order.calculateOriginalTotalCost()));
        builder.setTravelOrderId(order.getId().toString());
        // TODO: fix it for using OrdersService.addService
        if (order.getOrderItems().size() > 1 ||
                cpaProperties.getExtendedPartnerOrderIdFrom().isBefore(order.getCreatedAt())) {
            builder.setPartnerOrderId(order.getPrettyId() + ":" + itemNumber);
        } else {
            builder.setPartnerOrderId(order.getPrettyId());
        }
        builder.setBoyOrderId(order.getPrettyId());
        builder.setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()));
        builder.setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()));
        builder.setLabel(Strings.nullToEmpty(order.getLabel()));
        builder.setDiscountAmount(ProtoUtils.toTPrice(order.calculateDiscountAmount()));
        builder.setPromocodeDiscountAmount(ProtoUtils.toTPrice(order.calculatePromoDiscountAmount()));
        builder.setStrikeThroughDiscountAmount(ProtoUtils.toTPrice(order.calculateStrikeThroughDiscountAmount()));
        builder.addAllPromoCode(order.getPromoCodeApplications().stream()
                .map(p -> p.getPromoCodeActivation().getPromoCode().getCode()).collect(Collectors.toUnmodifiableList()));
        Set<String> actions = order.getPromoCodeApplications().stream()
                .map(p -> p.getPromoCodeActivation().getPromoCode().getPromoAction().getName())
                .collect(Collectors.toSet());
        // TODO(akormushkin): add actions from generated promo codes
        // order.getGeneratedPromoCodes().getPromoCodes().forEach(c -> actions.add(c.getPromoAction().getName()));
        builder.addAllPromoAction(actions);
        builder.setUsesDeferredPayment(order.getUseDeferredPayment());
        builder.setAmountReceiveFromUser(ProtoUtils.toTPrice(order.calculatePaidAmount()));
        if (order.getEligibleForDeferredPayment() != null) {
            builder.setEligibleForDeferredPayment(BoolValue.newBuilder().setValue(order.getEligibleForDeferredPayment()).build());
        }
        if (order.getPaymentSchedule() != null) {
            // NOTE (tivelkov): this items may be filled-in only for orders which were created with deferred payment.
            //  We may change this in future if the need arises.
            builder.setInitialPaymentAmount(ProtoUtils.toTPrice(order.getPaymentSchedule().getInitialPendingInvoice().getTotalAmount()));
            builder.setLastPaymentScheduledAt(
                    ProtoUtils.fromInstant(
                            order.getPaymentSchedule().getItems().stream()
                                    .max(Comparator.comparing(PaymentScheduleItem::getPaymentEndsAt))
                                    .map(PaymentScheduleItem::getPaymentEndsAt)
                                    .orElseThrow()));
            builder.setPenaltyAmountIfUnpaid(ProtoUtils.toTPrice(
                    order.getPaymentSchedule().getItems().stream()
                            .collect(CustomCollectors.summingMoney(PaymentScheduleItem::getPenaltyIfUnpaid,
                                    order.getCurrency()))));
            if (order.getPaymentSchedule().getClosedAt() != null && order.getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
                builder.setFullyPaidAt(ProtoUtils.fromInstant(order.getPaymentSchedule().getClosedAt()));
            }
        }
        if (addPersonalData) {
            var personalUserDataBuilder = TPersonalUserData.newBuilder();
            if (order.getEmail() != null) {
                personalUserDataBuilder.setEmail(order.getEmail());
            }
            if (order.getPhone() != null) {
                personalUserDataBuilder.setPhone(order.getPhone());
            }
            var owner = authService.getOrderOwner(order.getId());
            if (owner != null && owner.getLogin() != null) {
                personalUserDataBuilder.setOwnerLogin(owner.getLogin());
            }
            if (owner != null && owner.getPassportId() != null) {
                personalUserDataBuilder.setOwnerPassportId(owner.getPassportId());
            }
            builder.setPersonalUserData(personalUserDataBuilder.build());
        }
        enhanceBasicOrderInfo(builder, order, itemNumber);
        builder.setPostPayEligible(order.isPostPayEligible());
        builder.setPostPayUsed(order.isFullyPostPaid());
        return builder;
    }

    protected void enhanceBasicOrderInfo(TOrderSnapshot.Builder dto, Order order, int itemNumber) {
        // extend if necessary
    }

    @Override
    public ECpaOrderStatus getCpaOrderStatus(Order order) {
        throw new UnsupportedOperationException(String.format("Order %s public type %s is unsupported",
                order.getId(), order.getPublicType()));
    }

}
