package ru.yandex.travel.orders.services.cpa;

import java.time.Instant;
import java.util.List;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.cpa.ECpaOrderStatus;
import ru.yandex.travel.orders.cpa.TAviaExtraData;
import ru.yandex.travel.orders.cpa.TAviaFlight;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.grpc.CpaProperties;
import ru.yandex.travel.orders.repository.cpa.CpaAeroflotOrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;

import static java.util.stream.Collectors.toList;
import static ru.yandex.avia.booking.enums.PassengerCategory.ADULT;
import static ru.yandex.avia.booking.enums.PassengerCategory.CHILD;
import static ru.yandex.avia.booking.enums.PassengerCategory.INFANT;

@Component
public class AviaCpaSnapshotProvider extends AbstractCpaSnapshotProvider {
    private static final Money AEROFLOT_PROFIT = Money.of(215, "RUB"); // ~= 270 RUB - 20% НДС

    public static final List<EAeroflotOrderState> CPA_AEROFLOT_ORDER_STATES = List.of(EAeroflotOrderState.OS_CONFIRMED);

    private final CpaAeroflotOrderRepository cpaAeroflotOrderRepository;

    public AviaCpaSnapshotProvider(CpaProperties cpaProperties,
                                   CpaAeroflotOrderRepository cpaAeroflotOrderRepository,
                                   AuthorizationService authService) {
        super(cpaProperties, authService);
        this.cpaAeroflotOrderRepository = cpaAeroflotOrderRepository;
    }

    @Override
    public boolean supports(EOrderType orderType) {
        return orderType == EOrderType.OT_AVIA_AEROFLOT;
    }

    @Override
    public List<TOrderSnapshot> getSnapshotsV2(TListSnapshotsReqV2 request) {
        List<AeroflotOrder> orders;

        Instant updatedAtFrom = ProtoUtils.toInstant(request.getUpdatedAtFrom());
        Instant updatedAtTo = getUpdatedAtToOrNow(request);
        orders = cpaAeroflotOrderRepository.findByUpdatedAtAfterAndUpdatedAtBeforeAndStateInOrderByUpdatedAtAsc(
                updatedAtFrom,
                updatedAtTo,
                CPA_AEROFLOT_ORDER_STATES,
                PageRequest.of(0, request.getMaxSnapshots() + 1)
        );

        return orders.stream()
                .map(order -> toSnapshot(order, request.getAddPersonalData()))
                .collect(toList());
    }

    private TOrderSnapshot toSnapshot(AeroflotOrder order, boolean addPersonalData) {
        Preconditions.checkArgument(order.getState() == EAeroflotOrderState.OS_CONFIRMED,
                "Encountered unexpected order state %s", order.getState());
        TOrderSnapshot.Builder builder = mapBasicOrderInfo(order, order.getAeroflotOrderItem().getItemNumber(), addPersonalData);
        builder.setCpaOrderStatus(ECpaOrderStatus.OS_CONFIRMED);
        builder.setProfit(ProtoUtils.toTPrice(AEROFLOT_PROFIT));

        AeroflotServicePayload payload = order.getAeroflotOrderItem().getPayload();
        List<TAviaFlight> flights = payload.getVariant().getSegments().stream()
                .map(s -> TAviaFlight.newBuilder()
                        .setFrom(s.getDeparture().getAirportCode())
                        .setTo(s.getArrival().getAirportCode())
                        .setDepartureAt(ProtoUtils.fromLocalDateTime(s.getDeparture().getDateTime()))
                        .setArrivalAt(ProtoUtils.fromLocalDateTime(s.getArrival().getDateTime()))
                        .build())
                .collect(toList());

        String pnr = payload.getBookingRef() != null ? payload.getBookingRef().getPnr() : "";
        builder.setAviaExtraData(TAviaExtraData.newBuilder()
                .addAllFlights(flights)
                .setPnr(Strings.nullToEmpty(pnr))
                .setAdults(payload.getTravellersOfCategory(ADULT).size())
                .setChildren(payload.getTravellersOfCategory(CHILD).size())
                .setInfants(payload.getTravellersOfCategory(INFANT).size())
                .build());
        return builder.build();
    }
}
